/* The program converts a BASIC program in ASCII format to the Casio FX-700P
   tape image format (file transferred through the serial port).
   It's annotated for the "Splint" checking tool which is available for free
   at http://www.splint.org/	*/

#include <stdio.h>
#include <ctype.h>	/* islower, toupper, isdigit, iscntrl, isspace */
#include "bool.h"

typedef unsigned int uint;

/* structure for the keyword table tree */
typedef struct treestr
{
  char *tree_str;
  uint tree_val;
  /*@null@*/ struct treestr *left;
  /*@null@*/ struct treestr *right;
} TREE;


TREE keywords[] = {
  { ":",	0xFE,	NULL,		NULL		},	/* index 0 */
  { "<=",	0x0D,	keywords+0,	keywords+2	},	/* index 1 */
  { "<>",	0x0F,	NULL,		NULL		},	/* index 2 */
  { "=<",	0x0D,	keywords+1,	keywords+5	},	/* index 3 */
  { "=>",	0x0B,	NULL,		NULL		},	/* index 4 */
  { ">=",	0x0B,	keywords+4,	keywords+6	},	/* index 5 */
  { "ABS",	0x8C,	NULL,		NULL		},	/* index 6 */

  { "ACS",	0x84,	keywords+3,	keywords+11	},	/* index 7 */

  { "ASN",	0x83,	NULL,		NULL		},	/* index 8 */
  { "ATN",	0x85,	keywords+8,	keywords+10	},	/* index 9 */
  { "CLEAR",	0xC6,	NULL,		NULL		},	/* index 10 */
  { "COS",	0x81,	keywords+9,	keywords+13	},	/* index 11 */
  { "CSR",	0x94,	NULL,		NULL		},	/* index 12 */
  { "DEFM",	0xC0,	keywords+12,	keywords+14	},	/* index 13 */
  { "END",	0xAA,	NULL,		NULL		},	/* index 14 */


  { "EXP",	0x88,	keywords+7,	keywords+23	},	/* index 15 */


  { "FOR",	0xA0,	NULL,		NULL		},	/* index 16 */
  { "FRAC",	0x8B,	keywords+16,	keywords+18	},	/* index 17 */
  { "GET",	0xB3,	NULL,		NULL		},	/* index 18 */
  { "GOSUB",	0xA3,	keywords+17,	keywords+21	},	/* index 19 */
  { "GOTO",	0xA2,	NULL,		NULL		},	/* index 20 */
  { "IF",	0xA5,	keywords+20,	keywords+22	},	/* index 21 */
  { "INPUT",	0xA7,	NULL,		NULL		},	/* index 22 */

  { "INT",	0x8A,	keywords+19,	keywords+27	},	/* index 23 */

  { "KEY",	0x93,	NULL,		NULL		},	/* index 24 */
  { "LEN(",	0x90,	keywords+24,	keywords+26	},	/* index 25 */
  { "LIST",	0xC4,	NULL,		NULL		},	/* index 26 */
  { "LN",	0x87,	keywords+25,	keywords+29	},	/* index 27 */
  { "LOAD",	0xC2,	NULL,		NULL		},	/* index 28 */
  { "LOG",	0x86,	keywords+28,	keywords+30	},	/* index 29 */
  { "MID(",	0x92,	NULL,		NULL		},	/* index 30 */


  { "MODE",	0xA8,	keywords+15,	keywords+47	},	/* index 31 - ROOT */


  { "NEXT",	0xA1,	NULL,		NULL		},	/* index 32 */
  { "PI",	0x1B,	keywords+32,	keywords+34	},	/* index 33 */
  { "PRINT",	0xA6,	NULL,		NULL		},	/* index 34 */
  { "PUT",	0xB2,	keywords+33,	keywords+37	},	/* index 35 */
  { "RAN#",	0x8F,	NULL,		NULL		},	/* index 36 */
  { "RETURN",	0xA4,	keywords+36,	keywords+38	},	/* index 37 */
  { "RND(",	0x8E,	NULL,		NULL		},	/* index 38 */

  { "RUN",	0xC5,	keywords+35,	keywords+43	},	/* index 39 */

  { "SAVE",	0xC1,	NULL,		NULL		},	/* index 40 */
  { "SET",	0xB1,	keywords+40,	keywords+42	},	/* index 41 */
  { "SGN",	0x8D,	NULL,		NULL		},	/* index 42 */
  { "SIN",	0x80,	keywords+41,	keywords+45	},	/* index 43 */
  { "SQR",	0x89,	NULL,		NULL		},	/* index 44 */
  { "STEP",	0x96,	keywords+44,	keywords+46	},	/* index 45 */
  { "STOP",	0xA9,	NULL,		NULL		},	/* index 46 */


  { "TAN",	0x82,	keywords+39,	keywords+51	},	/* index 47 */


  { "THEN",	0x97,	NULL,		NULL		},	/* index 48 */
  { "TO",	0x95,	keywords+48,	keywords+50	},	/* index 49 */
  { "VAC",	0xB0,	NULL,		NULL		},	/* index 50 */
  { "VAL(",	0x91,	keywords+49,	keywords+52	},	/* index 51 */
  { "VER",	0xC3,	NULL,		NULL		}	/* index 52 */
};

#define KEYWORDS_ROOT (keywords+31)


/* character conversion table ASCII->Casio, ASCII codes 0x20-0x7A */
uint chartab[] = {
  0x00, 0x06, 0x07, 0x08, 0x09, 0x70, 0x74, 0x76,	/*  !"#$%&' */
  0x1D, 0x1C, 0x03, 0x01, 0x5D, 0x02, 0x1A, 0x04,	/* ()*+,-./ */
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,	/* 01234567 */
  0x18, 0x19, 0x5F, 0x5E, 0x0E, 0x0C, 0x0A, 0x5C,	/* 89:;<=>? */
  0x65, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,	/* @ABCDEFG */
  0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E,	/* HIJKLMNO */
  0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36,	/* PQRSTUVW */
  0x37, 0x38, 0x39, 0x73, 0x7A, 0x78, 0x05, 0x75,	/* XYZ[\]^_ */
  0x76, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,	/* `abcdefg */
  0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E,	/* hijklmno */
  0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56,	/* pqrstuvw */
  0x57, 0x58, 0x59					/* xyz */
};


#define MAXINPUTLINE 256
char input_line[MAXINPUTLINE];	/* buffer for the compiled line */
char *input_ptr;		/* pointer to the input_line buffer */

#define MAXOUTPUTLINE 64
uint output_line[MAXOUTPUTLINE];	/* buffer for the output line */
uint *output_ptr;		/* pointer to the output_line buffer */

#define MAXSEG 63		/* max. data segment size */
#define LEADER1 164		/* size of the header segment lead-in */
#define LEADER2 112		/* size of the data segment lead-in */


/* returns the Casio code of an ASCII character when conversion possible,
   otherwise returns 0x5C (Casio code for question mark) */
uint ascii2casio (char c)
{
  return (c>=' ' && c<='z') ? chartab[(uint) (c - ' ')] : 0x5C;
}


/* This function compares two strings without case sensitivity (first string
   pointed by ptr2, second string pointed by input_ptr), and updates the
   input_ptr when both string equivalent. */
int cmp_string (char *ptr2)
{
  char c1, c2;
  char *ptr1 = input_ptr;

  do {
    c1 = *ptr1++;
    if (islower(c1) != 0)
    {
      c1 = toupper(c1);
    }
    c2 = *ptr2++;
  } while (c1==c2 && c2!='\0');
  if (c2=='\0')
  {
    input_ptr = --ptr1;
  }
  return (c2=='\0') ? 0 : (int) (c1-c2);
}


/* This function searches the table of structure TREE for a string pointed
   by the input_ptr, and picks from the table the longest matching string.
   Returns pointer to the table and updated input_ptr when string found,
   or returns NULL and leaves input_ptr unchanged when not found.
   [Splint] The function creates a new pointer to an existing object pointed
   by root, thus the returned value is annotated as "dependent".	*/
/*@dependent@*//*@null@*/ TREE *find_tree (TREE *root)
{
  TREE *treeptr=NULL;
  char *ptr1=input_ptr;
  char *maxptr=input_ptr;
  int compare;

  while (root != NULL)
  {
    input_ptr=ptr1;
    if ((compare = cmp_string(root->tree_str)) == 0)
    {
      treeptr=root;
      maxptr=input_ptr;
    }
    root = (compare<0) ? root->left : root->right;
  }
  input_ptr = (treeptr != NULL) ? maxptr : ptr1;
  return treeptr;
}


/* converts an 8-bit value to BCD */
uint bin2bcd (uint x)
{
  uint y;
  y = x/10;
  return 16*y+x%10;
}


/* returns TRUE when input_line successfully parsed */
bool parse (void)
{
  uint line_number;
  TREE *treeptr1;
  bool quotes = FALSE;
  bool digit = FALSE;

  input_ptr = input_line;
  output_ptr = output_line;


/* skip leading spaces */
  while (isspace(*input_ptr) != 0)
  {
    input_ptr++;
  }

/* line number */
  while (isdigit(*input_ptr) != 0)
  {
    input_ptr++;
  }
  if (input_ptr == input_line)
  {
    return FALSE;		/* missing line number */
  }
  (void) sscanf(input_line, "%u", &line_number);
  *output_ptr++ = bin2bcd(line_number%100);
  *output_ptr++ = bin2bcd(line_number/100);

  while (*input_ptr!='\0' && output_ptr<output_line+MAXOUTPUTLINE)
  {

/* quotation mark ? */
    if (*input_ptr == '"')
    {
      *output_ptr++ = 0x07;	/* Casio code for the quotation mark */
      input_ptr++;
      quotes = !quotes;
      digit = FALSE;
    }

/* characters within quotes copied literally */
    else if (quotes)
    {
      if (iscntrl(*input_ptr) == 0)
      {
        *output_ptr++ = ascii2casio(*input_ptr);
      }
      input_ptr++;
      digit = FALSE;
    }

/* process characters outside quotes */
    else
    {

/* an attempt to tokenize */
      if ((treeptr1 = find_tree(KEYWORDS_ROOT)) != NULL)
      {
        *output_ptr++ = treeptr1->tree_val;
        digit = FALSE;
      }

/* exponent? */
      else if (digit && *input_ptr=='E')
      {
        input_ptr++;
        if (*input_ptr=='-')
        {
          *output_ptr++ = 0x1E;
          input_ptr++;
        }
        else
        {
          *output_ptr++ = 0x1F;
        }
        digit = FALSE;
      }

/* colon ? */
      else if (*input_ptr == ':')
      {
        *output_ptr++ = 0xFE;
        input_ptr++;
      }

/* carriage return ? */
      else if (*input_ptr == '\n')
      {
        *output_ptr++ = 0xFF;
        input_ptr++;
      }

/* skip blanks */
      else if (isspace(*input_ptr) != 0)
      {
        input_ptr++;
      }

/* other characters copied literally */
      else
      {
        *output_ptr++ = ascii2casio(*input_ptr);
        digit = (isdigit(*input_ptr) != 0);
        input_ptr++;
      }
    }

  }

  return (*input_ptr == '\0');
}


void encode(uint x)
{
  uint i;

/* calculate the parity bit */
  for (i=0x01; i<=0x80; i<<=1)
  {
    if ((x & i) != 0)
    {
      x ^= 0x0100;
    }
  }
/* add the start and stop bits */
  x <<= 1;
  x |= 0xfc00;
  printf ("%c%c", (char) ((x & 0x3F) + 0x30), (char) (((x>>6) & 0x3F) + 0x30));
}


/* inserts a new line every 32 character pairs */
#define crlf	if ((char_counter++ & 0x1F) == 0) printf ("\n");


int main (int argc, char *argv[])
{
  FILE *infp;
  uint errors = 0;
  uint line_counter = 0;
  uint char_counter = 0;
  uint seg_counter = MAXSEG;
  uint i;
  uint *ptr;

  if (argc<=1)
  {
    fprintf (stderr, "\nMissing file name, program aborted\n");
    return 1;
  }

  if ((infp = fopen(*++argv, "rt")) == NULL)
  {
    fprintf (stderr, "\nCannot open the file %s\n",*argv);
    return 1;
  }

/* lead-in of the header segment */
  for (i=0; i<LEADER1; i++)
  {
    crlf;
    printf ("oo");
  }

/* determine the file name length */
  for (i=0; i<8; i++)		/* up to 8 characters */
  {
    if ((*argv)[i] == '.' || (*argv)[i] == '\0')
    {
      break;
    }
  }

/* the header segment */
  encode (0xD0 + i);		/* the file identifier + the name length */
  for (i=0; i<8; i++)		/* 8 characters of the file name */
  {
    encode (ascii2casio ((*argv)[i]));
  }
  char_counter += 9;
/* the remaining 2 bytes with the line number will be added later */

/* the data segments */
  while (fgets(input_line, MAXINPUTLINE, infp)!=NULL && *input_line!='\032')
  {
    line_counter++;
    if (parse())
    {

/* last 2 bytes of the header segment */
      if (char_counter == LEADER1+9)
      {
        encode(output_line[0]);
        encode(output_line[1]);
        char_counter += 2;
      }

/* optional lead-in of the data segment */
      if (output_ptr-output_line > MAXSEG-seg_counter)
      {
        seg_counter = 0;

/* the End Of Data Segment marker = 0xF1 */
        if (char_counter != LEADER1+9+2)
        {
          crlf;
          printf ("Ro");
        }

/* the lead-in */
        for (i=0; i<LEADER2; i++)
        {
          crlf;
          printf ("oo");
        }

/* the Start Of Data Segment marker = 0x02 */
        crlf;
        printf ("4h");
      }

/* add the BASIC line to the data segment */
      seg_counter += output_ptr-output_line;
      for (ptr=output_line; ptr<output_ptr; ptr++)
      {
        crlf;
        encode(*ptr);
      }
    }

    else
    {
      fprintf (stderr, "\nError(s) found in line %u", line_counter);
      errors++;
    }
  }

/* the End Of Data Segment marker = 0xF0 */
  {
    crlf;
    printf ("Pg");
  }

  fprintf (stderr, "\n\nErrors: %u\n", errors);

  (void) fclose (infp);
  return 0;
}
