{ Casio NM-325 / NM-326 assembler }

unit Asem;


interface

  uses
    SysUtils;

  var
{ assembler input }
    loc: word;
    InBuf: string[64];
    InIndex: integer;
{ assembler output }
    OutBuf: array[0..2] of byte;
    OutIndex: integer;

  procedure Assemble;


implementation


type

  t_kind = (
    INHERENT,	{ no operands, single byte }
    REGIST,	{ single register }
    ARITHM,	{ register/memory,register/immediate }
    LDS_TYPE,	{ register,register }
    LDL_TYPE,	{ register,register }
    SEARCH,	{ register,register }
    GST_ARG,	{ register,status_register }
    PST_ARG,	{ status_register,register/immediate }
    LD_ARG,	{ register,various }
    ST_ARG,	{ memory,register }
    WREGIM16,	{ word_register,immediate }
    SBWADW,	{ index_register,register/immediate }
    IJMP_ARG,	{ word_register }
    ARYARYIM5,	{ array,array/immediate }
    LDM_ARG,	{ array,various }
    STM_ARG,	{ memory,array }
    STLM_ARG,	{ memory,array }
    ARY,	{ single array }
    RETURN_ARG,	{ cond_code }
    JUMP_ARG,	{ cond_code,immediate }
    SJUMP_ARG,	{ cond_code,short_immediate }
    LDSM_TYPE,	{ array,array }
    LDLM_TYPE,	{ array,array }
    DB_ARG	{ up to 3 data bytes }
  );


  tab = record
    str: string[6];
    val2: word;
    case boolean of
      True: (kind: t_kind);
      False: (val1: word);
  end;


const

  MINUS	= $0000;
  PLUS = $0004;
  ANYTHING = 0;		{ unused value }

  NMNEM = 131;		{ index of the last item in the 'mnem' array }

  mnem: array [0..NMNEM] of tab = (
{ no operands }
    (	str:'off';	val2:$00AF;	kind:INHERENT	),
    (	str:'wai';	val2:$00BF;	kind:INHERENT	),
    (	str:'rti';	val2:$00CF;	kind:INHERENT	),
    (	str:'nop';	val2:$00EF;	kind:INHERENT	),
    (	str:'trp';	val2:$00FF;	kind:INHERENT	),

{ single register }
    (	str:'rod';	val2:$0060;	kind:REGIST	),
    (	str:'rou';	val2:$8060;	kind:REGIST	),
    (	str:'did';	val2:$0061;	kind:REGIST	),
    (	str:'diu';	val2:$8061;	kind:REGIST	),
    (	str:'tcmpb';	val2:$0062;	kind:REGIST	),
    (	str:'cmpb';	val2:$8062;	kind:REGIST	),
    (	str:'tcmp';	val2:$0063;	kind:REGIST	),
    (	str:'cmp';	val2:$8063;	kind:REGIST	),
    (	str:'byd';	val2:$0064;	kind:REGIST	),
    (	str:'byu';	val2:$8064;	kind:REGIST	),
    (	str:'bnd';	val2:$0065;	kind:REGIST	),
    (	str:'bnu';	val2:$8065;	kind:REGIST	),
    (	str:'tmtb';	val2:$0066;	kind:REGIST	),
    (	str:'mtb';	val2:$8066;	kind:REGIST	),
    (	str:'tbiu';	val2:$0067;	kind:REGIST	),
    (	str:'biu';	val2:$8067;	kind:REGIST	),
    (	str:'bdn';	val2:$009E;	kind:REGIST	),
    (	str:'bup';	val2:$809E;	kind:REGIST	),

{ GST operands - register,status_register }
    (	str:'gst';	val2:$8098;	kind:GST_ARG	),

{ PST operands - status_register,register/immediate }
    (	str:'pst';	val2:$0098;	kind:PST_ARG	),

{ arithmetical/logical operations, byte size
  register,register
  register,immediate
  memory,register
  memory,immediate }
    (	str:'na';	val2:$8000;	kind:ARITHM	),
    (	str:'xr';	val2:$8001;	kind:ARITHM	),
    (	str:'sbb';	val2:$8002;	kind:ARITHM	),
    (	str:'sb';	val2:$8003;	kind:ARITHM	),
    (	str:'an';	val2:$8004;	kind:ARITHM	),
    (	str:'or';	val2:$8005;	kind:ARITHM	),
    (	str:'adb';	val2:$8006;	kind:ARITHM	),
    (	str:'ad';	val2:$8007;	kind:ARITHM	),
    (	str:'tna';	val2:$0000;	kind:ARITHM	),
    (	str:'txr';	val2:$0001;	kind:ARITHM	),
    (	str:'tsbb';	val2:$0002;	kind:ARITHM	),
    (	str:'tsb';	val2:$0003;	kind:ARITHM	),
    (	str:'tan';	val2:$0004;	kind:ARITHM	),
    (	str:'tor';	val2:$0005;	kind:ARITHM	),
    (	str:'tadb';	val2:$0006;	kind:ARITHM	),
    (	str:'tad';	val2:$0007;	kind:ARITHM	),
{ the first operand can be register only }
    (	str:'sbbc';	val2:$8022;	kind:ARITHM	),
    (	str:'sbc';	val2:$8023;	kind:ARITHM	),
    (	str:'adbc';	val2:$8026;	kind:ARITHM	),
    (	str:'adc';	val2:$8027;	kind:ARITHM	),
    (	str:'tsbbc';	val2:$0022;	kind:ARITHM	),
    (	str:'tsbc';	val2:$0023;	kind:ARITHM	),
    (	str:'tadbc';	val2:$0026;	kind:ARITHM	),
    (	str:'tadc';	val2:$0027;	kind:ARITHM	),
{ different encoding }
    (	str:'bit';	val2:$0019;	kind:ARITHM	),

{ register,register }
    (	str:'xc';	val2:$4021;	kind:LDS_TYPE	),
    (	str:'xcs';	val2:$4025;	kind:LDS_TYPE	),
    (	str:'lds';	val2:$C025;	kind:LDS_TYPE	),
{ instruction SWP can accept a single register as well }
    (	str:'swp';	val2:$4024;	kind:LDS_TYPE	),

{ register,register }
    (	str:'xcl';	val2:$402D;	kind:LDL_TYPE	),
    (	str:'xch';	val2:$4028;	kind:LDL_TYPE	),
    (	str:'ldl';	val2:$C02D;	kind:LDL_TYPE	),
    (	str:'ldh';	val2:$C028;	kind:LDL_TYPE	),
    (	str:'xcls';	val2:$4029;	kind:LDL_TYPE	),
    (	str:'xchs';	val2:$402C;	kind:LDL_TYPE	),
    (	str:'ldls';	val2:$C029;	kind:LDL_TYPE	),
    (	str:'ldhs';	val2:$C02C;	kind:LDL_TYPE	),

{ register,register }
    (	str:'sdn';	val2:$001C;	kind:SEARCH	),
    (	str:'sup';	val2:$801C;	kind:SEARCH	),

{ LD operands - register,various }
    (	str:'ld';	val2:0;		kind:LD_ARG	),

{ ST operands - memory,register }
    (	str:'st';	val2:0;		kind:ST_ARG	),

{ LDW operands - word_register,immediate }
    (	str:'ldw';	val2:$00D0;	kind:WREGIM16	),

{ ADW/SBW operands - index_register,register/immediate }
    (	str:'sbw';	val2:$00D8;	kind:SBWADW	),
    (	str:'adw';	val2:$00DC;	kind:SBWADW	),

{ PLA/IJMP operands - word_register }
    (	str:'pla';	val2:$0018;	kind:IJMP_ARG	),
    (	str:'ijmp';	val2:$8018;	kind:IJMP_ARG	),

{ single array }
    (	str:'rodm';	val2:$0070;	kind:ARY	),
    (	str:'roum';	val2:$8070;	kind:ARY	),
    (	str:'didm';	val2:$0071;	kind:ARY	),
    (	str:'dium';	val2:$8071;	kind:ARY	),
    (	str:'tcmpbm';	val2:$0072;	kind:ARY	),
    (	str:'cmpbm';	val2:$8072;	kind:ARY	),
    (	str:'tcmpm';	val2:$0073;	kind:ARY	),
    (	str:'cmpm';	val2:$8073;	kind:ARY	),
    (	str:'bydm';	val2:$0074;	kind:ARY	),
    (	str:'byum';	val2:$8074;	kind:ARY	),
    (	str:'bndm';	val2:$0075;	kind:ARY	),
    (	str:'bnum';	val2:$8075;	kind:ARY	),
    (	str:'tmtbm';	val2:$0076;	kind:ARY	),
    (	str:'mtbm';	val2:$8076;	kind:ARY	),
    (	str:'tbium';	val2:$0077;	kind:ARY	),
    (	str:'bium';	val2:$8077;	kind:ARY	),

{ RTN operands - cond_code }
    (	str:'rtn';	val2:$00C7;	kind:RETURN_ARG	),

{ JMP/CAL operands - cond_code,immediate }
    (	str:'jmp';	val2:$00A7;	kind:JUMP_ARG	),
    (	str:'cal';	val2:$00B7;	kind:JUMP_ARG	),

{ SJMP/SCAL operands - cond_code,short_immediate }
    (	str:'sjmp';	val2:$00E7;	kind:SJUMP_ARG	),
    (	str:'scal';	val2:$00F7;	kind:SJUMP_ARG	),

{ arithmetical multibyte operations - array,array/immediate }
    (	str:'sbbm';	val2:$8012;	kind:ARYARYIM5	),
    (	str:'sbm';	val2:$8013;	kind:ARYARYIM5	),
    (	str:'adbm';	val2:$8016;	kind:ARYARYIM5	),
    (	str:'adm';	val2:$8017;	kind:ARYARYIM5	),
    (	str:'sbbcm';	val2:$8032;	kind:ARYARYIM5	),
    (	str:'sbcm';	val2:$8033;	kind:ARYARYIM5	),
    (	str:'adbcm';	val2:$8036;	kind:ARYARYIM5	),
    (	str:'adcm';	val2:$8037;	kind:ARYARYIM5	),
    (	str:'tsbbm';	val2:$0012;	kind:ARYARYIM5	),
    (	str:'tsbm';	val2:$0013;	kind:ARYARYIM5	),
    (	str:'tadbm';	val2:$0016;	kind:ARYARYIM5	),
    (	str:'tadm';	val2:$0017;	kind:ARYARYIM5	),
    (	str:'tsbbcm';	val2:$0032;	kind:ARYARYIM5	),
    (	str:'tsbcm';	val2:$0033;	kind:ARYARYIM5	),
    (	str:'tadbcm';	val2:$0036;	kind:ARYARYIM5	),
    (	str:'tadcm';	val2:$0037;	kind:ARYARYIM5	),
{ logical multibyte operations - array,array only }
    (	str:'nam';	val2:$8010;	kind:ARYARYIM5	),
    (	str:'xrm';	val2:$8011;	kind:ARYARYIM5	),
    (	str:'anm';	val2:$8014;	kind:ARYARYIM5	),
    (	str:'orm';	val2:$8015;	kind:ARYARYIM5	),
    (	str:'tnam';	val2:$0010;	kind:ARYARYIM5	),
    (	str:'txrm';	val2:$0011;	kind:ARYARYIM5	),
    (	str:'tanm';	val2:$0014;	kind:ARYARYIM5	),
    (	str:'torm';	val2:$0015;	kind:ARYARYIM5	),

{ array,array }
    (	str:'xcm';	val2:$4031;	kind:LDSM_TYPE	),
    (	str:'xcsm';	val2:$4035;	kind:LDSM_TYPE	),
    (	str:'ldsm';	val2:$C035;	kind:LDSM_TYPE	),
{ instruction SWPM can accept a single array as well }
    (	str:'swpm';	val2:$4034;	kind:LDSM_TYPE	),

{ array,array }
    (	str:'xclm';	val2:$403D;	kind:LDLM_TYPE	),
    (	str:'xchm';	val2:$4038;	kind:LDLM_TYPE	),
    (	str:'ldlm';	val2:$C03D;	kind:LDLM_TYPE	),
    (	str:'ldhm';	val2:$C038;	kind:LDLM_TYPE	),
    (	str:'xclsm';	val2:$4039;	kind:LDLM_TYPE	),
    (	str:'xchsm';	val2:$403C;	kind:LDLM_TYPE	),
    (	str:'ldlsm';	val2:$C039;	kind:LDLM_TYPE	),
    (	str:'ldhsm';	val2:$C03C;	kind:LDLM_TYPE	),

{ LDM operands - array,various }
    (	str:'ldm';	val2:0;		kind:LDM_ARG	),

{ STM operands - memory,array }
    (	str:'stm';	val2:0;		kind:STM_ARG	),

{ STLM operands - memory,array }
    (	str:'stlm';	val2:$0078;	kind:STLM_ARG	),

{ pseudo instruction }
    (	str:'db';	val2:ANYTHING;	kind:DB_ARG	)
  );


{ status register names for the commands GST, PST
  val2: modifies the opcode for the immediate addr. mode
  val1: lower significant byte modifies the opcode for the register addr. mode
  set bits of 'val1' specify which commands accept the 'str' status register }

  PSTREG = $0100;
  PSTIM7 = $0200;
  PSTIM8 = $0400;

  NSREGS = 12;		{ index of the last item in the 'sregs' array }

  sregs: array[0..NSREGS] of tab = (
    (	str:'as'; val2:$0098 xor $808D;	val1:$0005+PSTREG+PSTIM7	),
    (	str:'ds'; val2:$0098 xor $0089;	val1:$0001+PSTREG+PSTIM8	),
    (	str:'f';  val2:0;		val1:$0004+PSTREG		),
    (	str:'ie'; val2:$0098 xor $0088;	val1:$0000+PSTREG+PSTIM8	),
    (	str:'ky'; val2:$0098 xor $008A;	val1:$0002+PSTREG+PSTIM8	),
    (	str:'pd'; val2:$0098 xor $008B;	val1:$0003+PSTREG+PSTIM8	),
    (	str:'pe'; val2:$0098 xor $808F;	val1:PSTIM7			),
    (	str:'s4'; val2:$0098 xor $008C;	val1:PSTIM7			),
    (	str:'s5'; val2:$0098 xor $008D;	val1:PSTIM7			),
    (	str:'s6'; val2:$0098 xor $008E;	val1:PSTIM7			),
    (	str:'s7'; val2:$0098 xor $008F;	val1:PSTIM7			),
    (	str:'s8'; val2:$0098 xor $808C;	val1:PSTIM7			),
    (	str:'tm'; val2:$0098 xor $808E;	val1:PSTIM7			)
  );


  NIREGS = 3;		{ index of the last item in the 'iregs' array }

{ table of index registers }
  iregs: array[0..NIREGS] of tab = (
    (	str:'ix';	val2:$0000;	val1:$0040	),
    (	str:'iy';	val2:$0001;	val1:$0050	),
    (	str:'iz';	val2:$0002;	val1:0		),
    (	str:'sp';	val2:$0003;	val1:0		)
  );


  NWREGS = 7;		{ index of the last item in the 'wregs' array }

{ table of word size registers }
  wregs: array[0..NWREGS] of tab = (
    (	str:'ix';	val2:$0000;	val1:ANYTHING	),
    (	str:'iy';	val2:$0001;	val1:ANYTHING	),
    (	str:'iz';	val2:$0002;	val1:ANYTHING	),
    (	str:'sp';	val2:$0007;	val1:ANYTHING	),
    (	str:'v0';	val2:$0006;	val1:ANYTHING	),
    (	str:'v1';	val2:$0005;	val1:ANYTHING	),
    (	str:'v2';	val2:$0004;	val1:ANYTHING	),
    (	str:'v3';	val2:$0003;	val1:ANYTHING	)
  );


  NCC = 13;		{ index of the last item in the 'cc' array }

{ table of conditional codes }
  cc: array[0..NCC] of tab = (
    (	str:'nk';	val2:$0000 xor $0007;	val1:ANYTHING	),
    (	str:'nlz';	val2:$0001 xor $0007;	val1:ANYTHING	),
    (	str:'nuz';	val2:$0002 xor $0007;	val1:ANYTHING	),
    (	str:'nz';	val2:$0003 xor $0007;	val1:ANYTHING	),
    (	str:'nv';	val2:$0004 xor $0007;	val1:ANYTHING	),
    (	str:'nh';	val2:$0005 xor $0007;	val1:ANYTHING	),
    (	str:'nc';	val2:$0006 xor $0007;	val1:ANYTHING	),
    (	str:'k';	val2:$0008 xor $0007;	val1:ANYTHING	),
    (	str:'lz';	val2:$0009 xor $0007;	val1:ANYTHING	),
    (	str:'uz';	val2:$000A xor $0007;	val1:ANYTHING	),
    (	str:'z';	val2:$000B xor $0007;	val1:ANYTHING	),
    (	str:'v';	val2:$000C xor $0007;	val1:ANYTHING	),
    (	str:'h';	val2:$000D xor $0007;	val1:ANYTHING	),
    (	str:'c';	val2:$000E xor $0007;	val1:ANYTHING	)
  );


{ compare the string 's' with the 'InBuf' at location 'InIndex' without
  the case sensitivity,
  update the 'InIndex' and return True if both string match }
function ParseString (s: string): boolean;
var
  n: integer;
begin
  ParseString := False;
  if InIndex + Length(s) - 1 > Length(InBuf) then exit;
  n := 0;
  while n < Length(s) do
  begin
    if s[n+1] <> LowerCase(InBuf[InIndex + n]) then exit;
    Inc (n);
  end {while};
  Inc (InIndex, n);
  ParseString := True;
end {ParseString};


{ This function searches the table for a string pointed to by the InIndex,
  and picks from the table the longest matching string.
  Returns index to the table and updates InIndex when string found,
  or leaves InIndex unchanged when not found. }
function ParseTable (
  out x: integer;	{ returned index to the table }
  var t: array of tab;	{ table to be searched }
  last: integer		{ index of the last item }
  ) : boolean;		{ TRUE when string found }
var
  maxindex, save, i: integer;
begin
  maxindex := InIndex;
  save := InIndex;
  ParseTable := FALSE;
  for i := 0 to last do
  begin
    InIndex := save;
    if ParseString (t[i].str) and (InIndex > maxindex) then
    begin
      ParseTable := TRUE;
      x := i;
      maxindex := InIndex;
    end {if};
  end {for};
  InIndex := maxindex;
end {ParseTable};


{ a specified character expected }
function ParseChar (c: char) : boolean;
begin
  result := (InIndex <= Length(InBuf)) and (InBuf[InIndex] = c);
  if result then Inc (InIndex);
end {ParseChar};


{ move the 'InIndex' to the first character different from space }
procedure SkipBlanks;
begin
  while ParseChar (' ') do ;
end {SkipBlanks};


{ comma expected }
function ParseComma : boolean;
begin
  SkipBlanks;
  ParseComma := ParseChar (',');
  SkipBlanks;
end {ParseComma};


{ value of a hex digit }
function GetDigit (c: char) : integer;
const
  digits: string[22] = '0123456789ABCDEFabcdef';
var
  i: integer;
begin
  i := 1;
  while (i<=22) and (c <> digits[i]) do Inc (i);
  if i>16 then GetDigit := i-7 else GetDigit := i-1;
end {GetDigit};


{ the function expects a number in base 'radix',
  updates the InIndex }
function ParseNumber (out value: word; radix: word): boolean;
var
  x, y: word;
begin
  value := 0;
  ParseNumber := FALSE;
  while InIndex <= Length(InBuf) do
  begin
    x := word(GetDigit(InBuf[InIndex]));
    if x >= radix then break;	{ stop when not a digit }
    y := value*radix + x;
    if y < value then break;	{ overflow, stop when too much digits }
    value := y;
    Inc (InIndex);
    ParseNumber := TRUE;
  end;
end {ParseNumber};


{ the function expects a hexadecimal (with a prefix &H) or decimal number
  within specified range }
function EvalAndTest (out value: word; range: word) : boolean;
var
  radix: word;
begin
  EvalAndTest := FALSE;
  SkipBlanks;
{ parse for the prefix of a hexadecimal numeral }
  if ParseString ('&h') then radix := 16 else radix := 10;
  if not ParseNumber (value, radix) then exit;	{ failure, missing number }
  EvalAndTest := (value <= range);		{ FALSE when out of range }
end {EvalAndTest};


{ the function expects a main register, returns the register index,
  index value in range 128..191 if indirect addressing specified }
function RegArgum (out x: word) : boolean;
var
  y: word;
begin
  SkipBlanks;
  RegArgum := FALSE;
  y := 0;
  if ParseString ('i') then y := $80
  else if not ParseString ('r') then exit;	{ failure, invalid register }
  if not ParseNumber (x, 10) then exit;		{ failure, missing number }
  if x > 127 then exit;				{ failure, out of range }
  Inc (x,y);
  RegArgum := (x < 192);
end {RegArgum};


{ swap the contents of two word-type variables }
procedure SwapWords (var w1, w2: word);
var
  temp: word;
begin
  temp := w1;
  w1 := w2;
  w2 := temp;
end {SwapWords};


function Reg6 (x: word) : word;
begin
  Reg6 := ((x shr 3) and $17) or ((x shl 5) and $E0);
end {Reg6};


function Reg9 (x, y: word) : word;
begin
  Reg9 := ((x shr 3) and $17) or ((y shl 5) and $E0);
end {Reg9};


{ expects a sign '+' or '-' }
function ParseSign (out x: word) : boolean;
begin
  SkipBlanks;
  x := PLUS;
  ParseSign := TRUE;
  if not ParseChar ('+') then
  begin
    x := MINUS;
    ParseSign := ParseChar ('-');
  end {if};
end {ParseSign};


{ the function continues parsing an array after the first register has been
  recognised }
function AryArgum (
  first: word;		{ already recognised first register of the array }
  var last: word	{ last register of the array }
  ) : boolean;		{ TRUE when success }
begin
  AryArgum := FALSE;
  if not ParseString ('..') then exit;		{ failure, invalid argument }
  if not RegArgum (last) then exit;		{ failure, register expected }
  AryArgum := (first and $FFF8) = (last and $FFF8);
end {AryArgum};


{ opening parenthesis followed by an index register }
function ParIr (out x: integer {index to the table 'iregs'}) : boolean;
begin
  x := 0;
  ParIr := FALSE;
  if not ParseChar ('(') then exit;
  SkipBlanks;
  ParIr := ParseTable (x, iregs, NIREGS);
end {ParIr};


{ assemble the instruction in the InBuf and place the result in the OutBuf,
  on exit InIndex contains the position of an error (warning: it can point
  past the end of the InBuf !), otherwise 0 }
procedure Assemble;
var
  sign: word;
  x1, x2, x3, x4: word;

  i: integer;		{ index to the tables }
  kod: word;		{ opcode }
begin
  InIndex := 1;
  OutIndex := 0;

  SkipBlanks;				{ skip leading blanks }
  if (InIndex > Length(InBuf))		{ empty InBuf? }
	or ParseChar (';') then		{ comment? }
  begin
    InIndex := 0;
    exit;				{ success }
  end {if};

{ parse the mnemonic }
  if not ParseTable (i, mnem, NMNEM) then exit;
					{ failure, mnemonic not recognised }
  kod := mnem[i].val2;

{ parse the arguments }
  case mnem[i].kind of

    INHERENT:		{ no operands, single byte }
      begin
        OutBuf[0] := byte(kod);
        OutIndex := 1;
      end {case INHERENT};


    REGIST:		{ single register }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if x1 > 127 then exit;		{ failure, illegal indirect }
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutIndex := 2;
      end {case REGISTER};


    ARITHM:		{ register/memory,register/immediate }
      begin
{ first operand: register }
        if RegArgum (x1) then
        begin
          if not ParseComma then exit;	{ failure, comma expected }
  { second operand: register }
          if RegArgum (x2) then
          begin
            if ((x1 xor x2) and $40) = 0 then exit; { failure, register bank }
            if (x1 and $40) = 0 then
            begin
              SwapWords(x1,x2);
              x1 := x1 and $3F;
            end {if};
            x2 := Reg6(x2);
          end

  { second operand: immediate }
          else
          begin
            if x1 > 127 then exit;	{ failure, indirect not allowed }
  { instruction BIT is differently encoded }
            if kod = $0019 then x2 := $8000 else x2 := $0008;
            kod := kod xor x2;
            if not EvalAndTest(x2,$FF) then exit; { failure, invalid value }
          end {if};

          OutBuf[0] := byte(kod);
          OutBuf[1] := byte(x1) or Hi(kod);
          OutBuf[2] := byte(x2);
          OutIndex := 3;
        end

{ first operand: memory }
        else
        begin
	  if (kod and $0030) <> 0 then exit;	{ BIT, add/sub. with carry }
          if not ParIr (i) then exit;		{ failure }
          if iregs[i].val1 = 0 then exit;	{ failure, IZ/SP illegal }
          if not ParseChar (')') then exit;	{ failure }
          kod := kod xor iregs[i].val1;
          if not ParseComma then exit;		{ failure, comma expected }

  { second operand: register }
          if RegArgum (x2) then
          begin
            if x2 > 127 then exit;	{ failure, indirect not allowed }
            { bit 6 of the register specification is inverted }
            x2 := x2 xor $40 xor Hi(kod);
          end

  { second operand: immediate }
          else
          begin
            if (kod and $8000) = 0 then exit; { failure, test mode illegal }
            kod := kod xor $0008;
            if not EvalAndTest(x2,$FF) then exit; { failure, invalid value }
          end {if};

          OutBuf[0] := byte(kod);
          OutBuf[1] := byte(x2);
          OutIndex := 2;
        end {if};
      end {case ARITHM};


    LDS_TYPE:		{ register,register }
      begin
        if not RegArgum (x1) then exit;		{ failure, register expected }
        SkipBlanks;
        if InBuf[InIndex] = ',' then
        begin
          Inc(InIndex);
          if not RegArgum (x2) then exit;	{ failure, register expected }
          if ((x1 xor x2) and $40) = 0 then exit; { failure, register bank }
          if (x1 and $40) = 0 then
          begin
            SwapWords(x1,x2);
            kod := kod xor $4000;
          end {if};
        end
        else
        begin
 { instruction SWP can accept a single register }
          if kod <> $4024 then exit;		{ failure, comma expected }
          kod := kod xor $8000;
          x2 := 0;
          if (x1 and $40) = 0 then
          begin
            SwapWords(x1,x2);
            kod := kod xor $4000;
          end {if};
        end {if};
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1 and $3F) or Hi(kod);
        OutBuf[2] := byte(Reg6(x2));
        OutIndex := 3;
      end {LDS_TYPE};


    LDL_TYPE:		{ register,register }
      begin
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not ParseComma then exit;		{ failure, comma expected }
        if not RegArgum (x2) then exit;		{ failure, register expected }
        if ((x1 xor x2) and $40) = 0 then exit;	{ failure, register bank }
        if (x1 and $40) = 0 then
        begin
          SwapWords(x1,x2);
          kod := kod xor $4005;
        end {if};
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1 and $3F) or Hi(kod);
        OutBuf[2] := byte(Reg6(x2));
        OutIndex := 3;
      end {LDL_TYPE};


    SEARCH:		{ register,register }
      begin
{ first register in the bank 1 }
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if x1 > 127 then exit;		{ failure, indirect not allowed }
        if x1 < 64 then exit;		{ failure, bank 0 not allowed}
        if not ParseComma then exit;	{ failure, comma expected }
{ second register in the bank 0 (indirect specification allowed) }
        if not RegArgum (x2) then exit;	{ failure, register expected }
        if (x2 and $40) <> 0 then exit;	{ failure, bank 1 not allowed}
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutBuf[2] := byte(Reg6(x2));
        OutIndex := 3;
      end {SEARCH};


    GST_ARG:		{ register,status_register }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if x1 > 127 then exit;		{ failure, illegal indirect }
        if not ParseComma then exit;	{ failure, comma expected }
        if not ParseTable (i, sregs, NSREGS) then exit;
				{ failure, status register expected }
        if (sregs[i].val1 and PSTREG) = 0 then exit;
				{ failure, unsupported status register }
        kod := kod xor (sregs[i].val1 and $00FF);
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutIndex := 2;
      end {case GST_ARG};


    PST_ARG:		{ status_register,register/immediate }
      begin
        SkipBlanks;
        if not ParseTable (i, sregs, NSREGS) then exit;
				{ failure, status register expected }
        if not ParseComma then exit;	{ failure, comma expected }

{ second operand: register }
        if RegArgum (x1) then
        begin
          if x1 > 127 then exit;	{ failure, illegal indirect }
          if (sregs[i].val1 and PSTREG) = 0 then exit;
				{ failure, unsupported status register }
          kod := kod xor (sregs[i].val1 and $00FF);
        end
{ second operand: immediate }
        else
        begin
          x2 := 0;
          if (sregs[i].val1 and PSTIM7) <> 0 then x2 := $7F;
          if (sregs[i].val1 and PSTIM8) <> 0 then x2 := $FF;
          if x2 = 0 then exit;	{ failure, unsupported status register }
          if not EvalAndTest(x1,x2) then exit;	{ failure, invalid value }
          kod := kod xor sregs[i].val2;
        end {if};

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutIndex := 2;
      end {case PST_ARG};


    LD_ARG:		{ register,various }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if not ParseComma then exit;	{ failure, comma expected }

{ second operand: register }
        if RegArgum (x2) then
        begin
{ optional third immediate operand (indirect destination not allowed) }
          SkipBlanks;
          if InBuf[InIndex] = ',' then
          begin
            Inc(InIndex);
            if (x1 or x2) > 127 then exit;	{ failure, indirect illegal }
            if (x1 xor x2) <> $40 then exit; { failure, unaligned registers }
            if not EvalAndTest(x2,$FF) then exit; { failure, invalid value }
            kod := $0020;
          end
{ no third operand (indirect specification allowed) }
          else
          begin
            if ((x1 xor x2) and $40) = 0 then exit; { failure, register bank }
            if (x1 and $40) = 0 then
            begin
              SwapWords(x1,x2);
              x1 := x1 and $3F;
            end {if};
            x2 := Reg6(x2);
            kod := $8021;
          end {if};
        end {if second operand: register}

{ second operand: memory (indirect destination not allowed) }
        else if ParIr (i) then
        begin
          kod := iregs[i].val2;
  { postincrement/postdecrement }
          if ParseChar (')') then
          begin
            if not ParseSign (sign) then exit;	{ failure }
            if x1 > 127 then exit;	{ failure, indirect not allowed }
            Inc (kod,$8068);
          end

  { with an immediate offset }
          else
          begin
            if not ParseSign (sign) then exit;	{ failure }
            if not EvalAndTest(x2,$FF) then exit; { failure, invalid value }
            if not ParseChar (')') then exit;	{ failure }
            Inc (kod,$8080);
          end {if};
          Inc (kod,sign);
        end {if second operand: memory}

{ second operand: immediate (indirect destination not allowed) }
        else
        begin
          if x1 > 127 then exit;	{ failure, indirect not allowed }
          if not EvalAndTest(x2,$FF) then exit;	{ failure, invalid value }
          kod := $8020;
        end {if};

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1 or $80);
        OutBuf[2] := byte(x2);
        if (kod and $0040) = 0 then OutIndex := 3 else OutIndex := 2;
      end {case LD_ARG};


    ST_ARG:		{ memory,register/immediate }
      begin
{ first operand: predecement/preincrement }
        if ParseSign (sign) then
        begin
          if not ParIr (i) then exit;		{ failure }
          kod := $0068;
        end
{ first operand: index register with an immediate offset }
        else
        begin
          if not ParIr (i) then exit;		{ failure }
          if not ParseSign (sign) then exit;	{ failure }
          if not EvalAndTest(x2,256) then exit;	{ failure, invalid value }
          if x2 = 0 then exit;			{ failure, invalid value }
          Dec (x2);
          kod := $0080;
        end {if};

        if not ParseChar (')') then exit;	{ failure }
        if not ParseComma then exit;		{ failure, comma expected }

{ second operand: register (indirect not allowed) }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if x1 > 127 then exit;		{ failure, indirect not allowed }

        OutBuf[0] := byte(kod + sign + iregs[i].val2);
        OutBuf[1] := byte(x1);
        OutBuf[2] := byte(x2);
        if kod = $00080 then OutIndex := 3 else OutIndex := 2;
      end {case ST_ARG};


    WREGIM16:		{ word_register,immediate }
      begin
        SkipBlanks;
        if not ParseTable (i, wregs, NWREGS) then exit;	{ failure }
        kod := kod xor wregs[i].val2;
        if not ParseComma then exit;		{ failure, comma expected }
        if not EvalAndTest(x1,$FFFF) then exit;	{ failure, invalid value }
        OutBuf[0] := byte(kod);
        OutBuf[1] := Hi(x1);
        OutBuf[2] := byte(x1);
        OutIndex := 3;
      end {case WREGIM16};


    SBWADW:		{ index_register,register/immediate }
      begin
        SkipBlanks;
        if not ParseTable (i, iregs, NWREGS) then exit;	{ failure }
        kod := kod xor iregs[i].val2;
        if not ParseComma then exit;		{ failure, comma expected }
  { second operand: register }
        if RegArgum (x1) then
        begin
          if x1 > 127 then exit;	{ failure, indirect not allowed }
          x1 := x1 or $80;
        end
  { second operand: immediate }
        else
        begin
          if not EvalAndTest(x1,$7F) then exit;	{ failure, invalid value }
        end {if};
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1);
        OutIndex := 2;
      end {case SBWADW};


    IJMP_ARG:		{ word_register }
      begin
        SkipBlanks;
        if ParseTable (i, wregs, NWREGS) then
        begin
          x1 := 56 + wregs[i].val2;
          x2 := x1;
        end
        else
        begin
          if not RegArgum (x1) then exit;	{ failure, register expected }
          if x1 > 127 then exit;	{ failure, indirect not allowed }
          if not ParseComma then exit;		{ failure, comma expected }
          if not RegArgum (x2) then exit;	{ failure, register expected }
          if ((x1 and $40) = 0) or ((x2 and $40) <> 0) then exit;
						{ failure, register bank }
          x1 := x1 and $3F;
        end {if};
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutBuf[2] := byte(Reg6(x2));
        OutIndex := 3;
      end {case IJMP_ARG};


    ARYARYIM5:		{ array,array/immediate }
      begin
{ first operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        if not ParseComma then exit;		{ failure, comma expected }

{ second operand: array }
        if RegArgum (x3) then
        begin
          if not AryArgum (x3, x4) then exit;	{ failure, invalid array }
          if ((x1 xor x3) and $40) = 0 then exit;	{ failure, reg. bank }
  { check the alignment, 'invalid array' error when fail }
          if (x1 and $0007) <> (x3 and $0007) then exit;	{ failure }
          if (x2 and $0007) <> (x4 and $0007) then exit;	{ failure }
          if (x1 and $40) = 0 then
          begin
            SwapWords (x1,x3);
            SwapWords (x2,x4);
            x1 := x1 and $3F;
          end {if};
          x2 := Reg9(x3,x4);
        end {if second operand: array or first register}

{ second operand: immediate (indirect destination not allowed) }
        else
        begin
  { logical operations don't allow an immediate operand }
          if (kod and $0002) = 0 then exit;	{ failure }
          kod := kod xor $0008;
  { indirect destination illegal }
          if x1 > 127 then exit;	{ failure, indirect not allowed }
  { immediate operand }
          if not EvalAndTest(x3,$1F) then exit;	{ failure, invalid value }
          x2 := Reg9(0,x2) or x3;
        end {if};

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case ARYARYIM5};


    LDM_ARG:		{ array,various }
      begin
{ first operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        if not ParseComma then exit;		{ failure, comma expected }

{ second operand: array }
        if RegArgum (x3) then
        begin
          if not AryArgum (x3, x4) then exit;	{ failure, invalid array }
{ optional third immediate operand (indirect destination not allowed) }
          SkipBlanks;
          if InBuf[InIndex] = ',' then
          begin
            Inc(InIndex);
            if (x1 or x3) > 127 then exit;	{ failure, indirect illegal }
            if (x1 xor x3) <> $40 then exit; { failure, unaligned registers }
            if (x2 xor x4) <> $40 then exit; { failure, unaligned registers }
            kod := $0030;
            if not EvalAndTest(x3,$1F) then exit; { failure, invalid value }
            x2 := Reg9(0,x2) or x3;
          end
{ no third operand (indirect specification allowed) }
          else
          begin
            if ((x1 xor x3) and $40) = 0 then exit;	{ failure, reg. bank }
  { check the alignment, 'invalid array' error when fail }
            if (x1 and $0007) <> (x3 and $0007) then exit;	{ failure }
            if (x2 and $0007) <> (x4 and $0007) then exit;	{ failure }
            if (x1 and $40) = 0 then
            begin
              SwapWords (x1,x3);
              SwapWords (x2,x4);
              x1 := x1 and $3F;
            end {if};
            kod := $8031;
            x2 := Reg9(x3,x4);
          end {if};
        end {if second operand: array}

{ second operand: memory (indirect destination not allowed) }
        else if ParIr (i) then
        begin
          if x1 > 127 then exit;	{ failure, indirect not allowed }

  { postincrement/postdecrement }
          if ParseChar(')') then
          begin
            if not ParseSign (sign) then exit;		{ failure }
            kod := $8078;
            if (x1 and $40) = 0 then
            begin
              x1 := x1 and $07;
              x2 := Reg6 (x2);
            end
            else
            begin
              x2 := Reg9(0,x2);
            end {if};
          end

  { with an immediate offset }
          else
          begin
            if not ParseSign (sign) then exit;		{ failure }
            if not EvalAndTest(x3,$1F) then exit; { failure, invalid value }
            if not ParseChar (')') then exit;		{ failure }
            kod := $8090;
            x2 := Reg9(0,x2) or x3;
          end {if};

          Inc (kod, sign + iregs[i].val2);
        end {if second operand: memory}

{ second operand: immediate (indirect destination not allowed) }
        else
        begin
          if x1 > 127 then exit;	{ failure, indirect not allowed }
          kod := $8030;
          if not EvalAndTest(x3,$1F) then exit; { failure, invalid value }
          x2 := Reg9(0,x2) or x3;
        end {if second operand: immediate};

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case LDM_ARG};


    STM_ARG:		{ memory,array }
      begin
{ first operand: predecement/preincrement }
        if ParseSign (sign) then
        begin
          if not ParIr (i) then exit;		{ failure }
          kod := $0078;
        end

{ first operand: index register with an immediate offset }
        else
        begin
          if not ParIr (i) then exit;		{ failure }
          if not ParseSign (sign) then exit;	{ failure }
          if not EvalAndTest(x3,$20) then exit;	{ failure, invalid value }
          if x3 = 0 then exit;			{ failure, invalid value }
          Dec(x3);
          kod := $0090;
        end {if};

        if not ParseChar (')') then exit;	{ failure }
        if not ParseComma then exit;		{ failure, comma expected }

{ second operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if x1 > 127 then exit;		{ failure, indirect not allowed }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }

        if kod = $0078 then	{ first operand: predecement/preincrement }
        begin
          if (x1 and $40) = 0 then
          begin
            x1 := x1 and $07;
            x2 := Reg6 (x2);
          end
          else
          begin
            x2 := Reg9(0,x2);
          end {if};
        end
        else	{ first operand: index register with an immediate offset }
        begin
          x2 := Reg9(0,x2) or x3;
        end {if};

        OutBuf[0] := byte(kod + sign + iregs[i].val2);
        OutBuf[1] := byte(x1);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case STM_ARG};


    STLM_ARG:		{ memory,array }
      begin
{ first operand: predecement/preincrement }
        if not ParseSign (sign) then exit;	{ failure, sign expected }
        if not ParIr (i) then exit;		{ failure }
        if not ParseChar (')') then exit;	{ failure }
        if not ParseComma then exit;		{ failure, comma expected }

{ second operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if x1 > 127 then exit;		{ failure, indirect not allowed }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }

        if (x1 and $40) = 0 then
        begin
          x1 := x1 and $07;
          x2 := Reg6 (x2);
        end
        else
        begin
          x2 := Reg9(0,x2);
        end {if};

        OutBuf[0] := byte(kod + sign + iregs[i].val2);
        OutBuf[1] := byte(x1);
        OutBuf[2] := byte(x2) or $08;
        OutIndex := 3;
      end {case STLM_ARG};


    ARY:		{ single array }
      begin
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }

        if (x1 and $40) = 0 then
        begin
          x2 := Reg9(x1,x2);
          x1 := x1 and $07;
        end
        else
        begin
          x2 := Reg9(0,x2);
        end {if};

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case ARY};


    RETURN_ARG:		{ cond_code }
      begin
        SkipBlanks;
        if ParseTable (i, cc, NCC) then kod := kod xor cc[i].val2;
        OutBuf[0] := byte(kod);
        OutIndex := 1;
      end {case RETURN_ARG};


    JUMP_ARG:		{ cond_code,immediate }
      begin
        SkipBlanks;
        if ParseTable (i, cc, NCC) then
        begin
          kod := kod xor cc[i].val2;
          if not ParseComma then exit;	{ failure, comma expected }
        end {if};
        if not EvalAndTest(x1,$FFFF) then exit;	{ failure, invalid value }
        OutBuf[0] := byte(kod);
        OutBuf[1] := Hi(x1);
        OutBuf[2] := byte(x1);
        OutIndex := 3;
      end {case JUMP_ARG};


    SJUMP_ARG:		{ cond_code,short_immediate }
      begin
        SkipBlanks;
        if ParseTable (i, cc, NCC) then
        begin
          kod := kod xor cc[i].val2;
          if not ParseComma then exit;	{ failure, comma expected }
        end {if};
        if not EvalAndTest(x1,$FFFF) then exit;	{ failure, invalid value }
        if ((loc xor x1) and $FF00) <> 0 then exit;	{ failure, page }
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1);
        OutIndex := 2;
      end {case SJUMP_ARG};


    LDSM_TYPE:		{ array,array }
      begin
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        SkipBlanks;
        if InBuf[InIndex] = ',' then
        begin
          Inc(InIndex);
          if not RegArgum (x3) then exit;	{ failure, register expected }
          if not AryArgum (x3, x4) then exit;	{ failure, invalid array }
          if ((x1 xor x3) and $40) = 0 then exit;	{ failure, reg. bank }
  { check the alignment, 'invalid array' error when fail }
          if (x1 and $0007) <> (x3 and $0007) then exit;	{ failure }
          if (x2 and $0007) <> (x4 and $0007) then exit;	{ failure }
          if (x1 and $40) = 0 then
          begin
            SwapWords (x1,x3);
            SwapWords (x2,x4);
            x1 := x1 and $3F;
            kod := kod xor $4000;
          end {if};
          x2 := Reg9(x3,x4);
        end
        else
        begin
{ instruction SWPM can accept a single array }
          if kod <> $4034 then exit;	{ failure, comma expected }
          kod := kod xor $8000;
          if (x1 and $40) = 0 then
          begin
            x2 := Reg9(x1,x2);
            x1 := x1 and $07;
            kod := kod xor $4000;
          end
          else
          begin
            x2 := Reg9(0,x2);
          end {if};
        end {if};

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case LDSM_TYPE};


    LDLM_TYPE:		{ array,array }
      begin
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        if not ParseComma then exit;		{ failure, comma expected }
        if not RegArgum (x3) then exit;		{ failure, register expected }
        if not AryArgum (x3, x4) then exit;	{ failure, invalid array }
        if ((x1 xor x3) and $40) = 0 then exit;	{ failure, reg. bank }
  { check the alignment, 'invalid array' error when fail }
        if (x1 and $0007) <> (x3 and $0007) then exit;	{ failure }
        if (x2 and $0007) <> (x4 and $0007) then exit;	{ failure }
        if (x1 and $40) = 0 then
        begin
          SwapWords (x1,x3);
          SwapWords (x2,x4);
          x1 := x1 and $3F;
          kod := kod xor $4005;
        end {if};
        x2 := Reg9(x3,x4);

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1) or Hi(kod);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case LDLM_TYPE};


    DB_ARG:		{ up to 3 data bytes }
      begin
        while (OutIndex < 3) and EvalAndTest (x1, $FF) do
        begin
          OutBuf[OutIndex] := x1;
          Inc (OutIndex);
          if not ParseComma then break;
        end {while};
      end {case DB_ARG};


{ else an internal error }

  end {case};

{ the rest of the InBuf is allowed to be padded with spaces only }
  SkipBlanks;
  if (InIndex > Length(InBuf)) 	{ end of line? }
	or ParseChar (';') then		{ comment? }
    InIndex := 0;			{ success }
{ otherwise failure, extra characters encountered }

end {Assemble};


end.
