unit Debug;

interface

uses
  SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, LCLType;

type
  TDebugForm = class(TForm)
    ListPanel: TPanel;
    ListPaintBox: TPaintBox;
    ListScrollBar: TScrollBar;
    ListEdit: TEdit;
    ListLabel: TLabel;

    RegPanel: TPanel;
    RegPaintBox: TPaintBox;
    RegScrollBar: TScrollBar;
    RegEdit: TEdit;
    RegLabel: TLabel;

    MainPanel: TPanel;
    MainPaintBox: TPaintBox;
    MainScrollBar: TScrollBar;
    MainEdit: TEdit;
    MainLabel: TLabel;

    BinPanel: TPanel;
    BinPaintBox: TPaintBox;
    BinScrollBar: TScrollBar;
    BinEdit: TEdit;
    BinLabel: TLabel;

    StepPanel: TPanel;
    StepButton: TButton;
    StepLabel: TLabel;

    TracePanel: TPanel;
    TraceEdit: TEdit;
    TraceButton: TButton;
    TraceLabel: TLabel;

    BpPanel: TPanel;
    BpEdit: TEdit;
    BpButton: TButton;
    BpLabel: TLabel;

{ DISASSEMBLY BOX EVENTS }
    procedure ListBoxScroll(Sender: TObject; ScrollCode: TScrollCode;
      var ScrollPos: Integer);
    procedure ListPanelClick(Sender: TObject);
    procedure ListPaintBoxMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ListEditKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure ListEditChange(Sender: TObject);
    procedure ListPaintBoxPaint(Sender: TObject);

{ REGISTER BOX EVENTS }
    procedure RegBoxScroll(Sender: TObject; ScrollCode: TScrollCode;
      var ScrollPos: Integer);
    procedure RegPanelClick(Sender: TObject);
    procedure RegPaintBoxMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure RegEditKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure RegEditChange(Sender: TObject);
    procedure RegPaintBoxPaint(Sender: TObject);

{ MAIN REGISTER BOX EVENTS }
    procedure MainBoxScroll(Sender: TObject; ScrollCode: TScrollCode;
      var ScrollPos: Integer);
    procedure MainPanelClick(Sender: TObject);
    procedure MainPaintBoxMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure MainEditKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure MainEditChange(Sender: TObject);
    procedure MainPaintBoxPaint(Sender: TObject);

{ BINARY EDITOR BOX EVENTS }
    procedure BinBoxScroll(Sender: TObject; ScrollCode: TScrollCode;
      var ScrollPos: Integer);
    procedure BinPanelClick(Sender: TObject);
    procedure BinPaintBoxMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure BinEditKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure BinEditChange(Sender: TObject);
    procedure BinPaintBoxPaint(Sender: TObject);

{ GENERAL FORM EVENTS }
    procedure DebugCreate(Sender: TObject);
    procedure DebugShow(Sender: TObject);
    procedure DebugHide(Sender: TObject);

{ MACHINE CODE EXECUTION CONTROL EVENTS }
    procedure StepPanelClick(Sender: TObject);
    procedure StepButtonClick(Sender: TObject);
    procedure TracePanelClick(Sender: TObject);
    procedure TraceEditChange(Sender: TObject);
    procedure TraceButtonClick(Sender: TObject);
    procedure BpPanelClick(Sender: TObject);
    procedure BpEditChange(Sender: TObject);
    procedure BpButtonClick(Sender: TObject);

  private
    { Private declarations }
  public
    { Public declarations }
  end;


var
  DebugForm: TDebugForm;


implementation

{$R *.dfm}

uses Def, Dis, Asem, Cpu;

const
  SELECTED = clBlue;

var
  ListAddr: word;
  RegAddr: word;
  MainAddr: word;
  BinAddr: word;

  EditState: (NoEditSt, ListAddrEditSt, ListInstrEditSt, RegEditSt,
	FlagEditSt, MainEditSt, BinAddrEditSt, BinDataEditSt,
	BinCharEditSt);
  EditAddr: word;	{address of the edited object - memory location, register}


{ set the font color of all TGrupBox controls to default }
procedure Unselect;
begin
  with DebugForm do
  begin
    ListPanel.Font.Color := clWindowText;
    RegPanel.Font.Color := clWindowText;
    MainPanel.Font.Color := clWindowText;
    BinPanel.Font.Color := clWindowText;
    StepPanel.Font.Color := clWindowText;
    TracePanel.Font.Color := clWindowText;
    BpPanel.Font.Color := clWindowText;
  end {with};
end {Unselect};


procedure BoxEdit (box: TPaintBox; ed: TEdit; Col, Row, W: integer);
var
  cx, cy, L, T: integer;
begin
  with box do
  begin
    cx := Canvas.TextWidth('0');
    cy := Canvas.TextHeight('0');
    L := Left;
    T := Top;
  end {with};
  with ed do
  begin
    Left := L + Col * cx;
    Top := T + Row * cy;
    Width := cx * W;
    Height := cy;
    MaxLength := W;
    Text := '';
  end {with};
end {BoxEdit};


{ value of a hex digit }
function GetDigit (c: char) : integer;
const
  digits: string[22] = '0123456789ABCDEFabcdef';
var
  i: integer;
begin
  i := 1;
  while (i<=22) and (c <> digits[i]) do Inc (i);
  if i>16 then GetDigit := i-7 else GetDigit := i-1;
end {GetDigit};


{ remove digits out of specified range from the edited string }
procedure CheckEdit (ed: TEdit; limit: integer);
var
  i: integer;
  s: string;
begin
  with ed do
  begin
    if Modified then
    begin
      s := Text;
      i := 1;
      while i <= Length(s) do
      begin
        if GetDigit(s[i]) >= limit then Delete (s, i, 1) else Inc (i);
      end {while};
      Text := s;
    end {if};
  end {with};
end {CheckEdit};


procedure CloseEdit;
begin
  EditState := NoEditSt;
  with DebugForm do
  begin
    with ListEdit do
    begin
      Text := '';
      Width := 0;
      Left := 0;
      Top := 0;
    end {with};
    with RegEdit do
    begin
      Text := '';
      Width := 0;
      Left := 0;
      Top := 0;
    end {with};
    with MainEdit do
    begin
      Text := '';
      Width := 0;
      Left := 0;
      Top := 0;
    end {with};
    with BinEdit do
    begin
      Text := '';
      Width := 0;
      Left := 0;
      Top := 0;
    end {with};
    ListPaintBox.Invalidate;
    RegPaintBox.Invalidate;
    MainPaintBox.Invalidate;
    BinPaintBox.Invalidate;
  end {with};
end {CloseEdit};


{ scrolling with the arrow keys,
  returns new value for Position or -1 when Position hasn't changed }
function ArrowKeys (Key: word; sb: TScrollBar) : integer;
begin
  with sb do
  begin
    Result := Position;
    case Key of
      VK_HOME:	Result := Min;
      VK_PRIOR:	Dec (Result, LargeChange);
      VK_UP:	Dec (Result, SmallChange);
      VK_DOWN:	Inc (Result, SmallChange);
      VK_NEXT:	Inc (Result, LargeChange);
      VK_END:	Result := Max;
    end {case};
    if Result < Min then Result := Min
    else if Result > Max then Result := Max;
    if Result = Position then Result := -1;
  end {with};
end;



{ DISASSEMBLY BOX EVENTS }

procedure TDebugForm.ListPanelClick(Sender: TObject);
begin
  ListEdit.SetFocus;
  Unselect;
  ListPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.ListBoxScroll(Sender: TObject;
  ScrollCode: TScrollCode; var ScrollPos: Integer);
begin
{ Lazarus: the value of ScrollPos may be outside the Min..Max range }
  with ListScrollBar do
    if ScrollPos > Max then ListAddr := word(Max)
    else ListAddr := word(ScrollPos);
  ListEdit.SetFocus;
  Unselect;
  ListPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.ListPaintBoxMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  Col, Row, cols, rows, i, w: integer;
  savepc: word;
  cx, cy: integer;	{ font size in pixels }
begin
  ListEdit.SetFocus;
  Unselect;
  ListPanel.Font.Color := SELECTED;
  CloseEdit;
  EditAddr := ListAddr;
  with ListPaintBox do
  begin
    cx := Canvas.TextWidth ('0');
    cy := Canvas.TextHeight ('0');
    cols := Width div cx;
    rows := Height div cy;
    Col := X div cx;
    Row := Y div cy;
  end {with};
  if Row >= rows then Exit;
  if (Col < 4) and (Row = 0) then
  begin
    EditState := ListAddrEditSt;
    Col := 0;
    w := 4;
    ListEdit.CharCase := ecUpperCase;
  end
  else if (Col >= 6) and (Col < cols) then
  begin
    EditState := ListInstrEditSt;
    savepc := pc;
    pc := ListAddr;
    w := 0;
    i := 0;
    while i < Row do
    begin
{ move the 'pc' to the next instruction, i.e. disassemble a single
  instruction without generating any output }
      Arguments (FetchOpcode);
      w := integer(cardinal(pc)) - integer(cardinal(ListAddr));
      if w < 0 then break;		{ when pc wraps around }
      Inc (i);
    end {while};
    EditAddr := pc;
    pc := savepc;
    if w < 0 then Exit;
    Col := 6;
    w := cols - 6;
    ListEdit.CharCase := ecNormal;
  end
  else
  begin
    Exit;
  end {if};
  BoxEdit (ListPaintBox, ListEdit, Col, Row, w);
end;


procedure TDebugForm.ListEditKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  i, x, valcode: integer;
begin
  i := ArrowKeys (Key, ListScrollBar);
  if (i >= 0) and (EditState = NoEditSt) then
  begin
    ListAddr := word(i);
    ListPaintBox.Invalidate;
  end

  else if Key = VK_RETURN then
  begin
    if EditState = ListAddrEditSt then
    begin
      Val ('$0'+Trim(ListEdit.Text), x, valcode);
      ListAddr := word(x);
      CloseEdit;
    end
    else if EditState = ListInstrEditSt then
    begin
{ assemble the instruction }
      InBuf := ListEdit.Text;
      Assemble;
{ when succeeded, copy the assembler output to the memory }
      if InIndex = 0 then
      begin
        i := 0;
        while i < OutIndex do
        begin
          if EditAddr < ROM0SIZE then
            rom0[EditAddr,i] := OutBuf[i]
          else
            SrcPtr(CODE_MEM, EditAddr+word(i))^ := OutBuf[i];
          Inc (i);
        end {while};
        dummysrc := $FF;
        CloseEdit;
      end
{ when failed, position the cursor just before the first offending character }
      else
      begin
        ListEdit.SelStart := InIndex-1;
      end {if};
    end {if};
  end

  else if key = VK_ESCAPE then CloseEdit;
end;


procedure TDebugForm.ListEditChange(Sender: TObject);
begin
  if EditState = ListAddrEditSt then CheckEdit (ListEdit, 16);
end;


procedure TDebugForm.ListPaintBoxPaint(Sender: TObject);
var
  i, rows, w: integer;
  savepc, index: word;
  cx, cy: integer;	{ font size in pixels }
begin
  savepc := pc;
  pc := ListAddr;
  with ListPaintBox do
  begin
    cx := Canvas.TextWidth ('0');
    cy := Canvas.TextHeight ('0');
    rows := Height div cy;
  end {with};
  with ListPaintBox.Canvas do
  begin
    w := 0;
    for i := 0 to rows-1 do
    begin
      TextOut (0, i*cy, IntToHex(pc, 4) + ':');
      index := FetchOpcode;
      if ((index and $FFE2) = $01E0) and ((opcode[2] and $08) <> 0) then
        TextOut (6*cx, i*cy, 'stlm')
      else
        TextOut (6*cx, i*cy, Mnemonic (index));
      TextOut (12*cx, i*cy, Arguments (index));
      w := integer(cardinal(pc)) - integer(cardinal(ListAddr));
      if w < 0 then break;		{ when pc wraps around }
    end {for};
  end {with};
{ set the scroll bar }
  with ListScrollBar do
  begin
    SetParams (integer(cardinal(ListAddr)), 0, 65535);
    if w > 0 then LargeChange := w;
  end {with};
  pc := savepc;
end;



{ REGISTER BOX EVENTS }

type
  reg_properties = record
    name: string[3];
    ptr: pointer;
    size: integer;
    mask: integer;
  end;

  flag_properties = record
    name: array[0..1] of string[3];
    column: integer;
    row: integer;
    mask: byte;
  end;

const
  REGROWS = 15;
  regset: array[0..REGROWS-1] of reg_properties = (
    ( name: 'PC:';	ptr: @pc;	size: 4;	mask: $FFFF; ),
    ( name: 'LA:';	ptr: @la;	size: 4;	mask: $FFFF; ),
    ( name: 'KC:';	ptr: @kc;	size: 2;	mask: $00FF; ),
    ( name: 'KI:';	ptr: @ki;	size: 2;	mask: $00FF; ),
    ( name: 'IE:';	ptr: @ie;	size: 2;	mask: $00FF; ),
    ( name: 'DS:';	ptr: @ds;	size: 2;	mask: $00FF; ),
    ( name: 'AS:';	ptr: @asreg;	size: 2;	mask: $007F; ),
    ( name: 'PD:';	ptr: @pd;	size: 2;	mask: $00FF; ),
    ( name: 'PE:';	ptr: @pe;	size: 2;	mask: $007F; ),
    ( name: 'S4:';	ptr: @s4;	size: 2;	mask: $007F; ),
    ( name: 'S5:';	ptr: @s5;	size: 2;	mask: $007F; ),
    ( name: 'S6:';	ptr: @s6;	size: 2;	mask: $007F; ),
    ( name: 'S7:';	ptr: @s7;	size: 2;	mask: $007F; ),
    ( name: 'S8:';	ptr: @s8;	size: 2;	mask: $007F; ),
    ( name: 'TM:';	ptr: @tm;	size: 2;	mask: $007F; )
  );

  flagset: array[0..5] of flag_properties = (
    ( name: ('NC', 'C ');	column: 3;	row: 0;	mask: C_bit;	),
    ( name: ('NV', 'V ');	column: 5;	row: 0;	mask: V_bit;	),
    ( name: ('NUZ', 'UZ ');	column: 7;	row: 0;	mask: UZ_bit;	),
    ( name: ('NH', 'H ');	column: 3;	row: 1;	mask: H_bit;	),
    ( name: ('NZ ', 'Z ');	column: 5;	row: 1;	mask: Z_bit;	),
    ( name: ('NLZ', 'LZ ');	column: 7;	row: 1;	mask: LZ_bit;	)
  );


procedure TDebugForm.RegPanelClick(Sender: TObject);
begin
  RegEdit.SetFocus;
  Unselect;
  RegPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.RegBoxScroll(Sender: TObject; ScrollCode: TScrollCode;
  var ScrollPos: Integer);
begin
{ Lazarus: the value of ScrollPos may be outside the Min..Max range }
  with RegScrollBar do
    if ScrollPos > Max then RegAddr := word(Max)
    else RegAddr := word(ScrollPos);
  RegEdit.SetFocus;
  Unselect;
  RegPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.RegPaintBoxMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  Col, Row, rows, w: integer;
  cx, cy: integer;	{ font size in pixels }
begin
  RegEdit.SetFocus;
  Unselect;
  RegPanel.Font.Color := SELECTED;
  CloseEdit;
  with RegPaintBox do
  begin
    cx := Canvas.TextWidth ('0');
    cy := Canvas.TextHeight ('0');
    rows := Height div cy;
    Col := X div cx;
    Row := Y div cy;
  end {with};
  if rows > REGROWS+2 then rows := REGROWS+2;
  if (Row > 1) and (Row < rows) and (Col >= 4) then
{ registers other than Flags }
  begin
    EditAddr := word(Row) + RegAddr - 2;
    w := regset[EditAddr].size;
    if Col >= 4+w then Exit;
    EditState := RegEditSt;
    BoxEdit (RegPaintBox, RegEdit, 4, Row, w);
  end {if};
  if (Row >= 0) and (Row < 2) and (Col >= 3) and (Col < 10) then
{ Flags register }
  begin
    EditState := FlagEditSt;
    EditAddr := 0;
    while (EditAddr < 2) and (Col >= flagset[EditAddr+1].column) do
      Inc (EditAddr);
    Inc (EditAddr, word(Row*3));
    flag := flag xor flagset[EditAddr].mask;
    RegPaintBox.Invalidate;
  end {if};
end;


procedure TDebugForm.RegEditKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  i, x, valcode: integer;
begin
  i := ArrowKeys (Key, RegScrollBar);
  if (i >= 0) and (EditState = NoEditSt) then
  begin
    RegAddr := word(i);
    RegPaintBox.Invalidate;
  end

  else if Key = VK_RETURN then
  begin
    if EditState = RegEditSt then
    begin
      Val ('$0'+Trim(RegEdit.Text), x, valcode);
      with regset[EditAddr] do
      begin
        x := x and integer(mask);
        if size > 2 then ptrw(ptr)^ := word(x) else ptrb(ptr)^ := byte(x);
      end {with};
      CloseEdit;
    end {if};
  end

  else if Key = VK_ESCAPE then CloseEdit;
end;


procedure TDebugForm.RegEditChange(Sender: TObject);
begin
  CheckEdit (RegEdit, 16);
end;


procedure TDebugForm.RegPaintBoxPaint(Sender: TObject);
var
  i, rows: integer;
  x: word;
  cx, cy: integer;	{ font size in pixels }
begin
  with RegPaintBox do
  begin
    Canvas.Brush.Style := bsSolid;
    Canvas.Brush.Color := Color;
    cx := Canvas.TextWidth ('0');
    cy := Canvas.TextHeight ('0');
    rows := Height div cy;
  end {with};
  if rows > REGROWS+2 then rows := REGROWS+2;
  with RegPaintBox.Canvas do
  begin
{ scrollable registers }
    for i := 0 to rows-3 do
    begin
      with regset[i+integer(cardinal(RegAddr))] do
      begin
        TextOut (0, (i+2)*cy, name);
        if size > 2 then x := ptrw(ptr)^ else x := word(ptrb(ptr)^);
        TextOut (4*cx, (i+2)*cy, IntToHex(x and mask, size));
      end {with};
    end {for};
{ unscrollable Flags register }
    TextOut (0, 0, 'F:');
    for i := 0 to 5 do
    begin
      if Odd (i) then Brush.Color := clLtGray else Brush.Color := clWhite;
      with flagset[i] do
      begin
        if (flag and mask) = 0 then x := 0 else x := 1;
{ Lazarus: TextOut ignores the Canvas.Brush specification, the background is always transparent }
        FillRect (column*cx, row*cy, column*cx + TextWidth(name[x]), row*cy + TextHeight(name[x]));
        TextOut (column*cx, row*cy, name[x]);
      end {with};
    end {for};
  end {with};
{ set the scroll bar }
  with RegScrollBar do
  begin
    SetParams (integer(cardinal(RegAddr)), 0, REGROWS+2-rows);
    LargeChange := rows-1;
  end {with};
end;



{ MAIN REGISTER BOX EVENTS }

const
  MAINROWS = 8;
  mrnames: array[0..MAINROWS-1] of string[11] = (
	'R000..R015:', 'R016..R031:', 'R032..R047:', 'R048..R063:',
	'R064..R079:', 'R080..R095:', 'R096..R111:', 'R112..R127:' );


procedure TDebugForm.MainPanelClick(Sender: TObject);
begin
  MainEdit.SetFocus;
  Unselect;
  MainPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.MainBoxScroll(Sender: TObject;
  ScrollCode: TScrollCode; var ScrollPos: Integer);
begin
{ Lazarus: the value of ScrollPos may be outside the Min..Max range }
  with MainScrollBar do
    if ScrollPos > Max then MainAddr := word(Max)
    else MainAddr := word(ScrollPos);
  MainAddr := MainAddr*16;
  MainEdit.SetFocus;
  Unselect;
  MainPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.MainPaintBoxMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  Col, Row, rows: integer;
  cx, cy: integer;	{ font size in pixels }
begin
  MainEdit.SetFocus;
  Unselect;
  MainPanel.Font.Color := SELECTED;
  CloseEdit;
  with MainPaintBox do
  begin
    cx := Canvas.TextWidth ('0');
    cy := Canvas.TextHeight ('0');
    rows := Height div cy;
    Col := X div cx;
    Row := Y div cy;
  end {with};
  Dec (Col, 12);
  if (Row >= 0) and (Row < rows) and (Col >= 0) and (Col < 51) and
	((Col mod 13) < 11) and (((Col mod 13) mod 3) < 2) then
  begin
    EditState := MainEditSt;
    Col := (Col - Col div 13) div 3;
    EditAddr := MainAddr + word(16*Row + Col);
    Col := 3*Col + Col div 4 + 12;
    BoxEdit (MainPaintBox, MainEdit, Col, Row, 2);
  end {if};
end;


procedure TDebugForm.MainEditKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  i, x, valcode: integer;
begin
  i := ArrowKeys (Key, MainScrollBar);
  if (i >= 0) and (EditState = NoEditSt) then
  begin
    MainAddr := word(16*i);
    MainPaintBox.Invalidate;
  end

  else if Key = VK_RETURN then
  begin
    if EditState = MainEditSt then
    begin
      Val ('$0'+Trim(MainEdit.Text), x, valcode);
      mr[EditAddr] := byte(x);
      CloseEdit;
    end {if};
  end

  else if Key = VK_ESCAPE then CloseEdit;
end;


procedure TDebugForm.MainEditChange(Sender: TObject);
begin
  CheckEdit (MainEdit, 16);
end;


procedure TDebugForm.MainPaintBoxPaint(Sender: TObject);
var
  i, j, rows, Col: integer;
  cx, cy: integer;	{ font size in pixels }
begin
  with MainPaintBox do
  begin
    Canvas.Brush.Style := bsSolid;
    Canvas.Brush.Color := Color;
    cx := Canvas.TextWidth ('0');
    cy := Canvas.TextHeight ('0');
    rows := Height div cy;
  end {with};
  if rows > MAINROWS then rows := MAINROWS;
  with MainPaintBox.Canvas do
  begin
    for j := 0 to rows-1 do
    begin
      TextOut (0, j*cy, mrnames[integer(cardinal(MainAddr)) div 16 + j]);
      for i := 0 to 15 do
      begin
        Col := 3*i + i div 4 + 12;
        TextOut (Col*cx, j*cy, IntToHex(mr[integer(cardinal(MainAddr))+16*j+i], 2));
      end {for i};
    end {for j};
  end {with};
{ set the scroll bar }
  with MainScrollBar do
  begin
    SetParams (integer(cardinal(MainAddr)) div 16, 0, MAINROWS-rows);
    LargeChange := rows;
  end {with};
end;



{ BINARY EDITOR BOX EVENTS }

procedure TDebugForm.BinPanelClick(Sender: TObject);
begin
  BinEdit.SetFocus;
  Unselect;
  BinPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.BinBoxScroll(Sender: TObject; ScrollCode: TScrollCode;
  var ScrollPos: Integer);
begin
{ Lazarus: the value of ScrollPos may be outside the Min..Max range }
  with BinScrollBar do
    if ScrollPos > Max then BinAddr := word(Max)
    else BinAddr := word(ScrollPos);
  BinAddr := BinAddr*16 + RAMSTART;
  BinEdit.SetFocus;
  Unselect;
  BinPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.BinPaintBoxMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  Col, Row, rows, w: integer;
  cx, cy: integer;	{ font size in pixels }
begin
  BinEdit.SetFocus;
  Unselect;
  BinPanel.Font.Color := SELECTED;
  CloseEdit;
  with BinPaintBox do
  begin
    cx := Canvas.TextWidth ('0');
    cy := Canvas.TextHeight ('0');
    rows := Height div cy;
    Col := X div cx;
    Row := Y div cy;
  end {with};
  if Row >= rows then Exit;
  if (Row = 0) and (Col < 4) then
  begin				{select BinAddr edition}
    EditState := BinAddrEditSt;
    EditAddr := 0;
    Col := 0;
    w := 4;
    BinEdit.CharCase := ecUpperCase;
  end
  else if (Col >= 6) and (Col < 54) and ((Col mod 3) < 2) then
  begin				{select byte edition in the BinBox}
    Col := (Col-6) div 3;
    EditAddr := BinAddr + word(16*Row + Col);
    if EditAddr >= RAMEND then Exit;
    Col := Col*3 + 6;
    EditState := BinDataEditSt;
    w := 2;
    BinEdit.CharCase := ecUpperCase;
  end
  else if (Col >= 54) and (Col < 70) then
  begin				{select character edition in the BinBox}
    EditAddr := BinAddr + word(16*Row + Col) - 54;
    if EditAddr >= RAMEND then Exit;
    EditState := BinCharEditSt;
    w := 1;
    BinEdit.CharCase := ecNormal;
  end
  else
  begin
    Exit;
  end {if};
  BoxEdit (BinPaintBox, BinEdit, Col, Row, w);
end;


procedure TDebugForm.BinEditKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  rows, i, x, valcode: integer;
begin
  with BinPaintBox do
  begin
    rows := Height div Canvas.TextHeight ('0');
  end {with};

  i := ArrowKeys (Key, BinScrollBar);
  if (i >= 0) and (EditState = NoEditSt) then
  begin
    BinAddr := word(16*i) + RAMSTART;
    BinPaintBox.Invalidate;
  end

  else if Key = VK_RETURN then
  begin
    if EditState = BinAddrEditSt then
    begin
      Val('$0'+Trim(BinEdit.Text), x, valcode);
      x := x and $FFF0;
      if (x >= RAMSTART) and ((x + 16*rows) < RAMEND) then BinAddr := word(x);
      CloseEdit;
    end
    else if EditState = BinDataEditSt then
    begin
      Val ('$0'+Trim(BinEdit.Text), x, valcode);
      DstPtr(DATA_MEM, EditAddr)^ := byte(x);
      CloseEdit;
    end
    else if EditState = BinCharEditSt then
    begin
      DstPtr(DATA_MEM, EditAddr)^ := byte(Ord(BinEdit.Text[1]));
      CloseEdit;
    end {if};
  end

  else if Key = VK_ESCAPE then CloseEdit;
end;


procedure TDebugForm.BinEditChange(Sender: TObject);
begin
  if (EditState = BinAddrEditSt) or (EditState = BinDataEditSt) then
    CheckEdit (BinEdit, 16);
end;


procedure TDebugForm.BinPaintBoxPaint(Sender: TObject);
var
  i, j, rows: integer;
  a: integer;
  x: byte;
  cx, cy: integer;	{ font size in pixels }
begin
  a := integer(cardinal(BinAddr));
  with BinPaintBox do
  begin
    cx := Canvas.TextWidth ('0');
    cy := Canvas.TextHeight ('0');
    rows := Height div cy;
  end {with};
  with BinPaintBox.Canvas do
  begin
    for i := 0 to rows-1 do
    begin
      if a >= RAMEND then break;
{ address }
      TextOut (0, i*cy, IntToHex(a, 4) + ':');
{ bytes }
      for j := 0 to 15 do
      begin
        TextOut ((6+3*j)*cx, i*cy, IntToHex(SrcPtr(DATA_MEM, a+j)^, 2));
      end {for};
{ characters }
      for j := 0 to 15 do
      begin
        x := SrcPtr(DATA_MEM, a+j)^;
        if (x < $20) or (x > $7E) then x := byte(Ord('.'));
        TextOut ((54+j)*cx, i*cy, Chr(x));
      end {for};
      Inc (a, 16);
    end {for};
  end {with};
{ set the scroll bar }
  with BinScrollBar do
  begin
    SetParams (integer(cardinal(BinAddr-RAMSTART)) div 16, 0, RAMSIZE div 16 - rows);
    LargeChange := rows;
  end {with};
end;



{ MACHINE CODE EXECUTION CONTROL }

procedure TDebugForm.StepPanelClick(Sender: TObject);
begin
  StepPanel.SetFocus;
  Unselect;
  StepPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.StepButtonClick(Sender: TObject);
begin
  EditState := NoEditSt;
  CpuRun;
  ListAddr := pc;
  StepPanel.SetFocus;
  Unselect;
  StepPanel.Font.Color := SELECTED;
  CloseEdit
end;


procedure TDebugForm.TracePanelClick(Sender: TObject);
begin
  TraceEdit.SetFocus;
  Unselect;
  TracePanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.TraceButtonClick(Sender: TObject);
var
  i, valcode: integer;
begin
  with TraceEdit do
  begin
    Val ('0'+Trim(TraceEdit.Text), i, valcode);
    SetFocus;
  end {with};
  Unselect;
  TracePanel.Font.Color := SELECTED;
  CloseEdit;
  if i > 0 then
  begin
    BreakPoint := -1;
    CpuSteps := i;
    Hide;
  end {if};
end;


{ remove digits out of specified range from the edited string }
procedure TDebugForm.TraceEditChange(Sender: TObject);
begin
  CheckEdit (TraceEdit, 10);
end;


procedure TDebugForm.BpPanelClick(Sender: TObject);
begin
  BpEdit.SetFocus;
  Unselect;
  BpPanel.Font.Color := SELECTED;
  CloseEdit;
end;


procedure TDebugForm.BpButtonClick(Sender: TObject);
var
  i, valcode: integer;
begin
  with BpEdit do
  begin
    Val ('$0'+Trim(BpEdit.Text), i, valcode);
    SetFocus;
  end {with};
  Unselect;
  BpPanel.Font.Color := SELECTED;
  CloseEdit;
  BreakPoint := i;
  CpuSteps := -1;
  Hide;
end;


procedure TDebugForm.BpEditChange(Sender: TObject);
begin
  CheckEdit (BpEdit, 16);
end;



{ GENERAL FORM EVENTS }

procedure TDebugForm.DebugCreate(Sender: TObject);
begin
  CloseEdit;
  RegAddr := 0;
  MainAddr := 0;
  BinAddr := RAMSTART;
  ListAddr := $0000;
end;


procedure TDebugForm.DebugShow(Sender: TObject);
begin
  CpuStop := True;
  CpuSteps := -1;
  BreakPoint := -1;
  ListAddr := pc;
  StepPanel.SetFocus;
end;


procedure TDebugForm.DebugHide(Sender: TObject);
begin
  Hide;
  EditState := NoEditSt;
  CpuDelay := 30;
  CpuStop := False;
end;


end.

