{ common stuff, CPU registers, memory }

unit Def;

interface

  type
    ptrb = ^byte;		{ unsigned 8-bit }
    ptrw = ^word;		{ unsigned 16-bit }

  const

{ status bits of the Flag Register F }
    C_bit	= $80;
    H_bit	= $40;
    V_bit	= $20;
    Z_bit	= $10;
    UZ_bit	= $08;
    LZ_bit	= $04;

    INTVECTORS	= 4;

{ bits of the IE register and the 'iserv' variable }

    INT0_bit	= $01;
    INT1_bit	= $02;
    INT2_bit	= $04;
    INTT_bit	= $08;

    intmask: array[0..INTVECTORS-1] of byte =
	( INT0_bit, INT1_bit, INT2_bit, INTT_bit );

{ patterns of the DS register }
    CODE_MEM	= $0;
    DATA_MEM	= $3;

    ROM0SIZE	= $1800;
    ROM1SIZE	= $8000;
    RAMSTART	= $0000;
    RAMSIZE	= $2000;
    RAMEND	= RAMSTART+RAMSIZE;

  var

    rom0: array[0..ROM0SIZE-1,0..2] of byte;
    rom1: array[0..ROM1SIZE-1] of byte;
    ram: array[0..RAMSIZE-1] of byte;

{ 8-bit registers }
    mr: array[0..127] of byte;	{ main (general purpose) register file }
    ds: byte;
    kc: byte;
    ki: byte;
    ie: byte;
    pd: byte;
    flag: byte;

{ 7-bit registers }
    asreg: byte;	{ 'as' is a reserved word in Pascal }
    pe: byte;
    s4: byte;
    s5: byte;
    s6: byte;
    s7: byte;
    s8: byte;
    tm: byte;

{ 16-bit register }
    pc: word;
    la: word;		{ display memory address }

    ireq: byte;			{ interrupt request flags }
    iserv: byte;		{ interrupt service flags }
    dummysrc: byte = $FF;	{ free adress space }
    dummydst: byte;		{ free address space }
    opcode: array[0..2] of byte;
    opindex: integer;		{ index to the opcode table }
    cycles: integer;		{ counter of the clock pulses }
    acycles: integer;		{ clock pulse counter accumulator }
    procptr: pointer;		{ pointer to a procedure that should be
				  executed before a machine code instruction,
				  usually to complete an I/O register write
				  cycle }
    OscFreq: integer;		{ CPU clock frequency in kHz }
    CpuStop: boolean;		{ True stops the CPU, used in the debug mode }
    CpuDelay: integer;		{ delay after hiding the Debug Window,
				  prevents the program from crashing when the
				  Debug Window was made visible too early }
    CpuSleep: boolean;		{ True if the CPU in the power-off state, can
				  be waken up by an INT2 interrupt or AC key }
    CpuWait: boolean;		{ True if the CPU in the wait_for_an_interrupt
				  state }
    CpuSteps: integer;		{ ignored when < 0 }
    BreakPoint: integer;	{ ignored when < 0 }
    SelfTest: integer;		{ for how many LCD refresh cycles (forever if
				  negative) the self-test jumper should be
				  connected after the reset }
    PowerSwitch: boolean;	{ True when the power switch is on }


  function SrcPtr (segment: byte; address: word) : ptrb;
  function DstPtr (segment: byte; address: word) : ptrb;
  function FetchOpcode: word;
  function FetchByte: byte;
  procedure IntReq (i: integer);


implementation


{ returns the pointer to a read-type resource at specified address }
function SrcPtr (segment: byte; address: word) : ptrb;
begin
  if segment = CODE_MEM then
    SrcPtr := @rom1[address and (ROM1SIZE-1)]
  else if segment = DATA_MEM then
    SrcPtr := @ram[address and (RAMSIZE-1)]
  else
    SrcPtr := @dummysrc;
end {SrcPtr};


{ returns the pointer to a write-type resource at specified address }
function DstPtr (segment: byte; address: word) : ptrb;
begin
  if segment = DATA_MEM then
    DstPtr := @ram[address and (RAMSIZE-1)]
  else
    DstPtr := @dummydst;
end {DstPtr};


{ returns an index to an instruction table }
function FetchOpcode: word;
begin
  if pc < ROM0SIZE then
  begin
    opcode[0] := rom0[pc,0];
    opcode[1] := rom0[pc,1];
    opcode[2] := rom0[pc,2];
  end
  else
  begin
    opcode[0] := rom1[pc and (ROM1SIZE-1)];
    opcode[1] := rom1[(pc+1) and (ROM1SIZE-1)];
    opcode[2] := rom1[(pc+2) and (ROM1SIZE-1)];
  end {if};
  opindex := 0;
  if opcode[0] < $A0 then
    FetchOpcode := (word(opcode[0]) shl 2) + (word(opcode[1]) shr 6)
  else
    FetchOpcode := word(opcode[0]) + $01E0;
end {FetchOpcode};


{ returns bytes of the instruction }
function FetchByte: byte;
begin
  FetchByte := opcode[opindex];
  if pc < ROM0SIZE then
  begin	{ accessing the internal ROM doesn't consume any clock cycles }
    if opindex = 0 then Inc (pc);
  end
  else
  begin
    Inc (pc);
    Inc (cycles, 2);
  end {if};
  Inc (opindex);
end {FetchByte};


{ interrupt request }
procedure IntReq (i: integer);
begin
  if ((ie and $80) <> 0) then
  begin
    CpuWait := False;
    ireq := ireq or intmask[i];
  end {if};
end {IntReq};


end.
