{ NM-326 instruction execution }

unit Exec;


interface

  var
    regstep: word;	{ determines the order of processed registers
			  in a register array,
			  +1 if the register index is incremented (default),
			  -1 if the register index is decremented } 


  function Wreg (x, y: byte) : word;
  procedure PushPC;
  function PreIncw (x, y: byte) : word;
  function PreDecw (x, y: byte) : word;
  function PostIncw (x, y: byte) : word;
  function PostDecw (x, y: byte) : word;
  function PlusOffset8 (x, y: byte) : word;
  function MinusOffset8 (x, y: byte) : word;
  function PlusOffset5 (x, y: byte) : word;
  function MinusOffset5 (x, y: byte) : word;
  procedure UnReg (op2: pointer);
  procedure TUnReg (op2: pointer);
  procedure XunAry (op2: pointer);
  procedure YunAry (op2: pointer);
  procedure TXunAry (op2: pointer);
  procedure TYunAry (op2: pointer);
  procedure Xreg (op2: pointer);
  procedure Yreg (op2: pointer);
  procedure TXreg (op2: pointer);
  procedure TYreg (op2: pointer);
  procedure RegIm8 (op2: pointer);
  procedure TRegIm8 (op2: pointer);
  procedure Ld2RegIm8 (op2: pointer);
  procedure Xary (op2: pointer);
  procedure Yary (op2: pointer);
  procedure TXary (op2: pointer);
  procedure TYary (op2: pointer);
  procedure AryIm5 (op2: pointer);
  procedure TAryIm5 (op2: pointer);
  procedure MemoReg (op2: pointer);
  procedure TMemoReg (op2: pointer);
  procedure MemoIm8 (op2: pointer);
  procedure Ldw (op2: pointer);
  procedure AdwSbw (op2: pointer);
  procedure Cond (op2: pointer);
  procedure NotCond (op2: pointer);
  procedure KeyCond (op2: pointer);
  procedure NotKeyCond (op2: pointer);
  procedure Jump (x: boolean);
  procedure Call (x: boolean);
  procedure SJump (x: boolean);
  procedure SCall (x: boolean);
  procedure Trap (op2: pointer);
  procedure PlaIjmp (op2: pointer);
  procedure Rtn (x: boolean);
  procedure Rti (op2: pointer);
  procedure Nop (op2: pointer);
  procedure BlockCopy (op2: pointer);
  procedure BlockSearch (op2: pointer);
  procedure StMemoReg (op2: pointer);
  procedure XmemoAry (op2: pointer);
  procedure YmemoAry (op2: pointer);
  procedure StmOffsAry (op2: pointer);
  procedure LdRegMemo (op2: pointer);
  procedure XaryMemo (op2: pointer);
  procedure YaryMemo (op2: pointer);
  procedure LdmAryOffs (op2: pointer);
  procedure PstIm7 (op2: pointer);
  procedure PstIm8 (op2: pointer);
  procedure PstReg (op2: pointer);
  procedure Gst (op2: pointer);
  procedure Off (op2: pointer);
  procedure Wai (op2: pointer);

  procedure OpAd (var x, y: byte);
  procedure OpSb (var x, y: byte);
  procedure OpAdb (var x, y: byte);
  procedure OpSbb (var x, y: byte);
  procedure OpAn (var x, y: byte);
  procedure OpBit (var x, y: byte);
  procedure OpXr (var x, y: byte);
  procedure OpNa (var x, y: byte);
  procedure OpOr (var x, y: byte);
  procedure OpLd (var x, y: byte);
  procedure OpLdh (var x, y: byte);
  procedure OpLdl (var x, y: byte);
  procedure OpLds (var x, y: byte);
  procedure OpLdhs (var x, y: byte);
  procedure OpLdls (var x, y: byte);

  function OpRod (var x: byte; y: byte) : byte;
  function OpRou (var x: byte; y: byte) : byte;
  function OpMtb (var x: byte; y: byte) : byte;
  function OpInv (var x: byte; y: byte) : byte;
  function OpCmp (var x: byte; y: byte) : byte;
  function OpCmpb (var x: byte; y: byte) : byte;
  function OpSwp1 (var x: byte; y: byte) : byte;
  function OpDiu (var x: byte; y: byte) : byte;
  function OpDid (var x: byte; y: byte) : byte;
  function OpByu (var x: byte; y: byte) : byte;
  function OpByd (var x: byte; y: byte) : byte;

  procedure OpXc (var x, y: byte);
  procedure OpXch (var x, y: byte);
  procedure OpXcl (var x, y: byte);
  procedure OpXcs (var x, y: byte);
  procedure OpXcls (var x, y: byte);
  procedure OpXchs (var x, y: byte);
  procedure OpSwp (var x, y: byte);
  procedure OpSwp2 (var x, y: byte);

  procedure OpIe (x: byte);
  procedure OpDs (x: byte);
  procedure OpKc (x: byte);
  procedure OpPd (x: byte);
  procedure OpFl (x: byte);
  procedure OpAs (x: byte);


implementation

  uses Def, Keyboard;


  type

    Proc3 = procedure (x, y: byte);
    Proc4 = procedure (x: byte);
    Proc5 = procedure (var x, y: byte);
    Proc6 = procedure (x: boolean);
    Func4 = function (x, y: byte) : word;
    Func5 = function (var x: byte; y: byte) : byte;


  const

    IX = 0;
    IY = 1;
    IZ = 2;
    V3 = 3;
    V2 = 4;
    V1 = 5;
    V0 = 6;
    SP = 7;

    cc: array[0..7] of byte = (
	$00, LZ_bit, UZ_bit, Z_bit, V_bit, H_bit, C_bit, $00 );

    NONE_REG = 255;	{ index of a non-existing general purpose register }


{ expects the second opcode byte (specification of a general purpose register
  in the bank 1),
  returns the address (index) of the general purpose register }
function RegA (y: byte) : byte;
begin
  RegA := (y and $3F) or $40;
end {RegA};


function RegB (y, z: byte) : byte;
begin
  RegB := ((z shl 3) and $38) or (y and $07);
end {RegB};


function RegC (y, z: byte) : byte;
begin
  RegC := (y and $38) or (z shr 5) or $40;
end {RegC};


{ expects the third opcode byte (specification of a general purpose register
  in the bank 0,
  returns the address (index) of the general purpose register }
function RegD (z: byte) : byte;
begin
  RegD := (z shr 5) or ((z shl 3) and $38);
end {RegD};


function RegE (y, z: byte) : byte;
begin
  RegE := (y and $78) or (z shr 5);
end {RegE};


{ register specified indirectly by the 'asreg' }
function RegI (z: byte) : byte;
begin
  if (asreg and $01) = 0 then RegI := (asreg shr 1) and $3F
  else  RegI := ((asreg shr 1) and $07) or ((z shl 3) and $38);
end {RegI};


function IndB (y, z: byte) : byte;
begin
  if (z and $10) = 0 then IndB := RegB (y, z) else IndB := RegI (z);
end {IndB};


function IndD (z: byte) : byte;
begin
  if (z and $10) = 0 then IndD := RegD (z) else IndD := RegI (z);
end {IndD};


{ optional last register of an indirectly specified array }
function AsLimit (z: byte) : byte;
begin
  if (z and $10) = 0 then
    AsLimit := NONE_REG	{ directly specified array, no limit }
  else if (asreg and $01) = 0 then
    AsLimit := ((asreg shr 1) and $3F) or $07
  else
    AsLimit := ((asreg shr 4) and $07) or ((z shl 3) and $38);
end {AsLimit};


{ index register for the first opcode byte x }
function Ireg (x: byte) : byte;
begin
  Result := x and $03;
  if Result > IZ then Result := SP;
end {Ireg};


{ data segment for the index register x }
function Iseg (x: byte) : byte;
begin
  case x of
    IX, IZ: Iseg := ds and $03;
    IY: Iseg := (ds shr 2) and $03;
    SP: Iseg := (ds shr 4) and $03;
    else Iseg := 0;		{ should never happen }
  end;
end {Iseg};


{ next processed register in an array }
procedure NextReg (var x: byte);
begin
  x := (x and $F8) or ((x + byte (regstep)) and $07);
end {NextReg};


{ returns the contents of a 16-bit register 'x',
  'y' is a dummy variable }
function Wreg (x, y: byte) : word;
begin
  Wreg := (word(mr[x or $78]) shl 8) or word(mr[x or $38]);
end {Wreg};


procedure PushPC;
begin
  DstPtr(DATA_MEM, PreDecw(SP,1))^ := Lo (pc);
  DstPtr(DATA_MEM, PreDecw(SP,1))^ := Hi (pc);
end {PushPC};


{ return the contents of the 16-bit register 'x',
  then add 8-bit value 'y' to this 16-bit register }
function PostIncw (x, y: byte) : word;
var
  z: word;
begin
  Result := Wreg(x,0);
  z := Result + word(y);
  mr[x or $38] := Lo (z);
  mr[x or $78] := Hi (z);
end {PostIncw};


{ return the contents of the 16-bit register 'x',
  then subtract 8-bit value 'y' from this 16-bit register }
function PostDecw (x, y: byte) : word;
var
  z: word;
begin
  regstep := word (-1);
  Result := Wreg(x,0);
  z := Result - word(y);
  mr[x or $38] := Lo (z);
  mr[x or $78] := Hi (z);
end {PostDecw};


{ add 8-bit value 'y' to the 16-bit register 'x',
  then return the contents of this 16-bit register }
function PreIncw (x, y: byte) : word;
begin
  Result := Wreg(x,0) + word(y);
  mr[x or $38] := Lo (Result);
  mr[x or $78] := Hi (Result);
end {PreIncw};


{ subtract 8-bit value 'y' from the 16-bit register 'x',
  then return the contents of this 16-bit register }
function PreDecw (x, y: byte) : word;
begin
  regstep := word (-1);
  Result := Wreg(x,0) - word(y);
  mr[x or $38] := Lo (Result);
  mr[x or $78] := Hi (Result);
end {PreDecw};


{ effective address <- index register + immediate 8-bit offset }
function PlusOffset8 (x {index register}, y {dummy}: byte) : word;
begin
  PlusOffset8 := Wreg (x,0) + word (FetchByte);
end {PlusOffset8};


{ effective address <- index register - immediate 8-bit offset }
function MinusOffset8 (x {index register}, y {dummy}: byte) : word;
begin
  regstep := word (-1);
  MinusOffset8 := Wreg (x,0) - word (FetchByte);
end {MinusOffset8};


{ effective address <- index register + immediate 5-bit offset }
function PlusOffset5 (x {index register}, y: byte) : word;
begin
  PlusOffset5 := Wreg (x,0) + word (y and $1F);
end {PlusOffset5};


{ effective address <- index register - immediate 5-bit offset }
function MinusOffset5 (x {index register}, y: byte) : word;
begin
  regstep := word (-1);
  MinusOffset5 := Wreg (x,0) - word (y and $1F);
end {MinusOffset5};


{ unary operation on a register }
procedure UnReg (op2: pointer);
var
  x, y: byte;
begin
  x := FetchByte;
  y := FetchByte and $7F;
  if x = $60 {ROD/ROU} then flag := (flag and C_bit) or Z_bit else
	flag := Z_bit;
  if x = $65 {BNU,BND} then x := mr[y] else x := 0;
  Func5 (op2) (mr[y], x);
  Inc (cycles, 4);
end {UnReg};


{ unary operation on a register, but the result is not stored }
procedure TUnReg (op2: pointer);
var
  y: byte;
begin
  FetchByte;
  y := mr[FetchByte and $7F];
  flag := Z_bit;
  Func5 (op2) (y, 0);
  Inc (cycles, 4);
end {TUnReg};


{ unary operation on an array in the bank 0 }
procedure XunAry (op2: pointer);
var
  x, y, z, v, dstf, dstl: byte;
begin
  x := FetchByte;
  y := FetchByte;
  z := FetchByte;
  dstf := IndB (y, z);		{ index of the first destination register }
  dstl := AsLimit (z);		{ index of the last destination register }
  y := y and $07;
  z := z shr 5;
  if x = $70 {RODM/ROUM} then flag := (flag and C_bit) or Z_bit else
	flag := Z_bit;
  if x = $75 {BNUM,BNDM} then x := mr[dstf] else x := 0;
  repeat
    x := Func5 (op2) (mr[dstf], x);
    Inc (cycles, 4);
    if y = z then Exit;
    if dstf = (dstl xor (byte(regstep shr 1) and $07)) then
    repeat
      NextReg (y);
      v := 0;
      x := Func5 (op2) (v, x);
      Inc (cycles, 4);
      if y = z then Exit;
    until FALSE;
    NextReg (y);
    NextReg (dstf);
  until FALSE;
end {XunAry};


{ unary operation on an array in the bank 1 }
procedure YunAry (op2: pointer);
var
  x, y, z: byte;
begin
  x := FetchByte;
  y := FetchByte;
  z := FetchByte;
  z := RegC (y, z);		{ index of the last destination register }
  y := RegA (y);		{ index of the first destination register }
  if x = $70 {ROD/ROU} then flag := (flag and C_bit) or Z_bit else
	flag := Z_bit;
  if x = $75 {BNUM,BNDM} then x := mr[y] else x := 0;
  repeat
    x := Func5 (op2) (mr[y], x);
    Inc (cycles, 4);
    if y = z then Exit;
    NextReg (y);
  until FALSE;
end {YunAry};


{ unary operation on an array in the bank 0, but the result is not stored }
procedure TXunAry (op2: pointer);
var
  x, y, z, v, dstf, dstl: byte;
begin
  FetchByte;
  y := FetchByte;
  z := FetchByte;
  dstf := IndB (y, z);		{ index of the first destination register }
  dstl := AsLimit (z);		{ index of the last destination register }
  y := y and $07;
  z := z shr 5;
  flag := Z_bit;
  x := 0;
  repeat
    v := mr[dstf];
    x := Func5 (op2) (v, x);
    Inc (cycles, 4);
    if y = z then Exit;
    if dstf = (dstl xor (byte(regstep shr 1) and $07)) then
    repeat
      NextReg (y);
      v := 0;
      x := Func5 (op2) (v, x);
      Inc (cycles, 4);
      if y = z then Exit;
    until FALSE;
    NextReg (y);
    NextReg (dstf);
  until FALSE;
end {TXunAry};


{ unary operation on an array in the bank 1, but the result is not stored }
procedure TYunAry (op2: pointer);
var
  x, y, z, v: byte;
begin
  FetchByte;
  y := FetchByte;
  z := FetchByte;
  z := RegC (y, z);		{ index of the last destination register }
  y := RegA (y);		{ index of the first destination register }
  flag := Z_bit;
  x := 0;
  repeat
    v := mr[y];
    x := Func5 (op2) (v, x);
    Inc (cycles, 4);
    if y = z then Exit;
    NextReg (y);
  until FALSE;
end {TYunAry};


{ destination_register <- destination_register op2 source_register
  source register in the bank 1, destination register in the bank 0 }
procedure Xreg (op2: pointer);
var
  x, src, dst: byte;
begin
  x := FetchByte;
  if x < $08 then flag := Z_bit
  else if (x and $FA) = $22 then flag := (flag and C_bit) or Z_bit;
  src := RegA (FetchByte);	{ index of the source register }
  dst := IndD (FetchByte);	{ index of the destination register }
  Proc5 (op2) (mr[dst], mr[src]);
  Inc (cycles, 4);
end {Xreg};


{ destination_register <- destination_register op2 source_register
  source register in the bank 0, destination register in the bank 1 }
procedure Yreg (op2: pointer);
var
  x, src, dst: byte;
begin
  x := FetchByte;
  if x < $08 then flag := Z_bit
  else if (x and $FA) = $22 then flag := (flag and C_bit) or Z_bit;
  dst := RegA (FetchByte);	{ index of the destination register }
  src := IndD (FetchByte);	{ index of the source register }
  Proc5 (op2) (mr[dst], mr[src]);
  Inc (cycles, 4);
end {Yreg};


{ check flags of the operation: destination_register op2 source_register
  source register in the bank 1, destination register in the bank 0 }
procedure TXreg (op2: pointer);
var
  src, dst: byte;
begin
  if FetchByte < $08 then flag := Z_bit
  else flag := (flag and C_bit) or Z_bit;
  src := mr[RegA (FetchByte)];	{ contents of the source register }
  dst := mr[IndD (FetchByte)];	{ contents of the destination register }
  Proc5 (op2) (dst, src);
  Inc (cycles, 4);
end {TXreg};


{ check flags of the operation: destination_register op2 source_register
  source register in the bank 0, destination register in the bank 1 }
procedure TYreg (op2: pointer);
var
  src, dst: byte;
begin
  if FetchByte < $08 then flag := Z_bit
  else flag := (flag and C_bit) or Z_bit;
  dst := mr[RegA (FetchByte)];	{ contents of the destination register }
  src := mr[IndD (FetchByte)];	{ contents of the source register }
  Proc5 (op2) (dst, src);
  Inc (cycles, 4);
end {TYreg};


{ register <- register op2 immediate_byte }
procedure RegIm8 (op2: pointer);
var
  x, y, z: byte;
begin
  x := FetchByte;
  y := FetchByte and $7F;
  z := FetchByte;
  if x < $10 then flag := Z_bit
  else if (x and $FA) = $2A then flag := (flag and C_bit) or Z_bit;
  Proc5 (op2) (mr[y], z);
  Inc (cycles, 4);
end {RegIm8};


{ check flag of the operation: register op2 immediate_byte }
procedure TRegIm8 (op2: pointer);
var
  x, y, z: byte;
begin
  x := FetchByte;
  y := mr[FetchByte and $7F];	{ contents of the register }
  z := FetchByte;
  if x < $10 then flag := Z_bit
  else flag := (flag and C_bit) or Z_bit;
  Proc5 (op2) (y, z);
  Inc (cycles, 4);
end {TRegIm8};


{ two registers <- immediate_byte }
procedure Ld2RegIm8 (op2: pointer {dummy});
var
  y, z: byte;
begin
  FetchByte;
  y := FetchByte and $7F;
  z := FetchByte;
  mr[y] := z;
  mr[y xor $40] := z;
  Inc (cycles, 4);
end {Ld2RegIm8};


{ destination_array <- destination_array op2 source_array
  source array in the bank 1, destination array in the bank 0 }
procedure Xary (op2: pointer);
var
  x, y, z, srcf, srcl, dstf, dstl: byte;
begin
  x := FetchByte;
  if x < $18 then flag := Z_bit
  else if (x and $FA) = $32 then flag := (flag and C_bit) or Z_bit;
  y := FetchByte;
  z := FetchByte;
  srcf := RegA (y);		{ index of the first source register }
  srcl := RegC (y, z);		{ index of the last source register }
  dstf := IndB (y, z);		{ index of the first destination register }
  dstl := AsLimit (z);		{ index of the last destination register }
  repeat
    Proc5 (op2) (mr[dstf], mr[srcf]);
    Inc (cycles, 4);
    if srcf = srcl then Exit;
    if dstf = dstl then
    repeat
      NextReg (srcf);
      y := 0;
      Proc5 (op2) (y, mr[srcf]);
      Inc (cycles, 4);
      if srcf = srcl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {Xary};


{ destination_array <- destination_array op2 source_array
  source array in the bank 0, destination array in the bank 1 }
procedure Yary (op2: pointer);
var
  x, y, z, srcf, srcl, dstf, dstl: byte;
begin
  x := FetchByte;
  if x < $18 then flag := Z_bit
  else if (x and $FA) = $32 then flag := (flag and C_bit) or Z_bit;
  y := FetchByte;
  z := FetchByte;
  dstf := RegA (y);		{ index of the first destination register }
  dstl := RegC (y, z);		{ index of the last destination register }
  srcf := IndB (y, z);		{ index of the first source register }
  srcl := AsLimit (z);		{ index of the last source register }
  repeat
    Proc5 (op2) (mr[dstf], mr[srcf]);
    Inc (cycles, 4);
    if dstf = dstl then Exit;
    if srcf = srcl then
    repeat
      NextReg (dstf);
      z := 0;
      Proc5 (op2) (mr[dstf], z);
      Inc (cycles, 4);
      if dstf = dstl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {Yary};


{ check flags of the operation: destination_array op2 source_array
  source array in the bank 1, destination array in the bank 0 }
procedure TXary (op2: pointer);
var
  y, z, srcf, srcl, dstf, dstl: byte;
begin
  if FetchByte < $18 then flag := Z_bit
  else flag := (flag and C_bit) or Z_bit;
  y := FetchByte;
  z := FetchByte;
  srcf := RegA (y);		{ index of the first source register }
  srcl := RegC (y, z);		{ index of the last source register }
  dstf := IndB (y, z);		{ index of the first destination register }
  dstl := AsLimit (z);		{ index of the last destination register }
  repeat
    y := mr[dstf];
    z := mr[srcf];
    Proc5 (op2) (y, z);
    Inc (cycles, 4);
    if srcf = srcl then Exit;
    if dstf = dstl then
    repeat
      NextReg (srcf);
      y := 0;
      z := mr[srcf];
      Proc5 (op2) (y, z);
      Inc (cycles, 4);
      if srcf = srcl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {TXary};


{ check flags of the operation: destination_array <- destination_array op2 source_array
  source array in the bank 0, destination array in the bank 1 }
procedure TYary (op2: pointer);
var
  y, z, srcf, srcl, dstf, dstl: byte;
begin
  if FetchByte < $18 then flag := Z_bit
  else flag := (flag and C_bit) or Z_bit;
  y := FetchByte;
  z := FetchByte;
  dstf := RegA (y);		{ index of the first destination register }
  dstl := RegC (y, z);		{ index of the last destination register }
  srcf := IndB (y, z);		{ index of the first source register }
  srcl := AsLimit (z);		{ index of the last source register }
  repeat
    y := mr[dstf];
    z := mr[srcf];
    Proc5 (op2) (y, z);
    Inc (cycles, 4);
    if dstf = dstl then Exit;
    if srcf = srcl then
    repeat
      NextReg (dstf);
      y := mr[dstf];
      z := 0;
      Proc5 (op2) (y, z);
      Inc (cycles, 4);
      if dstf = dstl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {TYary};


{ array <- array op2 immediate_value }
procedure AryIm5 (op2: pointer);
var
  x, y, z, im5: byte;
begin
  x := FetchByte;
  if x < $20 then flag := Z_bit
  else if x > $30 then flag := (flag and C_bit) or Z_bit;
  y := FetchByte and $7F;	{ index of the first processed register }
  z := FetchByte;
  im5 := z and $1F;		{ immediate operand }
  z := RegE (y, z);		{ index of the last processed register }
  repeat
    Proc5 (op2) (mr[y], im5);
    Inc (cycles, 4);
    if y = z then Break;
    NextReg (y);
    if x <> $30 then im5 := 0;
  until FALSE;
end {AryIm5};


{ check flags of the operation: array op2 immediate_value }
procedure TAryIm5 (op2: pointer);
var
  x, y, z, im5: byte;
begin
  if FetchByte < $20 then flag := Z_bit
  else flag := (flag and C_bit) or Z_bit;
  y := FetchByte and $7F;	{ index of the first processed register }
  z := FetchByte;
  im5 := z and $1F;		{ immediate operand }
  z := RegE (y, z);		{ index of the last processed register }
  repeat
    x := mr[y];
    Proc5 (op2) (x, im5);
    Inc (cycles, 4);
    if y = z then Break;
    NextReg (y);
    im5 := 0;
  until FALSE;
end {TAryIm5};


procedure MemoReg (op2: pointer);
var
  i, s, x: byte;	{ index register, memory segment, memory contents }
  a: word;		{ address }
begin
  flag := Z_bit;
  i := (FetchByte shr 4) and $01;
  s := Iseg (i);
  a := Wreg (i, 0);
  x := SrcPtr (s, a)^;
  Proc5 (op2) (x, mr[(FetchByte and $7F) xor $40]);
  DstPtr (s, a)^ := x;
  Inc (cycles, 8);
end {MemoReg};


procedure TMemoReg (op2: pointer);
var
  i, s, x: byte;	{ index register, memory segment, memory contents }
  a: word;		{ address }
begin
  flag := Z_bit;
  i := (FetchByte shr 4) and $01;
  s := Iseg (i);
  a := Wreg (i, 0);
  x := SrcPtr (s, a)^;
  Proc5 (op2) (x, mr[(FetchByte and $7F) xor $40]);
  Inc (cycles, 6);
end {TMemoReg};


procedure MemoIm8 (op2: pointer);
var
  i, s, x: byte;	{ index register, memory segment, memory contents }
  a: word;		{ address }
begin
  flag := Z_bit;
  i := (FetchByte shr 4) and $01;
  s := Iseg (i);
  a := Wreg (i, 0);
  x := SrcPtr (s, a)^;
  i := FetchByte;	{ immediate operand }
  Proc5 (op2) (x, i);
  DstPtr (s, a)^ := x;
  Inc (cycles, 8);
end {MemoIm8};


procedure Ldw (op2: pointer {dummy});
var
  dst: byte;
begin
  dst := (FetchByte and $07) or $38;
  mr[dst or $40] := FetchByte;
  mr[dst] := FetchByte;
  Inc (cycles, 4);
end {Ldw};


procedure AdwSbw (op2: pointer);
var
  x, y: byte;
begin
  x := FetchByte;
  y := FetchByte;
  if (y and $80) <> 0 then
  begin
    y := mr[y and $7F];
    Inc (cycles, 2);
  end {if};
  Proc3 (op2) (Ireg(x), y);
  Inc (cycles, 8);
end {Adw};



{ conditional execution (when true) of 'op2' }
procedure Cond (op2: pointer);
begin
  Proc6 (op2) ((flag and cc[FetchByte and $07]) <> 0);
  Inc (cycles, 4);
end {Cond};


{ conditional execution (when false) of 'op2' }
procedure NotCond (op2: pointer);
begin
  Proc6 (op2) ((flag and cc[FetchByte and $07]) = 0);
  Inc (cycles, 4);
end {NotCond};


{ conditional execution (when key pressed) of 'op2' }
procedure KeyCond (op2: pointer);
begin
  FetchByte;
  Proc6 (op2) (ki <> 0);
  Inc (cycles, 4);
end {KeyCond};


{ conditional execution (when key not pressed) of 'op2' }
procedure NotKeyCond (op2: pointer);
begin
  FetchByte;
  Proc6 (op2) (ki = 0);
  Inc (cycles, 4);
end {NotKeyCond};


procedure Jump (x: boolean);
var
  y, z: byte;
begin
  y := FetchByte;
  z := FetchByte;
  if x then pc := (word(y) shl 8) or word(z);
{ Alternatively, the above statement could be replaced with following ones:
  ptrb(PChar(@pc)+1)^ := y;
  ptrb(@pc)^ := z;
  Perhaps they would be more effective, especially if the left sides would
  evaluate to constants. Unfortunately, the pointer to the 'pc' variable is
  probably unknown at the compile state. Also they have a disadvantage to be
  less portable, because little endian order of bytes is assumed. }
end {Jump};


procedure Call (x: boolean);
var
  y, z: byte;
begin
  y := FetchByte;
  z := FetchByte;
  if x then
  begin
    PushPC;
    pc := (word(y) shl 8) or word(z);
  end {if};
  Inc (cycles, 4);
end {Call};


procedure SJump (x: boolean);
var
  y: byte;
begin
  y := FetchByte;
  if x then pc := (pc and $FF00) or word(y);
end {SJump};


procedure SCall (x: boolean);
var
  y: byte;
begin
  y := FetchByte;
  if x then
  begin
    PushPC;
    pc := (pc and $FF00) or word(y);
  end {if};
  Inc (cycles, 4);
end {SCall};


procedure Trap (op2: pointer {dummy});
begin
  PushPC;
  FetchByte;
  pc := word(mr[119] shl 8) or word(mr[55]);
  Inc (cycles, 10);
end {Trap};


{ for illegal operands the emulation may differ from the actual hardware }
procedure PlaIjmp (op2 {destination register}: pointer);
var
  y: byte;
begin
  FetchByte;
  y := FetchByte;
  (ptrw (op2))^ := (word(mr[RegA(y)]) shl 8) or word(mr[RegD(FetchByte)]);
  Inc (cycles, 4);
end {PlaIjmp};


procedure Rtn (x: boolean);
var
  y, z: byte;
begin
  if x then
  begin
    y := SrcPtr(DATA_MEM, PostIncw(SP,1))^;
    z := SrcPtr(DATA_MEM, PostIncw(SP,1))^;
    pc := (word(y) shl 8) or word(z);
  end {if};
  Inc (cycles, 6);
end {Rtn};


procedure Rti (op2: pointer {dummy});
var
  i: integer;
begin
  for i:=0 to INTVECTORS-1 do
  begin
    if (ireq and intmask[i]) <> 0 then
    begin
      iserv := iserv and not intmask[i];
      ireq := ireq and not intmask[i];
      Break;
    end {if};
  end {for};
  Rtn (True);
  Inc (cycles, 4);
end {Rti};


procedure Nop (op2: pointer {dummy});
begin
  FetchByte;
  Inc (cycles, 4);
end {Nop};


{ The emulation of the instructions BUP/BDN (BlockCopy) and SUP/SDN
  (BlockSearch) isn't accurate, because the real processor fetches the
  opcode bytes twice (regardless of the amount of data processed) plus
  once every time an interrupt occurs. }

procedure BlockCopy (op2: pointer);
var
  savepc: word;
  y: byte;
begin
  savepc := pc;
  FetchByte;
  y := FetchByte and $7F;
  DstPtr((ds shr 2) and $03, Func4 (op2) (IY,1))^ :=
		SrcPtr(ds and $03, Func4 (op2) (IX,1))^;
  Dec (mr[y]);
  if mr[y] <> 0 then pc := savepc;
  Inc (cycles, 12);
end {BlockCopy};


procedure BlockSearch (op2: pointer);
var
  savepc: word;
  y: byte;
begin
  savepc := pc;
  FetchByte;
  y := FetchByte and $7F;
  Dec (mr[y]);
  if SrcPtr(ds and $03, Func4 (op2) (IX,1))^ <> mr[IndD(FetchByte)] then
  begin
    if mr[y] <> 0 then pc := savepc;
  end {if};
  Inc (cycles, 12);
end {BlockSearch};


procedure StMemoReg (op2: pointer);
var
  i, y: byte;
  a: word;
begin
  i := Ireg (FetchByte);
  y := FetchByte;
  a := Func4 (op2) (i,1) + regstep;
  DstPtr(Iseg (i), a)^ := mr[y and $7F];
  Inc (cycles, 10);
end {StMemoReg};


procedure XmemoAry (op2: pointer);
var
  i, s, y, z: byte;
begin
  i := Ireg (FetchByte);
  s := Iseg (i);
  y := FetchByte;
  z := FetchByte;
  y := RegB (y, z);	{ index of the first processed register }
  z := RegD (z);	{ index of the last processed register }
  repeat
    DstPtr(s, Func4 (op2) (i,1))^ := mr[y];
    Inc (cycles, 4);
    if y = z then Break;
    NextReg (y);
  until FALSE;
  Inc (cycles, 4);
end {XmemoAry};


procedure YmemoAry (op2: pointer);
var
  i, s, y, z: byte;
begin
  i := Ireg (FetchByte);
  s := Iseg (i);
  y := FetchByte;
  z := FetchByte;
  z := RegC (y, z);	{ index of the last processed register }
  y := RegA (y);	{ index of the first processed register }
  repeat
    DstPtr(s, Func4 (op2) (i,1))^ := mr[y];
    Inc (cycles, 4);
    if y = z then Break;
    NextReg (y);
  until FALSE;
  Inc (cycles, 4);
end {YmemoAry};


procedure StmOffsAry (op2: pointer);
var
  i, s, y, z: byte;
  a: word;
begin
  i := Ireg (FetchByte);
  s := Iseg (i);
  y := FetchByte;
  z := FetchByte;
  a := Func4 (op2) (i, z);	{ destination address }
  z := RegE (y, z);		{ index of the last processed register }
  y := y and $7F;		{ index of the first processed register }
  repeat
    Inc (a, regstep);
    DstPtr(s, a)^ := mr[y];
    Inc (cycles, 4);
    if y = z then Break;
    NextReg (y);
  until FALSE;
  Inc (cycles, 6);
end {StmOffsAry};


procedure LdRegMemo (op2: pointer);
var
  i, y: byte;
begin
  i := Ireg (FetchByte);
  y := FetchByte;
  mr[y and $7F] := SrcPtr(Iseg (i), Func4 (op2) (i,1))^;
  Inc (cycles, 10);
end {LdRegMemo};


procedure XaryMemo (op2: pointer);
var
  i, s, y, z: byte;
begin
  i := Ireg (FetchByte);
  s := Iseg (i);
  y := FetchByte;
  z := FetchByte;
  y := RegB (y, z);	{ index of the first processed register }
  z := RegD (z);	{ index of the last processed register }
  repeat
    mr[y] := SrcPtr(s, Func4 (op2) (i,1))^;
    Inc (cycles, 4);
    if y = z then Break;
    NextReg (y);
  until FALSE;
  Inc (cycles, 4);
end {XaryMemo};


procedure YaryMemo (op2: pointer);
var
  i, s, y, z: byte;
begin
  i := Ireg (FetchByte);
  s := Iseg (i);
  y := FetchByte;
  z := FetchByte;
  z := RegC (y, z);	{ index of the last processed register }
  y := RegA (y);	{ index of the first processed register }
  repeat
    mr[y] := SrcPtr(s, Func4 (op2) (i,1))^;
    Inc (cycles, 4);
    if y = z then Break;
    NextReg (y);
  until FALSE;
  Inc (cycles, 4);
end {YaryMemo};


procedure LdmAryOffs (op2: pointer);
var
  i, s, y, z: byte;
  a: word;
begin
  i := Ireg (FetchByte);
  s := Iseg (i);
  y := FetchByte;
  z := FetchByte;
  a := Func4 (op2) (i, z);	{ source address }
  z := RegE (y, z);		{ index of the last processed register }
  y := y and $7F;		{ index of the first processed register }
  repeat
    mr[y] := SrcPtr(s, a)^;
    Inc (cycles, 4);
    if y = z then Break;
    NextReg (y);
    Inc (a, regstep);
  until FALSE;
  Inc (cycles, 4);
end {LdmAryOffs};


procedure PstIm7 (op2: pointer);
begin
  FetchByte;
  ptrb(op2)^ := FetchByte and $7F;
  Inc (cycles, 4);
end {PstIm8};


procedure PstIm8 (op2: pointer);
begin
  FetchByte;
  Proc4 (op2) (FetchByte);
  Inc (cycles, 4);
end {PstIm8};


procedure PstReg (op2: pointer);
begin
  FetchByte;
  Proc4 (op2) (mr[FetchByte and $7F]);
  Inc (cycles, 4);
end {PstReg};


procedure Gst (op2: pointer);
begin
  FetchByte;
  mr[FetchByte and $7F] := ptrb(op2)^;
  Inc (cycles, 4);
end {Gst};


procedure Off (op2: pointer {dummy});
begin
  FetchByte;
  CpuSleep := True;
  pc := $0000;
  ie := 0;
  iserv := 0;
  ireq := 0;
  s6 := 0;
  kc := $40;
  KeyHandle;
end {Gst};


procedure Wai (op2: pointer {dummy});
begin
  FetchByte;
  CpuWait := True;
end {Wai};



{ ARITHMETICAL AND LOGICAL OPERATIONS }


procedure ZeroBits (x: byte);
begin
  if x <> 0 then flag := flag and not Z_bit;
  if (x and $F0) = 0 then flag := flag or UZ_bit;
  if (x and $0F) = 0 then flag := flag or LZ_bit;
end {ZeroBits};


{ binary operations }

{ addition with carry x <- x + y + Carry }
procedure OpAd (var x, y: byte);
var
  in1, in2, temp, out: cardinal;
begin
  in1 := cardinal(x);
  in2 := cardinal(y);
  out := in1 + in2;
  if (flag and C_bit) <> 0 then Inc(out);
  temp := in1 xor in2 xor out;
  flag := flag and Z_bit;
  if out > $FF then flag := flag or C_bit;
  if (temp and $10) <> 0 then flag := flag or H_bit;
  if (temp and $80) <> 0 then flag := flag or V_bit;
  x := byte(out);
  ZeroBits (x);
end {OpAd};


{ subtraction with borrow x <- x - y - Carry }
procedure OpSb (var x, y: byte);
var
  in1, in2, temp, out: cardinal;
begin
  in1 := cardinal(x);
  in2 := cardinal(y);
  out := in1 - in2;
  if (flag and C_bit) <> 0 then Dec(out);
  temp := in1 xor in2 xor out;
  flag := flag and Z_bit;
  if out > $FF then flag := flag or C_bit;
  if (temp and $10) <> 0 then flag := flag or H_bit;
  if (temp and $80) <> 0 then flag := flag or V_bit;
  x := byte(out);
  ZeroBits (x);
end {OpSb};


{ BCD addition with carry x <- x + y + Carry }
procedure OpAdb (var x, y: byte);
var
  in1, in2, out: cardinal;
begin
  in1 := cardinal(x);
  in2 := cardinal(y);
{ lower nibble }
  out := (in1 and $0F) + (in2 and $0F);
  if (flag and C_bit) <> 0 then Inc(out);
  flag := flag and Z_bit;
{ decimal adjustement }
  if out > $09 then
  begin
    out := ((out + $06) and $0F) or $10;
    if out > $1F then Dec(out,$10);
    flag := flag or H_bit;
  end {if};
{ upper nibble }
  Inc(out, (in1 and $F0) + (in2 and $F0));
  if ((in1 xor in2 xor out) and $80) <> 0 then flag := flag or V_bit;
{ decimal adjustement }
  if out > $9F then
  begin
    Inc(out,$60);
    flag := flag or C_bit;
  end {if};
  x := byte(out);
  ZeroBits (x);
end {OpAdb};


{ BCD subtraction with borrow x <- x - y - Carry }
procedure OpSbb (var x, y: byte);
var
  in1, in2, out: cardinal;
begin
  in1 := cardinal(x);
  in2 := cardinal(y);
{ lower nibble }
  out := (in1 and $0F) - (in2 and $0F);
  if (flag and C_bit) <> 0 then Dec(out);
  flag := flag and Z_bit;
{ decimal adjustement }
  if out > $09 then
  begin
    out := (out - $06) or cardinal (-$10);
    flag := flag or H_bit;
  end {if};
{ upper nibble }
  Inc(out, (in1 and $F0) - (in2 and $F0));
  if ((in1 xor in2 xor out) and $80) <> 0 then flag := flag or V_bit;
{ decimal adjustement }
  if out > $9F then
  begin
    Dec(out,$60);
    flag := flag or C_bit;
  end {if};
  x := byte(out);
  ZeroBits (x);
end {OpSbb};


procedure OpAn (var x, y: byte);
begin
  x := x and y;
  flag := flag and Z_bit;
  ZeroBits (x);
end {OpAn};


procedure OpBit (var x, y: byte);
begin
  x := not x and y;
  flag := flag and Z_bit;
  ZeroBits (x);
end {OpBit};


procedure OpXr (var x, y: byte);
begin
  x := x xor y;
  flag := flag and Z_bit;
  ZeroBits (x);
end {OpXr};


procedure OpNa (var x, y: byte);
begin
  x := not (x and y);
  flag := (flag and Z_bit) or (C_bit or V_bit or H_bit);
end {OpNa};


procedure OpOr (var x, y: byte);
begin
  x := x or y;
  flag := (flag and Z_bit) or (C_bit or V_bit or H_bit);
  ZeroBits (x);
end {OpOr};


procedure OpLd (var x, y: byte);
begin
  x := y;
end {OpLd};


{ upper digit of y -> lower digit of x }
procedure OpLdh (var x, y: byte);
begin
  x := (x and $F0) or (y shr 4);
end {OpLdh};


{ lower digit of y -> upper digit of x }
procedure OpLdl (var x, y: byte);
begin
  x := (x and $0F) or (y shl 4);
end {OpLdl};


procedure OpLds (var x, y: byte);
begin
  x := y;
  OpSwp (x, y);
end {OpLds};


{ upper digit of y -> lower digit of x, then swap digits of x }
procedure OpLdhs (var x, y: byte);
begin
  x := (x shr 4) or (y and $F0);
end {OpLdhs};


{ lower digit of y -> upper digit of x, then swap digits of x }
procedure OpLdls (var x, y: byte);
begin
  x := (x shl 4) or (y and $0F);
end {OpLdls};


{ unary operations, dummy second operand and returned value }


function OpRod (var x: byte; y: byte) : byte;
var
  z: byte;
begin
  regstep := word (-1);
  z := x;
  x := x shr 1;
  if (flag and C_bit) <> 0 then x := x or $80;
  flag := flag and Z_bit;
  if (z and $01) <> 0 then flag := flag or C_bit;
  ZeroBits (x);
  OpRod := 0;
end {OpRod};


function OpRou (var x: byte; y: byte) : byte;
var
  temp: byte;
begin
  temp := x;
  OpAd (x, temp);
  OpRou := 0;
end {OpRou};


function OpMtb (var x: byte; y: byte) : byte;
var
  temp: byte;
begin
  temp := x;
  OpAdb (x, temp);
  OpMtb := 0;
end {OpMtb};


function OpInv (var x: byte; y: byte) : byte;
begin
  x := not x;
  flag := (flag and Z_bit) or (C_bit or V_bit or H_bit);
  ZeroBits (x);
  OpInv := 0;
end {OpInv};


function OpCmp (var x: byte; y: byte) : byte;
var
  temp: byte;
begin
  temp := x;
  x := 0;
  OpSb (x, temp);
  OpCmp := 0;
end {OpCmp};


function OpCmpb (var x: byte; y: byte) : byte;
var
  temp: byte;
begin
  temp := x;
  x := 0;
  OpSbb (x, temp);
  OpCmpb := 0;
end {OpCmpb};


function OpSwp1 (var x: byte; y: byte) : byte;
begin
  x := (x shl 4) or (x shr 4);
  OpSwp1 := 0;
end {OpSwp1};


{ shifts by 4 and 8 bits }

function OpDiu (var x: byte; y: byte) : byte;
begin
  OpDiu := x shr 4;
  x := (x shl 4) or y;
  flag := flag and Z_bit;
  ZeroBits (x);
end {OpDiu};


function OpDid (var x: byte; y: byte) : byte;
begin
  regstep := word (-1);
  OpDid := x shl 4;
  x := (x shr 4) or y;
  flag := flag and Z_bit;
  ZeroBits (x);
end {OpDid};


function OpByu (var x: byte; y: byte) : byte;
begin
  OpByu := x;
  x := y;
  flag := flag and Z_bit;
  ZeroBits (x);
end {OpByu};


function OpByd (var x: byte; y: byte) : byte;
begin
  regstep := word (-1);
  OpByd := x;
  x := y;
  flag := flag and Z_bit;
  ZeroBits (x);
end {OpByu};


{ exchanges }

{ exchange of two bytes }
procedure OpXc (var x, y: byte);
var
  temp: byte;
begin
  temp := x;
  x := y;
  y := temp;
end {OpXc};


{ upper digit of y <-> lower digit of x }
procedure OpXch (var x, y: byte);
var
  temp: byte;
begin
  temp := x;
  x := (x and $F0) or (y shr 4);
  y := (y and $0F) or (temp shl 4);
end {OpXch};


{ lower digit of y <-> upper digit of x }
procedure OpXcl (var x, y: byte);
var
  temp: byte;
begin
  temp := x;
  x := (x and $0F) or (y shl 4);
  y := (y and $F0) or (temp shr 4);
end {OpXcl};


{ exchange of two bytes then swap digits }
procedure OpXcs (var x, y: byte);
begin
  OpXc (x, y);
  OpSwp2 (x, y);
end {OpXcs};


{ rotation of four digits (nibbles) counterclockwise }
procedure OpXcls (var x, y: byte);
var
  temp: byte;
begin
  temp := (y shr 4) or (x and $F0);
  x := (x shl 4) or (y and $0F);
  y := temp;
end {OpXcls};


{ rotation of four digits (nibbles) clockwise }
procedure OpXchs (var x, y: byte);
var
  temp: byte;
begin
  temp := (y shl 4) or (x and $0F);
  x := (x shr 4) or (y and $F0);
  y := temp;
end {OpXchs};


{ swap nibbles in a single byte }
procedure OpSwp (var x, y: byte);
begin
  x := (x shl 4) or (x shr 4);
end {OpSwp};


{ swap nibbles in two bytes }
procedure OpSwp2 (var x, y: byte);
begin
  x := (x shl 4) or (x shr 4);
  y := (y shl 4) or (y shr 4);
end {OpSwp2};



{ WRITING TO THE STATUS REGISTERS }


procedure OpIe (x: byte);
begin
  ie := x;
  if (x and $80) = 0 then
  begin
    iserv := 0;
    ireq := 0;
  end {if};
end {OpIe};


procedure OpDs (x: byte);
begin
  ds := x;
end {OpDs};


procedure OpKc (x: byte);
begin
  kc := x;
  KeyHandle;
end {OpKc};


procedure OpPd (x: byte);
begin
  pd := x;
end {OpPd};


procedure OpFl (x: byte);
begin
  flag := x;
end {OpFl};


procedure OpAs (x: byte);
begin
  asreg := x and $7F;
end {OpAs};


end.
