unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, StdCtrls, IniFiles, ThdTimer, Buttons;

type
  TMainForm = class(TForm)
    RunTimer: TThreadedTimer;
    RefreshTimer: TTimer;
    PwrSwImage: TImage;
    LcdImage: TImage;
    FaceImage: TImage;
    procedure OnRunTimer(Sender: TObject);
    procedure OnRefreshTimer(Sender: TObject);
    procedure FaceMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FaceMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormShow(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormDeactivate(Sender: TObject);
    procedure ApplicationDeactivate(Sender: TObject);
    procedure PwrSwMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
  private
    { Private declarations }
  protected
    procedure RebuildWindowRgn;
  public
    constructor Create(AOwner: TComponent); override;
  end;

var
    MainForm: TMainForm;

implementation

{$R *.dfm}

uses
    Def, Cpu, Debug, Keyboard, Lcd;

const
    FaceName: string = 'face.bmp';
    KeyMapName: string = 'keymap.dat';
    IniName: string = 'fx7500g.ini';
    Rom0Name: string = 'rom0.bin';
    Rom1Name: string = 'rom1.bin';
    RamName: string = 'ram.bin';
    RegName: string = 'register.bin';
    LoadMsg: string = 'Failed to load the file ';
    SaveMsg: string = 'Failed to save the file ';

    KEYMAPSIZE = 128;

var
    FaceBmp, LcdBmp, PwrSwOnBmp, PwrSwOffBmp: TBitMap;
    RedrawReq: boolean;		{ true if the LcdBmp image has changed and
				  needs to be redrawn }

{ keyboard }
    keymap: array[0..KEYMAPSIZE-1] of byte;

{ LCD }
    BkColor: TColor = clWhite;
    ScrMem: array[0..LCDSIZE-1] of byte; { shadow LCD data memory }
    ScrOn: boolean;		{ bit 0 of the S6 register state }

{ CPU }
    CpuSpeed: integer;		{ how many instructions executes the emulated
				  CPU at each RunTimer event call }


procedure ResetAll;
begin
  LcdInit;
  TestCounter := SelfTest;
  CpuReset;
end {ResetAll};


{ draw the LCD contents }
procedure View;
const
  PIXSIZE = 2;
var
  Pixel, Index, X, Y: Integer;
  Data: byte;
  S6Bit0: boolean;
begin
  with LcdBmp.Canvas, LcdBmp.Canvas.Brush do
  begin
    Style := bsSolid;

{ handle the LCD control register }
    S6Bit0 := (s6 and $01) <> 0;
    if ScrOn <> S6Bit0 then
    begin
      RedrawReq := True;
      ScrOn := S6Bit0;
      if ScrOn then
      begin	{turn the display on}
{ it is assummed that the lcdimage is cleared when the LCD is turned off }
        FillChar (ScrMem, LCDSIZE, 0);
        Color := BkColor;
      end
      else
      begin	{turn the display off}
        Color := clLtGray;
      end {if};
      FillRect (Rect(0, 0, 192, 128));
    end {if};

{ draw the pixels }
    if not ScrOn then Exit;	{display turned off}
    Index := 0;
    Y :=(LCDHEIGHT-1) * PIXSIZE;
    while Y >= 0 do
    begin
      X :=(8*LCDWIDTH-1) * PIXSIZE;
      while X >= 0 do
      begin
        Data := lcdimage[Index];
        if ScrMem[Index] <> Data then
        begin
          RedrawReq := True;
          ScrMem[Index] := Data;
          for Pixel := 0 to 7 do
          begin
            if (Data and $01) <> 0 then Color := clBlack
                                   else Color := BkColor;
            Data := Data shr 1;
            FillRect (Rect(X, Y, X+PIXSIZE, Y+PIXSIZE));
            Dec (X, PIXSIZE);
          end {for Pixel};
        end
        else
        begin
          Dec (X, 8*PIXSIZE);
        end {if};
        Inc (Index);
      end {Row};
      Dec (Y, PIXSIZE);
    end {Col};
  end {with};
end; {proc View}


procedure TMainForm.OnRefreshTimer(Sender: TObject);
begin
  View;
  if RedrawReq = True then LcdImage.Picture.Bitmap := LcdBmp;
  RedrawReq := False;
  if TestCounter > 0 then
  begin
    Dec (TestCounter);
    KeyHandle;
  end {if};
end;


{ called when mouse button pressed }
procedure TMainForm.FaceMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  i, r, c, k: Integer;
begin
{ proceed only when left mouse button pressed }
  if Button <> mbLeft then Exit;

  KeyCode[1] := 1;
  for i := 0 to KEYPADS do
  begin
    with keypad[i] do
    begin
      if (X >= L) and (X < L+SX*col) and (((X-L) mod SX) < W) and
	(Y >= T) and (((Y-T) mod SY) < H) then
      begin
        c := (X-L) div SX;
        r := (Y-T) div SY;
        k := col*r + c;
        if k < cnt then
        begin
          Inc (KeyCode[1], k);
          break;
        end {if};
      end {if};
      Inc (KeyCode[1], cnt);
    end {with};
  end {for};

  if KeyCode[1] > LASTKEYCODE-1 then	{ no valid key pressed }
  begin
    KeyCode[1] := 0;
{ dragging a captionless form by clicking anywhere on the client area outside
  the controls }
    if BorderStyle = bsNone then
    begin
      ReleaseCapture;
      SendMessage (Handle, WM_NCLBUTTONDOWN, HTCAPTION, 0);
    end {if};
  end {if};

  KeyHandle;
end {proc};


{ called when mouse button released }
procedure TMainForm.FaceMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
{ proceed only when left mouse button was pressed }
  if Button <> mbLeft then Exit;

{ what to do if the mouse button was released over a pressed ... }
  case KeyCode[1] of
    1:  Application.Minimize;		{ ...Application Minimize key }
    2:  Close;				{ ...Application Close key }
  end {case};
  KeyCode[1] := 0;
  KeyHandle;
end;


procedure TMainForm.FormShow(Sender: TObject);
begin
  KeyCode[1] := 0;
  KeyCode[2] := 0;
  PowerSwitch := True;
  CpuStop := False;
  CpuSleep := False;
  CpuDelay := 0;
  CpuSteps := -1;
  BreakPoint := -1;
{ background image }
  if FileExists (FaceName) then
  begin
    FaceBmp.LoadFromFile (FaceName);
    FaceBmp.Transparent := False;
    FaceImage.Picture.Bitmap := FaceBmp;
    { power switch }
    PwrSwOnBmp.Canvas.Draw (0, -328, FaceBmp);
    PwrSwOffBmp.Canvas.Draw (0, -53, PwrSwOnBmp);
    PwrSwOffBmp.Canvas.Draw (0, 15, PwrSwOnBmp);
  end
  else
    MessageDlg (LoadMsg + FaceName, mtWarning, [mbOk], 0);
{ clear the LCD area }
  with LcdBmp.Canvas do
  begin
    Brush.Style := bsSolid;
    Brush.Color := BkColor;;
    FillRect (Rect(0, 0, 192, 128));
  end {with};
  CpuSpeed := OscFreq * integer(RunTimer.Interval);
  ResetAll;
  ScrOn := False;
  RunTimer.Enabled := PowerSwitch;
  RefreshTimer.Enabled := True;
  RedrawReq := True;
end;


{ load the memory images }
procedure MemLoad;
var
  f: file;
  transferred: integer;
begin
{ load the keymap table }
  FillChar (keymap, KEYMAPSIZE, 0);
  if FileExists (KeyMapName) then
  begin
    AssignFile (f, KeyMapName);
    Reset (f, 1);
    BlockRead (f, keymap, KEYMAPSIZE, transferred);
    CloseFile (f);
  end;
{ load the ROM0 image }
  if FileExists (Rom0Name) then
  begin
    AssignFile (f, Rom0Name);
    Reset (f, 3);
    BlockRead (f, rom0, ROM0SIZE, transferred);
    CloseFile (f);
  end
  else MessageDlg (LoadMsg + Rom0Name, mtWarning, [mbOk], 0);
{ load the ROM1 image }
  if FileExists (Rom1Name) then
  begin
    AssignFile (f, Rom1Name);
    Reset (f, 1);
    BlockRead (f, rom1, ROM1SIZE, transferred);
    CloseFile (f);
  end
  else MessageDlg (LoadMsg + Rom1Name, mtWarning, [mbOk], 0);
{ load the RAM image }
  if FileExists (RamName) then
  begin
    AssignFile (f, RamName);
    Reset (f, 1);
    BlockRead (f, ram, RAMSIZE, transferred);
    CloseFile (f);
  end {if};
{ load the register file image }
  if FileExists (RegName) then
  begin
    AssignFile (f, RegName);
    Reset (f, 1);
    BlockRead (f, mr, 128, transferred);
    CloseFile (f);
  end {if};
end {MemLoad};


procedure IniLoad;
var
  Ini1: TIniFile;
begin
  Ini1 := TIniFile.Create (ExpandFileName(IniName));
  with Ini1 do
  begin
    OscFreq := ReadInteger ('Settings', 'OscFreq', 2000);
    SelfTest := ReadInteger ('Settings', 'SelfTest', 0);
  end {with};
  Ini1.Free;
end {IniLoad};


{ initialise the application }
procedure TMainForm.FormCreate(Sender: TObject);
begin
  Brush.Style := bsClear;	{ transparent form }
  FaceBmp := TBitMap.Create;
  FaceBmp.Width := 492;
  FaceBmp.Height := 561;
  LcdBmp := TBitMap.Create;
  LcdBmp.Width := 192;
  LcdBmp.Height := 128;
  PwrSwOnBmp := TBitMap.Create;
  PwrSwOnBmp.Width := 12;
  PwrSwOnBmp.Height := 69;
  PwrSwOffBmp := TBitMap.Create;
  PwrSwOffBmp.Width := 12;
  PwrSwOffBmp.Height := 69;
  IniLoad;
  MemLoad;
end;


{ save the memory images }
procedure MemSave;
var
  f: file;
begin
  {$I-}
{ save the RAM image }
  AssignFile (f, RamName);
  Rewrite (f, 1);
  BlockWrite (f, ram, RamSize);
  CloseFile (f);
  if IOResult <> 0 then MessageDlg (SaveMsg + RamName, mtWarning, [mbOk], 0);
{ save the register file image }
  AssignFile (f, RegName);
  Rewrite (f, 1);
  BlockWrite (f, mr, 128);
  CloseFile (f);
  if IOResult <> 0 then MessageDlg (SaveMsg + RamName, mtWarning, [mbOk], 0);
  {$I+}
end {MemSave};


{ terminate the application }
procedure TMainForm.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
{ As the memory appears to be deallocated before destroying the timers, it is
  necessary to prevent the emulated CPU to access the memory after it has been
  freed. }
  CpuStop := True;
  RunTimer.Enabled := False;
  RefreshTimer.Enabled := False;
  MemSave;
  FaceBmp.Free;
  LcdBmp.Free;
  PwrSwOnBmp.Free;
  PwrSwOffBmp.Free;
end;


procedure TMainForm.FormKeyPress(Sender: TObject; var Key: Char);
var
  x: integer;
begin
  x := integer(Ord(Key));
  if x < 32 then Exit;
  if x >= KEYMAPSIZE then Exit;
  x := integer(keymap[x]);
  if x = 0 then Exit;
  KeyCode[2] := x;
  KeyHandle;
end;


procedure TMainForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  case Key of
    VK_F3:	DebugForm.Show;
    VK_HOME:	KeyCode[2] := 30;		{ SHIFT }
    VK_END:	KeyCode[2] := 31;		{ ALPHA }
    VK_ESCAPE:	KeyCode[2] := 46;		{ AC }
    VK_DELETE:	KeyCode[2] := 45;		{ DEL }
    VK_RETURN:  KeyCode[2] := 61;		{ EXE }
    VK_LEFT:	KeyCode[2] := keymap[29];	{ left }
    VK_RIGHT:	KeyCode[2] := keymap[28];	{ right }
    VK_UP:	KeyCode[2] := keymap[30];	{ up }
    VK_DOWN:	KeyCode[2] := keymap[31];	{ down }
    VK_F8:	KeyCode[2] := 62;		{ Init }
  end {case};
  KeyHandle;
end;


procedure TMainForm.FormKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  KeyCode[2] := 0;
  KeyHandle;
end;


constructor TMainForm.Create(AOwner: TComponent);
begin
  inherited;
  RebuildWindowRgn;
end;


{ custom window region due to irregular shape of the FaceBmp image }
procedure TMainForm.RebuildWindowRgn;
var
  FullRgn, Rgn: THandle;
  ClientX, ClientY, X, Y: Integer;
  Img: TBitMap;
const
  TRCOLOR: TColor = $00FFFFFF;	{ transparent color }
begin
{ determine the relative coordinates of the client area }
  ClientX := (Width - ClientWidth) div 2;
  ClientY := Height - ClientHeight - ClientX;
{ create the region for the entire client area }
  FullRgn := CreateRectRgn (ClientX, ClientY, ClientWidth, ClientHeight);
{ subtract transparent areas from the FullRgn }
  Img := TBitMap.Create;
  if FileExists (FaceName) then
  begin
    Img.LoadFromFile (FaceName);
    Img.Width := ClientWidth;
    Img.Height := ClientHeight;
    for Y := 0 to Img.Height-1 do
    begin
{ regions for the transparent areas on the left side of the Face image }
      X := 0;
      while Img.Canvas.Pixels [X, Y] = TRCOLOR do Inc (X);
      if X <> 0 then
      begin
        Rgn := CreateRectRgn (ClientX, ClientY + Y,
		ClientX + X, ClientY + Y + 1);
        CombineRgn (FullRgn, FullRgn, Rgn, rgn_Diff);
      end {if};
{ regions for the transparent areas on the right side of the Face image }
      X := Img.Width - 1;
      while Img.Canvas.Pixels [X, Y] = TRCOLOR do Dec (X);
      if X <> Img.Width - 1 then
      begin
        Rgn := CreateRectRgn (ClientX + X + 1, ClientY + Y,
		ClientX + Img.Width, ClientY + Y + 1);
        CombineRgn (FullRgn, FullRgn, Rgn, rgn_Diff);
      end {if};
    end {for};
  end {if};
  Img.Free;
{ set window region }
  SetWindowRgn(Handle, FullRgn, True);
end;


procedure TMainForm.OnRunTimer(Sender: TObject);
begin
  if CpuDelay > 0 then
  begin
    Dec (CpuDelay);
    acycles := 0;
    Exit;
  end {if};
  Inc (acycles, CpuSpeed);
  while acycles > 0 do
  begin
    if CpuStop then
    begin
      acycles := 0;
      break;
    end {if};

    cycles := 0;
    CpuRun;
    LcdTransfer;
    Dec (acycles,cycles);

    if CpuSteps > 0 then
    begin
      Dec (CpuSteps);
      if CpuSteps = 0 then
      begin
        CpuStop := True;
        acycles := 0;
        DebugForm.Show;
        break;
      end {if};
    end {if};

    if (BreakPoint >= 0) and (BreakPoint = pc) then
    begin
      CpuStop := True;
      acycles := 0;
      DebugForm.Show;
      break;
    end {if};

  end {while};
end;


procedure TMainForm.FormDeactivate(Sender: TObject);
begin
  KeyCode[2] := 0;
  KeyHandle;
end;


procedure TMainForm.ApplicationDeactivate(Sender: TObject);
begin
  KeyCode[2] := 0;
  KeyHandle;
end;


procedure TMainForm.PwrSwMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  PowerSwitch := not PowerSwitch;
  ResetAll;
  RunTimer.Enabled := PowerSwitch;
  with PwrSwImage.Picture do
  begin
    if PowerSwitch then Bitmap := PwrSwOnBmp else Bitmap := PwrSwOffBmp;
  end {with};
end;

end.
