{ Casio uPD1007 (NM-320) assembler }

unit Asem;


interface

  uses
    SysUtils;

  var
{ assembler input }
    InBuf: string[64];
    InIndex: integer;
{ assembler output }
    OutBuf: array[0..2] of byte;
    OutIndex: integer;

  procedure Assemble;


implementation


type

  t_kind = (
    INHERENT,	{ no operands, single byte }
    INH2,	{ no operands, two bytes }
    REGISTER,	{ single register }
    SEARCH,	{ single register, bank implied by the opcode }
    GST_ARG,	{ register,status_register }
    PST_ARG,	{ status_register,register/immediate }
    REGRIM8,	{ register,register/immediate }
    REGIM8,	{ register,immediate }
    REGREG,	{ register,register (irregular) }
    LD_ARG,	{ register,various }
    ST_ARG,	{ memory,register/immediate }
    IREGIM8,	{ index_register,immediate }
    WREGIM16,	{ word_register,immediate }
    IJMP_ARG,	{ cond_code,word_register }
    JUMP_ARG,	{ cond_code,immediate }
    ARYARYIM6,	{ array,array/immediate }
    ARYIM6,	{ array,immediate }
    ARYARY,	{ array,array }
    LDM_ARG,	{ array,various }
    STM_ARG,	{ memory,array }
    ARY,	{ single array }
    MTBM_ARG,	{ single array, instructions CMPBM and MTBM }
    DB_ARG	{ up to 3 data bytes }
  );


  tab = record
    str: string[5];
    val2: word;
    case boolean of
      True: (kind: t_kind);
      False: (val1: word);
  end;


const

  PLUS = $0000;
  MINUS = $0004;
  ANYTHING = 0;		{ unused value }

  NMNEM = 94;		{ index of the last item in the 'mnem' array }

  mnem: array [0..NMNEM] of tab = (
{ no operands, single byte }
    (	str:'rtn';	val2:$0058;	kind:INHERENT	),
    (	str:'cani';	val2:$0068;	kind:INHERENT	),
    (	str:'rti';	val2:$0078;	kind:INHERENT	),
    (	str:'bup';	val2:$00E2;	kind:INHERENT	),
    (	str:'bdn';	val2:$00F2;	kind:INHERENT	),
    (	str:'nop';	val2:$00FA;	kind:INHERENT	),

{ no operands, two bytes }
    (	str:'off';	val2:$8050;	kind:INH2	),
    (	str:'trp';	val2:$80FF;	kind:INH2	),

{ single register }
    (	str:'cmp';	val2:$00E0;	kind:REGISTER	),
    (	str:'cmpb';	val2:$00E4;	kind:REGISTER	),
    (	str:'rou';	val2:$00E6;	kind:REGISTER	),
    (	str:'inv';	val2:$00F0;	kind:REGISTER	),
    (	str:'rod';	val2:$00F6;	kind:REGISTER	),
    (	str:'byu';	val2:$80E3;	kind:REGISTER	),
    (	str:'biu';	val2:$80E4;	kind:REGISTER	),
    (	str:'diu';	val2:$80E5;	kind:REGISTER	),
    (	str:'mtb';	val2:$80E6;	kind:REGISTER	),
    (	str:'byd';	val2:$80F3;	kind:REGISTER	),
    (	str:'did';	val2:$80F5;	kind:REGISTER	),

{ single register, bank implied by the opcode }
    (	str:'sup';	val2:$8034;	kind:SEARCH	),
    (	str:'sdn';	val2:$803C;	kind:SEARCH	),

{ GST operands - register,status_register }
    (	str:'gst';	val2:$00D4;	kind:GST_ARG	),

{ PST operands - status_register,register/immediate }
    (	str:'pst';	val2:$00C4;	kind:PST_ARG	),

{ arithmetical/logical operations - register,register/immediate }
    (	str:'adb';	val2:$0000;	kind:REGRIM8	),
    (	str:'sbb';	val2:$0001;	kind:REGRIM8	),
    (	str:'ad';	val2:$0002;	kind:REGRIM8	),
    (	str:'sb';	val2:$0003;	kind:REGRIM8	),
    (	str:'an';	val2:$0004;	kind:REGRIM8	),
    (	str:'na';	val2:$0005;	kind:REGRIM8	),
    (	str:'or';	val2:$0006;	kind:REGRIM8	),
    (	str:'xr';	val2:$0007;	kind:REGRIM8	),
    (	str:'tadb';	val2:$0020;	kind:REGRIM8	),
    (	str:'tsbb';	val2:$0021;	kind:REGRIM8	),
    (	str:'tad';	val2:$0022;	kind:REGRIM8	),
    (	str:'tsb';	val2:$0023;	kind:REGRIM8	),
    (	str:'tan';	val2:$0024;	kind:REGRIM8	),
    (	str:'tna';	val2:$0025;	kind:REGRIM8	),
    (	str:'tor';	val2:$0026;	kind:REGRIM8	),
    (	str:'txr';	val2:$0027;	kind:REGRIM8	),
{ instruction BIT with an immediate operand needs to be handled separately }
    (	str:'bit';	val2:$8053;	kind:REGRIM8	),

{ register,immediate }
    (	str:'ldle';	val2:$80C4;	kind:REGIM8	),
    (	str:'ldlo';	val2:$80C5;	kind:REGIM8	),
    (	str:'stle';	val2:$80C6;	kind:REGIM8	),
    (	str:'stlo';	val2:$80C7;	kind:REGIM8	),

{ register,register (irregular)
  instruction SWP can accept a single register as well }
    (	str:'swp';	val2:$8014;	kind:REGREG	),
    (	str:'xcls';	val2:$8016;	kind:REGREG	),
    (	str:'xc';	val2:$801C;	kind:REGREG	),
    (	str:'xchs';	val2:$801E;	kind:REGREG	),

{ LD operands - register,various }
    (	str:'ld';	val2:ANYTHING;	kind:LD_ARG	),

{ ST operands - memory,register/immediate }
    (	str:'st';	val2:ANYTHING;	kind:ST_ARG	),

{ LDW operands - word_register,immediate }
    (	str:'ldw';	val2:$0040;	kind:WREGIM16	),

{ ADW/SBW operands - index_register,immediate }
    (	str:'adw';	val2:$0048;	kind:IREGIM8	),
    (	str:'sbw';	val2:$004C;	kind:IREGIM8	),

{ IJMP operands - cond_code,word_register }
    (	str:'ijmp';	val2:$0050;	kind:IJMP_ARG	),

{ JMP/CAL operands - cond_code,immediate }
    (	str:'jmp';	val2:$0070;	kind:JUMP_ARG	),
    (	str:'cal';	val2:$0060;	kind:JUMP_ARG	),

{ arithmetical multibyte operations - array,array/immediate }
    (	str:'adbm';	val2:$0010;	kind:ARYARYIM6	),
    (	str:'sbbm';	val2:$0011;	kind:ARYARYIM6	),
    (	str:'adm';	val2:$0012;	kind:ARYARYIM6	),
    (	str:'sbm';	val2:$0013;	kind:ARYARYIM6	),
    (	str:'tadbm';	val2:$0030;	kind:ARYARYIM6	),
    (	str:'tsbbm';	val2:$0031;	kind:ARYARYIM6	),
    (	str:'tadm';	val2:$0032;	kind:ARYARYIM6	),
    (	str:'tsbm';	val2:$0033;	kind:ARYARYIM6	),
{ logical multibyte operations - array,array only }
    (	str:'anm';	val2:$0014;	kind:ARYARYIM6	),
    (	str:'nam';	val2:$0015;	kind:ARYARYIM6	),
    (	str:'orm';	val2:$0016;	kind:ARYARYIM6	),
    (	str:'xrm';	val2:$0017;	kind:ARYARYIM6	),
    (	str:'tanm';	val2:$0034;	kind:ARYARYIM6	),
    (	str:'tnam';	val2:$0035;	kind:ARYARYIM6	),
    (	str:'torm';	val2:$0036;	kind:ARYARYIM6	),
    (	str:'txrm';	val2:$0037;	kind:ARYARYIM6	),

{ array,immediate }
    (	str:'ldlem';	val2:$80D4;	kind:ARYIM6	),
    (	str:'ldlom';	val2:$80D5;	kind:ARYIM6	),
    (	str:'stlem';	val2:$80D6;	kind:ARYIM6	),
    (	str:'stlom';	val2:$80D7;	kind:ARYIM6	),

{ array,array (irregular) }
    (	str:'swpm';	val2:$8015;	kind:ARYARY	),
    (	str:'xclsm';	val2:$8017;	kind:ARYARY	),
    (	str:'xcm';	val2:$801D;	kind:ARYARY	),
    (	str:'xchsm';	val2:$801F;	kind:ARYARY	),

{ LDM operands - array,various }
    (	str:'ldm';	val2:ANYTHING;	kind:LDM_ARG	),

{ STM operands - memory,array }
    (	str:'stm';	val2:ANYTHING;	kind:STM_ARG	),

{ single array }
    (	str:'cmpm';	val2:$00E1;	kind:ARY	),
    (	str:'byum';	val2:$00E3;	kind:ARY	),
    (	str:'dium';	val2:$00E5;	kind:ARY	),
    (	str:'roum';	val2:$00E7;	kind:ARY	),
    (	str:'invm';	val2:$00F1;	kind:ARY	),
    (	str:'bnusm';	val2:$80F0;	kind:ARY	),
    (	str:'bium';	val2:$80F4;	kind:ARY	),
    (	str:'bydm';	val2:$00F3;	kind:ARY	),
    (	str:'didm';	val2:$00F5;	kind:ARY	),
    (	str:'rodm';	val2:$00F7;	kind:ARY	),

{ single array, different encoding }
    (	str:'cmpbm';	val2:$00F4;	kind:MTBM_ARG	),
    (	str:'mtbm';	val2:$80F6;	kind:MTBM_ARG	),

{ pseudo instruction }
    (	str:'db';	val2:ANYTHING;	kind:DB_ARG	)
  );


{ status register names for the commands GST, PST }

{ set bits of 'val1' specify which commands accept the 'str' status register }
  PSTREG = $0100;
  PSTIMM = $0200;

  NSREGS = 5;		{ index of the last item in the 'sregs' array }

  sregs: array[0..NSREGS] of tab = (
    (	str:'as'; val2:$0002;		val1:$0002+PSTREG+PSTIMM	),
    (	str:'f';  val2:$0001;		val1:$0001+PSTREG		),
    (	str:'ie'; val2:$0003;		val1:$0003+PSTREG+PSTIMM	),
    (	str:'if'; val2:$80F1 xor $00D4;	val1:$0001+PSTIMM		),
    (	str:'ki'; val2:$8051 xor $00D4;	val1:0				),
    (	str:'ko'; val2:$0000;		val1:$0000+PSTREG+PSTIMM	)
  );


  NIREGS = 3;		{ index of the last item in the 'iregs' array }

{ table of index registers }
  iregs: array[0..NIREGS] of tab = (
    (	str:'ix';	val2:$0000;	val1:ANYTHING	),
    (	str:'iy';	val2:$0001;	val1:ANYTHING	),
    (	str:'iz';	val2:$0002;	val1:ANYTHING	),
    (	str:'sp';	val2:$0003;	val1:ANYTHING	)
  );


  NWREGS = 7;		{ index of the last item in the 'wregs' array }

{ table of word size registers }
  wregs: array[0..NWREGS] of tab = (
    (	str:'ix';	val2:$0000;	val1:ANYTHING	),
    (	str:'iy';	val2:$0001;	val1:ANYTHING	),
    (	str:'iz';	val2:$0002;	val1:ANYTHING	),
    (	str:'sp';	val2:$0007;	val1:ANYTHING	),
    (	str:'v0';	val2:$0006;	val1:ANYTHING	),
    (	str:'v1';	val2:$0005;	val1:ANYTHING	),
    (	str:'v2';	val2:$0004;	val1:ANYTHING	),
    (	str:'v3';	val2:$0003;	val1:ANYTHING	)
  );


  NCC = 13;		{ index of the last item in the 'cc' array }

{ table of conditional codes }
  cc: array[0..NCC] of tab = (
    (	str:'k';	val2:$0001;	val1:ANYTHING	),
    (	str:'lz';	val2:$0002;	val1:ANYTHING	),
    (	str:'uz';	val2:$0003;	val1:ANYTHING	),
    (	str:'nz';	val2:$0004;	val1:ANYTHING	),
    (	str:'v';	val2:$0005;	val1:ANYTHING	),
    (	str:'h';	val2:$0006;	val1:ANYTHING	),
    (	str:'c';	val2:$0007;	val1:ANYTHING	),
    (	str:'nk';	val2:$0009;	val1:ANYTHING	),
    (	str:'nlz';	val2:$000A;	val1:ANYTHING	),
    (	str:'nuz';	val2:$000B;	val1:ANYTHING	),
    (	str:'z';	val2:$000C;	val1:ANYTHING	),
    (	str:'nv';	val2:$000D;	val1:ANYTHING	),
    (	str:'nh';	val2:$000E;	val1:ANYTHING	),
    (	str:'nc';	val2:$000F;	val1:ANYTHING	)
  );


{ compare the string 's' with the 'InBuf' at location 'InIndex' without
  the case sensitivity,
  update the 'InIndex' and return True if both string match }
function ParseString (s: string): boolean;
var
  n: integer;
begin
  ParseString := False;
  if InIndex + Length(s) - 1 > Length(InBuf) then exit;
  n := 0;
  while n < Length(s) do
  begin
    if s[n+1] <> LowerCase(InBuf[InIndex + n]) then exit;
    Inc (n);
  end {while};
  Inc (InIndex, n);
  ParseString := True;
end {ParseString};


{ This function searches the table for a string pointed to by the InIndex,
  and picks from the table the longest matching string.
  Returns index to the table and updates InIndex when string found,
  or leaves InIndex unchanged when not found. }
function ParseTable (
  out x: integer;	{ returned index to the table }
  var t: array of tab;	{ table to be searched }
  last: integer		{ index of the last item }
  ) : boolean;		{ TRUE when string found }
var
  maxindex, save, i: integer;
begin
  maxindex := InIndex;
  save := InIndex;
  ParseTable := FALSE;
  for i := 0 to last do
  begin
    InIndex := save;
    if ParseString (t[i].str) and (InIndex > maxindex) then
    begin
      ParseTable := TRUE;
      x := i;
      maxindex := InIndex;
    end {if};
  end {for};
  InIndex := maxindex;
end {ParseTable};


{ a specified character expected }
function ParseChar (c: char) : boolean;
begin
  result := (InIndex <= Length(InBuf)) and (InBuf[InIndex] = c);
  if result then Inc (InIndex);
end {ParseChar};


{ move the 'InIndex' to the first character different from space }
procedure SkipBlanks;
begin
  while ParseChar (' ') do ;
end {SkipBlanks};


{ comma expected }
function ParseComma : boolean;
begin
  SkipBlanks;
  ParseComma := ParseChar (',');
  SkipBlanks;
end {ParseComma};


{ value of a hex digit }
function GetDigit (c: char) : integer;
const
  digits: string[22] = '0123456789ABCDEFabcdef';
var
  i: integer;
begin
  i := 1;
  while (i<=22) and (c <> digits[i]) do Inc (i);
  if i>16 then GetDigit := i-7 else GetDigit := i-1;
end {GetDigit};


{ the function expects a number in base 'radix',
  updates the InIndex }
function ParseNumber (out value: word; radix: word): boolean;
var
  x, y: word;
begin
  value := 0;
  ParseNumber := FALSE;
  while InIndex <= Length(InBuf) do
  begin
    x := word(GetDigit(InBuf[InIndex]));
    if x >= radix then break;	{ stop when not a digit }
    y := value*radix + x;
    if y < value then break;	{ overflow, stop when too much digits }
    value := y;
    Inc (InIndex);
    ParseNumber := TRUE;
  end;
end {ParseNumber};


{ the function expects a hexadecimal (with a prefix &H) or decimal number
  within specified range }
function EvalAndTest (out value: word; range: word) : boolean;
var
  radix: word;
begin
  EvalAndTest := FALSE;
  SkipBlanks;
{ parse for the prefix of a hexadecimal numeral }
  if ParseString ('&h') then radix := 16 else radix := 10;
  if not ParseNumber (value, radix) then exit;	{ failure, missing number }
  EvalAndTest := (value <= range);		{ FALSE when out of range }
end {EvalAndTest};


{ the function expects a main register, returns the register index,
  index value in range 128..191 if indirect addressing specified }
function RegArgum (out x: word) : boolean;
var
  y: word;
begin
  SkipBlanks;
  RegArgum := FALSE;
  y := 0;
  if ParseString ('i') then y := $C0
  else if not ParseString ('r') then exit;	{ failure, invalid register }
  if not ParseNumber (x, 10) then exit;		{ failure, missing number }
  if x > 127 then exit;				{ failure, out of range }
  x := x xor y;
  RegArgum := (x < 192);
end {RegArgum};


{ swap the contents of two word-type variables }
procedure SwapWords (var w1, w2: word);
var
  temp: word;
begin
  temp := w1;
  w1 := w2;
  w2 := temp;
end {SwapWords};


function Reg1 (x: word) : word;
begin
  Reg1 := (x and $07) or ((x shl 1) and $70);
end {Reg1};


function Reg2 (x: word) : word;
begin
  Reg2 := ((x shr 3) and $17) or ((x shl 5) and $E0);
end {Reg2};


function Reg3 (x, y: word) : word;
begin
  Reg3 := ((y shr 3) and $17) or ((x shl 5) and $E0);
end {Reg3};


function Im6 (reg, imm: word) : word;
begin
  Im6 := ((reg shl 5) and $E0) or (imm and $1F);
end {Im6};


{ test whether both registers belong to different register banks }
function CheckRegPair (x1, x2: word) : boolean;
begin
  CheckRegPair := (x1 < 64) xor (x2 < 64);
end {CheckRegPair};


{ expects a sign '+' or '-' }
function ParseSign (out x: word) : boolean;
begin
  SkipBlanks;
  x := PLUS;
  ParseSign := TRUE;
  if not ParseChar ('+') then
  begin
    x := MINUS;
    ParseSign := ParseChar ('-');
  end {if};
end {ParseSign};


{ the function continues parsing an array after the first register has been
  recognised }
function AryArgum (
  first: word;		{ already recognised first register of the array }
  var last: word	{ last register of the array }
  ) : boolean;		{ TRUE when success }
begin
  AryArgum := FALSE;
  if not ParseString ('..') then exit;		{ failure, invalid argument }
  if not RegArgum (last) then exit;		{ failure, register expected }
  AryArgum := (first and $FFF8) = (last and $FFF8);
end {AryArgum};


function AddArySize (x, y: word) : word;
begin
  AddArySize := (x and $FFF8) or ((x + y) and $0007);
end {AddArySize};


{ opening parenthesis followed by an index register }
function ParIr (out x: integer {index to the table 'iregs'}) : boolean;
begin
  x := 0;
  ParIr := False;
  if not ParseChar ('(') then Exit;
  SkipBlanks;
  ParIr := ParseTable (x, iregs, NIREGS);
end {ParIr};


{ assemble the instruction in the InBuf and place the result in the OutBuf,
  on exit InIndex contains the position of an error (warning: it can point
  past the end of the InBuf !), otherwise 0 }
procedure Assemble;
var
  sign: word;
  x1, x2, x3, x4: word;

  i: integer;		{ index to the tables }
  kod: word;		{ opcode }
begin
  InIndex := 1;
  OutIndex := 0;

  SkipBlanks;				{ skip leading blanks }
  if (InIndex > Length(InBuf))	{ empty InBuf? }
	or ParseChar (';') then		{ comment? }
  begin
    InIndex := 0;
    exit;				{ success }
  end {if};

{ parse the mnemonic }
  if not ParseTable (i, mnem, NMNEM) then exit;
					{ failure, mnemonic not recognised }
  kod := mnem[i].val2;

{ parse the arguments }
  case mnem[i].kind of

    INHERENT:		{ no operands, single byte }
      begin
        OutBuf[0] := byte(kod);
        OutIndex := 1;
      end {case INHERENT};


    INH2:		{ no operands, two bytes }
      begin
        OutBuf[0] := byte(kod);
        OutBuf[1] := Hi(kod);
        OutIndex := 2;
      end {case INHERENT};


    REGISTER:		{ single register }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if x1 > 127 then exit;		{ failure, illegal indirect }
        if x1 < 64 then kod := kod xor $08;
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        OutIndex := 2;
      end {case REGISTER};


    SEARCH:		{ single register, bank implied by the opcode }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if x1 > 127 then exit;		{ failure, illegal indirect }
        if not CheckRegPair(x1, (kod shl 3) and $40) then exit;	{ failure, bank }
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        OutIndex := 2;
      end {case REGISTER};


    GST_ARG:		{ register,status_register }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if x1 > 127 then exit;		{ failure, illegal indirect }
        if x1 < 64 then kod := kod xor $08;
        if not ParseComma then exit;	{ failure, comma expected }
        if not ParseTable (i, sregs, NSREGS) then exit;
				{ failure, status register expected }
        kod := kod xor sregs[i].val2;
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        OutIndex := 2;
      end {case GST_ARG};


    PST_ARG:		{ status_register,register/immediate }
      begin
        SkipBlanks;
        if not ParseTable (i, sregs, NSREGS) then exit;
				{ failure, status register expected }
        kod := kod xor ($80FF and sregs[i].val1);
        if not ParseComma then exit;	{ failure, comma expected }

{ second operand: register }
        if RegArgum (x1) then
        begin
          if x1 > 127 then exit;	{ failure, illegal indirect }
          if (sregs[i].val1 and PSTREG) = 0 then exit;
				{ failure, unsupported status register }
          if x1 < 64 then kod := kod xor $08;
          OutBuf[0] := byte(kod);
          OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        end
{ second operand: immediate }
        else
        begin
          if (sregs[i].val1 and PSTIMM) = 0 then exit;
				{ failure, unsupported status register }
          if not EvalAndTest(x2,$FF) then exit;	{ failure, invalid value }
          OutBuf[0] := byte(kod xor $04);
          OutBuf[1] := byte(x2);
        end {if};
        OutIndex := 2;
      end {case PST_ARG};


    REGRIM8:		{ register,register/immediate }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if x1 < 64 then kod := kod xor $08;
        if not ParseComma then exit;	{ failure, comma expected }

{ second operand: register (indirect allowed) }
        if RegArgum (x2) then
        begin
          if not CheckRegPair(x1,x2) then exit;	{ failure, register bank }
          if x1 > x2 then SwapWords(x1,x2);
          x2 := Reg2(x2);
        end

{ second operand: immediate (indirect not allowed),
  instruction BIT with an immediate operand needs to be handled separately }
        else
        begin
          if x1 > 127 then exit;	{ failure, indirect not allowed }
          if (kod and $FFF7) = $8053 then x2 := $0001 else x2 := $8000;
          kod := kod xor x2;
          if not EvalAndTest(x2,$FF) then exit; { failure, invalid value }
        end {if};
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case REGRIM8};


    REGIM8:		{ register,immediate (indirect not allowed) }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if x1 > 127 then exit;		{ failure, indirect not allowed }
        if x1 < 64 then kod := kod xor $08;
        if not ParseComma then exit;	{ failure, comma ecpected }
        if not EvalAndTest(x2,$FF) then exit;	{ failure, invalid value }
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case REGIM8};


    REGREG:		{ register,register (irregular) }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if ParseComma then
        begin
          if not RegArgum (x2) then exit;	{ failure, register expected }
          if not CheckRegPair(x1,x2) then exit;	{ failure, register bank }
          if x1 > x2 then
            SwapWords(x1,x2)
          else if (kod and $FFF7) = $8016 then	{ XCLS/XCHS ? }
            kod := kod xor $08;
          OutBuf[0] := byte(kod);
          OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
          OutBuf[2] := byte(Reg2(x2));
          OutIndex := 3;
        end
        else
        begin
{ instruction SWP can accept a single register (indirect not allowed) }
          if kod <> $8014 then exit;		{ failure, comma expected }
          kod := kod xor ($80E0 xor $8014);
          if x1 > 127 then exit;		{ failure, illegal indirect }
          if x1 < 64 then kod := kod xor $08;
          OutBuf[0] := byte(kod);
          OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
          OutIndex := 2;
        end {if};
      end {case REGREG};


    LD_ARG:		{ register,various }
      begin
        if not RegArgum (x1) then exit;	{ failure, register expected }
        if not ParseComma then exit;	{ failure, comma expected }

{ second operand: register (indirect allowed) }
        if RegArgum (x2) then
        begin
          if not CheckRegPair(x1,x2) then exit;	{ failure, register bank }
          kod := $5C;
          if x1 > x2 then
          begin
            SwapWords(x1,x2);
            kod := $54;
          end {if};
          OutBuf[0] := byte(kod);
          OutBuf[1] := byte(Reg1(x1) or $80);
          OutBuf[2] := byte(Reg2(x2));
          OutIndex := 3;
        end

{ second operand: memory pointed to by an index register }
        else if ParIr (i) then
        begin

  { postincrement/postdecrement (indirect destination not allowed) }
          if ParseChar (')') then
          begin
            if not ParseSign (sign) then exit;	{ failure }
            if x1 > 127 then exit;	{ failure, indirect not allowed }
            if x1 < 64 then kod := $A8 else kod := $A0;
            OutBuf[0] := byte(kod + sign + iregs[i].val2);
            OutBuf[1] := byte(Reg1(x1));
            OutIndex := 2;
          end

  { with an offset }
          else
          begin
            if not ParseSign (sign) then exit;	{ failure }
  { register as offset (indirect destination allowed) }
            if RegArgum (x2) then
            begin
              if x1 < 64 then exit;	{ failure, register bank }
              if x2 >= 64 then exit;	{ failure, register bank }
              kod := $A0;
              SwapWords (x1,x2);
              x2 := Reg2(x2);
            end
  { immediate offset }
            else
            begin
              if not EvalAndTest(x2,$FF) then exit; { failure, invalid value }
              if x1 >= 64 then exit;	{ failure, register bank }
              kod := $A8;
            end {if};

            if not ParseChar (')') then exit;	{ failure }
            OutBuf[0] := byte(kod + sign + iregs[i].val2);
            OutBuf[1] := byte(Reg1(x1) or $80);
            OutBuf[2] := byte(x2);
            OutIndex := 3;
          end {if};

        end {if second operand: memory}

{ second operand: immediate (indirect destination not allowed) }
        else
        begin
          if x1 > 127 then exit;	{ failure, indirect not allowed }
          if x1 < 64 then kod := $5E else kod := $56;
          if not EvalAndTest(x2,$FF) then exit;	{ failure, invalid value }
          OutBuf[0] := byte(kod);
          OutBuf[1] := byte(Reg1(x1) or $80);
          OutBuf[2] := byte(x2);
          OutIndex := 3;
        end {if second operand: immediate};

      end {case LD_ARG};


    ST_ARG:		{ memory,register/immediate }
      begin
{ first operand: predecement/preincrement }
        if ParseSign (sign) then
        begin
          if not ParIr (i) then exit;		{ failure }
          if not ParseChar (')') then exit;	{ failure }
          if not ParseComma then exit;		{ failure, comma expected }

  { second operand: register (indirect not allowed) }
          if RegArgum (x1) then
          begin
            if x1 > 127 then exit;	{ failure, indirect not allowed }
            if x1 < 64 then kod := $88 else kod := $80;
            x1 := Reg1(x1);
          end

  { second operand: immediate }
          else
          begin
            kod := $C8;
            sign := sign shl 2;
            if not EvalAndTest(x1,$FF) then exit; { failure, invalid value }
          end {if};

          OutBuf[0] := byte(kod + sign + iregs[i].val2);
          OutBuf[1] := byte(x1);
          OutIndex := 2;
        end {if first operand: predecement/preincrement}

{ first operand: index register with an optional offset }
        else
        begin
          if not ParIr (i) then exit;		{ failure }

  { no offset }
          if ParseChar (')') then
          begin
            if not ParseComma then exit;	{ failure, comma expected }
            if not EvalAndTest(x2,$FF) then exit; { failure, invalid value }
            OutBuf[0] := byte($D0 + iregs[i].val2);
            OutBuf[1] := byte(x2);
            OutIndex := 2;
          end

  { with an offset }
          else
          begin
            if not ParseSign (sign) then exit;		{ failure }
  { register as offset (indirect source allowed) }
            if RegArgum (x1) then
            begin
              if x1 >= 64 then exit;	{ failure, register bank }
              kod := $80;
            end
  { immediate offset }
            else
            begin
              if not EvalAndTest(x1,$FF) then exit; { failure, invalid value }
              kod := $88;
            end {if};

            if not ParseChar (')') then	exit;	{ failure }
            if not ParseComma then exit;	{ failure, comma expected }

  { second operand: register }
            if not RegArgum (x2) then exit;		{ failure }

            if kod = $80 then	{ register as offset }
            begin
              if x2 < 64 then exit;		{ failure, register bank }
              x2 := Reg2(x2);
            end
            else		{ immediate offset }
            begin
              if x2 >= 64 then exit;		{ failure, register bank }
              SwapWords (x1,x2);
            end {if};

            OutBuf[0] := byte(kod + sign + iregs[i].val2);
            OutBuf[1] := byte(Reg1(x1) or $80);
            OutBuf[2] := byte(x2);
            OutIndex := 3;
          end {if};

        end {if first operand: index register with an optional offset};

      end {case ST_ARG};


    IREGIM8:		{ word_register,immediate }
      begin
        SkipBlanks;
        if not ParseTable (i, iregs, NIREGS) then exit;	{ failure }
        kod := kod xor iregs[i].val2;
        if not ParseComma then exit;		{ failure, comma expected }
        if not EvalAndTest(x1,$FF) then exit;	{ failure, invalid value }
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(x1);
        OutIndex := 2;
      end {case IREGIM8};


    WREGIM16:		{ word_register,immediate }
      begin
        SkipBlanks;
        if not ParseTable (i, wregs, NWREGS) then exit;	{ failure }
        kod := kod xor wregs[i].val2;
        if not ParseComma then exit;		{ failure, comma expected }
        if not EvalAndTest(x1,$FFFF) then exit;	{ failure, invalid value }
        OutBuf[0] := byte(kod);
        OutBuf[1] := Hi(x1);
        OutBuf[2] := byte(x1);
        OutIndex := 3;
      end {case WREGIM16};


    IJMP_ARG:		{ cond_code,word_register }
      begin
        SkipBlanks;
        if ParseTable (i, cc, NCC) then
        begin
          kod := kod xor cc[i].val2;
          if not ParseComma then exit;	{ failure, comma expected }
        end {if};
        SkipBlanks;
        if ParseTable (i, wregs, NWREGS) then
        begin
          x2 := 56 + wregs[i].val2;
          x1 := x2;
        end
        else
        begin
          if not RegArgum (x2) then exit;	{ failure, register expected }
          if not ParseComma then exit;		{ failure, comma expected }
          if not RegArgum (x1) then exit;	{ failure, register expected }
          if (x2 < 64) or (x1 >= 64) then exit;	{ failure, register bank }
        end {if};
        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        OutBuf[2] := byte(Reg2(x2));
        OutIndex := 3;
      end {case IJMP_ARG};


    JUMP_ARG:		{ cond_code,immediate }
      begin
        SkipBlanks;
        if ParseTable (i, cc, NCC) then
        begin
          kod := kod xor cc[i].val2;
          if not ParseComma then exit;	{ failure, comma expected }
        end {if};
        if not EvalAndTest(x1,$FFFF) then exit;	{ failure, invalid value }
        OutBuf[0] := byte(kod);
        OutBuf[1] := Hi(x1);
        OutBuf[2] := byte(x1);
        OutIndex := 3;
      end {case JUMP_ARG};


    ARYARYIM6:		{ array,array/immediate }
      begin
{ first operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        if x1 < 64 then kod := kod xor $08;
        if not ParseComma then exit;		{ failure, comma expected }

{ second operand: array }
        if RegArgum (x3) then
        begin
          if not AryArgum (x3, x4) then exit;	{ failure, invalid array }
          if not CheckRegPair (x1,x3) then exit;	{ failure }
          if x1 > x3 then
          begin
            SwapWords (x1,x3);
            SwapWords (x2,x4);
          end {if};

  { check the alignment, 'invalid array' error when fail }
          if (x1 and $0007) <> (x3 and $0007) then exit;	{ failure }
          if (x2 and $0007) <> (x4 and $0007) then exit;	{ failure }

          x2 := Reg3(x2,x4);
          x3 := 0;
        end {if second operand: array or first register}

{ second operand: immediate (indirect destination not allowed) }
        else
        begin
  { logical operations don't allow an immediate operand }
          if (kod and $04) <> 0 then exit;	{ failure }
          kod := kod xor $8000;
  { indirect destination illegal }
          if x1 > 127 then exit;	{ failure, indirect not allowed }
  { immediate operand }
          if not EvalAndTest(x3,$3F) then exit;	{ failure, invalid value }
          x2 := Im6(x2,x3);
          x3 := (not x3 shr 2) and $08;	{ inverted most significant bit }
        end {if};

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod) or byte(x3);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case ARYARYIM6};


    ARYIM6:		{ array,immediate }
      begin
{ first operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        if x1 > 127 then exit;		{ failure, indirect not allowed }
        if x1 < 64 then kod := kod xor $08;
        if not ParseComma then exit;		{ failure, comma expected }
{ immediate operand }
        if not EvalAndTest(x3,$3F) then exit;	{ failure, invalid value }
        x2 := Im6(x2,x3);
        x3 := (not x3 shr 2) and $08;	{ inverted most significant bit }

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod) or byte(x3);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case ARYIM6};


    ARYARY:		{ array,array }
      begin
{ first operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        if not ParseComma then exit;		{ failure, comma expected }
{ second operand: array }
        if not RegArgum (x3) then exit;		{ failure, register expected }
        if not AryArgum (x3, x4) then exit;	{ failure, invalid array }
        if not CheckRegPair (x1,x3) then exit;	{ failure }
        if x1 > x3 then
        begin
          SwapWords (x1,x3);
          SwapWords (x2,x4);
        end
        else if (kod and $FFF7) = $8017	then	{ XCLSM/XCHSM? }
        begin
          kod := kod xor $08;
        end {if};
  { check the alignment, 'invalid array' error when fail }
        if (x1 and $0007) <> (x3 and $0007) then exit;		{ failure }
        if (x2 and $0007) <> (x4 and $0007) then exit;		{ failure }

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        OutBuf[2] := byte(Reg3(x2,x4));
        OutIndex := 3;
      end {case ARYARY};


    LDM_ARG:		{ array,various }
      begin
{ first operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        if not ParseComma then exit;		{ failure, comma expected }

{ second operand: array }
        if RegArgum (x3) then
        begin
          if not AryArgum (x3, x4) then exit;	{ failure, invalid array }
          if not CheckRegPair (x1,x3) then exit;	{ failure }
          kod := $805D;
          if x1 > x3 then
          begin
            kod := $8055;
            SwapWords (x1,x3);
            SwapWords (x2,x4);
          end {if};

  { check the alignment, 'invalid array' error when fail }
          if (x1 and $0007) <> (x3 and $0007) then exit;	{ failure }
          if (x2 and $0007) <> (x4 and $0007) then exit;	{ failure }

          x2 := Reg3(x2,x4);
          x3 := 0;
        end { if second operand: array }

{ indirect destination is illegal for the remaining source addressing modes }
        else if x1 > 127 then exit	{ failure, indirect not allowed }

{ second operand: memory pointed to by an index register }
        else if ParIr (i) then
        begin

  { postincrement/postdecrement }
          if ParseChar (')') then
          begin
            if not ParseSign (sign) then exit;		{ failure }
            if x1 < 64 then kod := $00B8 else kod := $00B0;
            x2 := Reg3(x2,0);
            x3 := 0;
          end

  { with an offset }
          else
          begin
            if not ParseSign (sign) then exit;		{ failure }

  { register as offset }
            if RegArgum (x3) then
            begin
              if x1 < 64 then exit;		{ failure, register bank }
              if x3 >= 64 then exit;		{ failure, register bank }
              if not CheckRegPair (x1,x3) then exit;	{ failure }
              if not ParseChar (')') then exit;		{ failure }
              x4 := AddArySize (x3, x2-x1);
              kod := $80B0;
              x2 := Reg3(x4,x1);
              x1 := x3;
              x3 := 0;
            end

  { immediate offset }
            else
            begin
              if not EvalAndTest(x3,$3F) then exit; { failure, invalid value }
              if x1 >= 64 then exit;		{ failure, register bank }
              if not ParseChar (')') then exit;		{ failure }
              kod := $80B8;
              x2 := Im6(x2,x3);
              x3 := (not x3 shr 2) and $08; { inverted most significant bit }
            end {if};
          end {if};

          Inc (kod, sign + iregs[i].val2);
        end { if second operand: memory }

{ second operand: immediate }
        else
        begin
          if x1 < 64 then kod := $805F else kod := $8057;
  { immediate operand }
          if not EvalAndTest(x3,$3F) then exit;	{ failure, invalid value }
          x2 := Im6(x2,x3);
          x3 := (not x3 shr 2) and $08;	{ inverted most significant bit }
        end { if second operand: immediate };

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod) or byte(x3);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case LDM_ARG};


    STM_ARG:		{ memory,array }
      begin
{ first operand: predecement/preincrement }
        if ParseSign (sign) then
        begin
          if not ParIr (i) then exit;		{ failure }
          kod := $0090;
        end

{ first operand: index register with an offset }
        else
        begin
          if not ParIr (i) then exit;		{ failure }
          if not ParseSign (sign) then exit;	{ failure }
  { register as offset }
          if RegArgum (x3) then
          begin
            if x3 >= 64 then exit;		{ failure, register bank }
            kod := $8090;
          end
  { immediate offset }
          else
          begin
            if not EvalAndTest(x3,$3F) then exit; { failure, invalid value }
            kod := $8098;
          end {if};
        end {if};

        if not ParseChar (')') then exit;	{ failure }
        if not ParseComma then exit;		{ failure, comma expected }

{ second operand: array }
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if x1 > 127 then exit;		{ failure, indirect not allowed }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }

        if (kod = $8090) and (x1 < 64) then exit; { failure, register bank }
        if (kod > $8090) and (x1 >= 64) then exit; { failure, register bank }

        if kod < $8090 then
        begin
          if x1 < 64 then kod := kod xor $08;
          x2 := Reg3(x2,0);
          x3 := 0;
        end
        else if kod = $8090 then
        begin
          x4 := AddArySize (x3,x2-x1);
          x2 := Reg3(x4,x1);
          x1 := x3;
          x3 := 0;
        end
        else
        begin
          x2 := Im6(x2,x3);
          x3 := (not x3 shr 2) and $08;	{ inverted most significant bit }
        end {if};

        OutBuf[0] := byte(kod + sign + iregs[i].val2);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod) or byte(x3);
        OutBuf[2] := byte(x2);
        OutIndex := 3;
      end {case STM_ARG};


    ARY,		{ single array }
    MTBM_ARG:		{ single array, instructions CMPBM and MTBM }
      begin
        if not RegArgum (x1) then exit;		{ failure, register expected }
        if not AryArgum (x1, x2) then exit;	{ failure, invalid array }
        if x1 < 64 then kod := kod xor $08;

        if mnem[i].kind = MTBM_ARG then
        begin
          if x1 > 127 then exit;	{ failure, indirect not allowed }
          x2 := x2 and $07;
        end
        else
        begin
          if x1 < 64 then x2 := x2 and $07 else x1 := x1 and $07;
        end {if};

        OutBuf[0] := byte(kod);
        OutBuf[1] := byte(Reg1(x1)) or Hi(kod);
        OutBuf[2] := byte(Reg2(x2));
        OutIndex := 3;
      end {cases ARY, MTBM_ARG};


    DB_ARG:		{ up to 3 data bytes }
      begin
        while (OutIndex < 3) and EvalAndTest (x1, $FF) do
        begin
          OutBuf[OutIndex] := x1;
          Inc (OutIndex);
          if not ParseComma then break;
        end {while};
      end {case DB_ARG};


{ else an internal error }

  end {case};

{ the rest of the InBuf is allowed to be padded with spaces only }
  SkipBlanks;
  if (InIndex > Length(InBuf)) 	{ end of line? }
	or ParseChar (';') then		{ comment? }
    InIndex := 0;			{ success }
{ otherwise failure, extra characters encountered }

end {Assemble};


end.
