{ common stuff, CPU registers, memory }

unit Def;

interface

  type
    ptrb = ^byte;		{ unsigned 8-bit }
    ptrw = ^word;		{ unsigned 16-bit }
    nibble = byte;		{ it should be ensured that any returned or
				  stored data of this type have upper four
				  bits cleared }

  const

{ status bits of the Flag Register F }
    C_bit	= $80;
    H_bit	= $40;
    V_bit	= $20;
    NZ_bit	= $10;
    UZ_bit	= $08;
    LZ_bit	= $04;

{ bits of the IE register }
    INT_enable:	array[0..2] of byte = ( $10, $20, $40 );

{ bits of the IF register }
    INT_serv:	array[0..2] of byte = ( $20, $40, $80 );
    INT_input:	array[0..2] of byte = ( $02, $04, $08 );
    EN2_bit	= $10;
    EN1_bit	= $01;

{ bits of the LCD control port in the immediate operand of the LDL*/STL*
  instructions }
    VDD2_bit	= $80;	{ port EN1 controlled by bit 0 of the IF register }
    OP_bit	= $20;
    CE3_bit	= $04;
    CE2_bit	= $02;
    CE1_bit	= $01;
    LCDCE	= CE1_bit;

    ROM0SIZE	= $1800;
    ROM1SIZE	= $8000;
    RAMSTART	= $C000;
    RAMSIZE	= $2000;
    RAMEND	= RAMSTART+RAMSIZE;

    INT_LATENCY	= 7;	{ number of clock cycles, must be odd }

  var

    rom0: array[0..ROM0SIZE-1,0..2] of byte;
    rom1: array[0..ROM1SIZE-1] of byte;
    ram: array[0..RAMSIZE-1] of byte;

{ 8-bit registers }
    mr: array[0..127] of byte;	{ main (general purpose) register file }
    ko: byte;
    ki: byte;
    ie: byte;
    ifreg: byte;	{ 'if' is a reserved word in Pascal }
    asreg: byte;	{ 'as' is a reserved word in Pascal }
    flag: byte;

{ 16-bit register }
    pc: word;

    savepc: word;		{ address of the executed instruction }
    irqcnt: array[0..2] of integer;	{ interrupt request counters
				  implementing the interrupt latency:
				  value < 0 - counting in progress
				  value = 0 - idle
				  value > 0 - waiting to be serviced }
    dummysrc: byte = $FF;	{ free adress space }
    dummydst: byte;		{ free address space }
    opcode: array[0..2] of byte;
    opindex: integer;		{ index to the opcode table }
    cycles: integer;		{ counter of the clock pulses }
    acycles: integer;		{ clock pulse counter accumulator }
    procptr: pointer;		{ pointer to a procedure that should be
				  executed after a machine code instruction,
				  usually to complete an I/O register write
				  cycle }
    OscFreq: integer;		{ CPU clock frequency in kHz }
    CpuStop: boolean;		{ True stops the CPU, used in the debug mode }
    CpuDelay: integer;		{ delay after hiding the Debug Window,
				  prevents the program from crashing when the
				  Debug Window was made visible too early }
    CpuSleep : boolean;		{ True if the CPU in the power-off state, can
				  be waken up by an INT2 interrupt or AC key }
    CpuSteps: integer;		{ ignored when < 0 }
    BreakPoint: integer;	{ ignored when < 0 }
    SelfTest: integer;		{ for how many LCD refresh cycles (forever if
				  negative) the self-test jumper should be
				  connected after the reset }


  function SrcPtr (address: word) : ptrb;
  function DstPtr (address: word) : ptrb;
  function FetchOpcode: word;
  function FetchByte: byte;
  procedure DoPorts;


implementation

  uses Cpu, Lcd;


{ returns the pointer to a read-type resource at specified address }
function SrcPtr (address: word) : ptrb;
begin
  if (ie and $03) <> 0 then SrcPtr := @dummysrc
  else if address < $4000 then SrcPtr := @rom1[address]
  else if address < $6000 then SrcPtr := @ram[address-$4000]
  else if address < $8000 then SrcPtr := @ram[address-$6000]
  else if address < $C000 then SrcPtr := @rom1[address-$4000]
  else if address < $E000 then SrcPtr := @ram[address-$C000]
  else SrcPtr := @ram[address-$E000];
end {SrcPtr};


{ returns the pointer to a write-type resource at specified address
  Note: due to incomplete address decoding the RAM can be accessed at
  following address ranges:
  $4000..$5FFF, $6000..$7FFF, $C000..$DFFF, $E000..$FFFF }
function DstPtr (address: word) : ptrb;
begin
  if ((ie and $03) = 0) and ((address and $4000) <> 0) then
    DstPtr := @ram[address and $1FFF]
  else
    DstPtr := @dummydst;
end {DstPtr};


{ 9-bit instruction opcode
  Note: accessing the internal ROM doesn't consume any clock cycles }
function FetchOpcode: word;
var
  saveie: byte;
begin
  savepc := pc;
  if pc < ROM0SIZE then
  begin
    opcode[0] := rom0[pc,0];
    opcode[1] := rom0[pc,1];
    opcode[2] := rom0[pc,2];
  end
  else
  begin
    saveie := ie;
    ie := 0;
    opcode[0] := SrcPtr(pc)^;
    opcode[1] := SrcPtr(pc+1)^;
    opcode[2] := SrcPtr(pc+2)^;
    ie := saveie;
    Inc (cycles, 2);
  end {if};
  opindex := 1;
  Inc (pc);
  FetchOpcode := word(opcode[0]) or ((word(opcode[1]) shl 1) and $0100);
end {FetchOpcode};


{ subsequent bytes of the instruction }
function FetchByte: byte;
begin
  FetchByte := opcode[opindex];
  if pc > ROM0SIZE then
  begin
    Inc (pc);
    Inc (cycles, 2);
  end {if};
  Inc (opindex);
end {FetchByte};


procedure DoPorts;
begin
{ the EN1 output controls the LCD power supply }
  if (ifreg and EN1_bit) <> 0 then
    lcdctrl := lcdctrl or VDD2_bit
  else
  begin
    if (lcdctrl and VDD2_bit) <> 0 then LcdInit;
    lcdctrl := lcdctrl and not VDD2_bit;
    ifreg := ifreg and not INT_input[1];
  end {if};
{ the EN2 output is wired to the INT0 input }
  if (ifreg and EN2_bit) <> 0 then
  begin
    if (ifreg and INT_input[0]) = 0 then IntReq(0);
    ifreg := ifreg or INT_input[0];
  end
  else
    ifreg := ifreg and not INT_input[0];
end {DoPorts};


end.
