{ disassembler Casio uPD1007 }

unit Dis;


interface

  function Mnemonic (kod: word) : string;
  function Arguments (kod: word) : string;


implementation

uses SysUtils, Def;

type

  t_kind = (
    NONE,	{ no arguments }
    DROP1,	{ two-byte instructions, no arguments }
    WRIM16,	{ Word Register, 16-bit immediate value }
    IRIM8,	{ Index Register, 8-bit immediate value }
    CCIM16,	{ Condition code, 16-bit immediate value }
    MEMOIM8,	{ (IR), Immediate byte }
    SMEMOIM8,	{ +/-(IR), Immediate byte }
    XREG,
    YREG,
    REGIM8,	{ Rd,Im8 }
    XARY,
    YARY,
    ARYIM6,
    MEMOREG,	{ +/-(IR),Rs }
    REG,	{ Rd }
    OFFSREG,	{ (IR+/-Ri),Rs or (IR+/-nn),Rs }
    REGMEMO,	{ Rd,(IR)+/- }
    REGOFFS,	{ Rd,(IR+/-Ri) or Rd,(IR+/-nn) }
    MEMOARY,	{ +/-(IR),Rsf..Rsl }
    OFFSARY,	{ (IR+/-Ri),Rsf..Rsl or (IR+/-nn),Rsf..Rsl }
    ARY,	{ Rdf..Rdl }
    ARYMTBM,	{ instructions CMPM/MTBM Rdf..Rdl }
    ARYMEMO,	{ Rdf..Rdl,(IR)+/- }
    ARYOFFS,	{ Rdf..Rdl,(IR+/-Ri) or Rdf..Rdl,(IR+/-nn) }
    STREG,	{ status register, data register }
    STIM8,	{ status register, Im8 }
    REGST,	{ data register, status register }
    REGIF,	{ Rd,IF }
    REGKI,	{ Rd,KI }
    CCINDIR,
    DATA1
  );


  tab = record
    str: string[6];
    kind: t_kind;
  end;


const

  mnem: array[0..511] of tab = (
{ bit 7 of the second byte cleared }
    (	str:'adb';	kind:XREG	),	{ code $00 }
    (	str:'sbb';	kind:XREG	),	{ code $01 }
    (	str:'ad';	kind:XREG	),	{ code $02 }
    (	str:'sb';	kind:XREG	),	{ code $03 }
    (	str:'an';	kind:XREG	),	{ code $04 }
    (	str:'na';	kind:XREG	),	{ code $05 }
    (	str:'or';	kind:XREG	),	{ code $06 }
    (	str:'xr';	kind:XREG	),	{ code $07 }
    (	str:'adb';	kind:YREG	),	{ code $08 }
    (	str:'sbb';	kind:YREG	),	{ code $09 }
    (	str:'ad';	kind:YREG	),	{ code $0A }
    (	str:'sb';	kind:YREG	),	{ code $0B }
    (	str:'an';	kind:YREG	),	{ code $0C }
    (	str:'na';	kind:YREG	),	{ code $0D }
    (	str:'or';	kind:YREG	),	{ code $0E }
    (	str:'xr';	kind:YREG	),	{ code $0F }
    (	str:'adbm';	kind:XARY	),	{ code $10 }
    (	str:'sbbm';	kind:XARY	),	{ code $11 }
    (	str:'adm';	kind:XARY	),	{ code $12 }
    (	str:'sbm';	kind:XARY	),	{ code $13 }
    (	str:'anm';	kind:XARY	),	{ code $14 }
    (	str:'nam';	kind:XARY	),	{ code $15 }
    (	str:'orm';	kind:XARY	),	{ code $16 }
    (	str:'xrm';	kind:XARY	),	{ code $17 }
    (	str:'adbm';	kind:YARY	),	{ code $18 }
    (	str:'sbbm';	kind:YARY	),	{ code $19 }
    (	str:'adm';	kind:YARY	),	{ code $1A }
    (	str:'sbm';	kind:YARY	),	{ code $1B }
    (	str:'anm';	kind:YARY	),	{ code $1C }
    (	str:'nam';	kind:YARY	),	{ code $1D }
    (	str:'orm';	kind:YARY	),	{ code $1E }
    (	str:'xrm';	kind:YARY	),	{ code $1F }
    (	str:'tadb';	kind:XREG	),	{ code $20 }
    (	str:'tsbb';	kind:XREG	),	{ code $21 }
    (	str:'tad';	kind:XREG	),	{ code $22 }
    (	str:'tsb';	kind:XREG	),	{ code $23 }
    (	str:'tan';	kind:XREG	),	{ code $24 }
    (	str:'tna';	kind:XREG	),	{ code $25 }
    (	str:'tor';	kind:XREG	),	{ code $26 }
    (	str:'txr';	kind:XREG	),	{ code $27 }
    (	str:'tadb';	kind:YREG	),	{ code $28 }
    (	str:'tsbb';	kind:YREG	),	{ code $29 }
    (	str:'tad';	kind:YREG	),	{ code $2A }
    (	str:'tsb';	kind:YREG	),	{ code $2B }
    (	str:'tan';	kind:YREG	),	{ code $2C }
    (	str:'tna';	kind:YREG	),	{ code $2D }
    (	str:'tor';	kind:YREG	),	{ code $2E }
    (	str:'txr';	kind:YREG	),	{ code $2F }
    (	str:'tadbm';	kind:XARY	),	{ code $30 }
    (	str:'tsbbm';	kind:XARY	),	{ code $31 }
    (	str:'tadm';	kind:XARY	),	{ code $32 }
    (	str:'tsbm';	kind:XARY	),	{ code $33 }
    (	str:'tanm';	kind:XARY	),	{ code $34 }
    (	str:'tnam';	kind:XARY	),	{ code $35 }
    (	str:'torm';	kind:XARY	),	{ code $36 }
    (	str:'txrm';	kind:XARY	),	{ code $37 }
    (	str:'tadbm';	kind:YARY	),	{ code $38 }
    (	str:'tsbbm';	kind:YARY	),	{ code $39 }
    (	str:'tadm';	kind:YARY	),	{ code $3A }
    (	str:'tsbm';	kind:YARY	),	{ code $3B }
    (	str:'tanm';	kind:YARY	),	{ code $3C }
    (	str:'tnam';	kind:YARY	),	{ code $3D }
    (	str:'torm';	kind:YARY	),	{ code $3E }
    (	str:'txrm';	kind:YARY	),	{ code $3F }
    (	str:'ldw';	kind:WRIM16	),	{ code $40 }
    (	str:'ldw';	kind:WRIM16	),	{ code $41 }
    (	str:'ldw';	kind:WRIM16	),	{ code $42 }
    (	str:'ldw';	kind:WRIM16	),	{ code $43 }
    (	str:'ldw';	kind:WRIM16	),	{ code $44 }
    (	str:'ldw';	kind:WRIM16	),	{ code $45 }
    (	str:'ldw';	kind:WRIM16	),	{ code $46 }
    (	str:'ldw';	kind:WRIM16	),	{ code $47 }
    (	str:'adw';	kind:IRIM8	),	{ code $48 }
    (	str:'adw';	kind:IRIM8	),	{ code $49 }
    (	str:'adw';	kind:IRIM8	),	{ code $4A }
    (	str:'adw';	kind:IRIM8	),	{ code $4B }
    (	str:'sbw';	kind:IRIM8	),	{ code $4C }
    (	str:'sbw';	kind:IRIM8	),	{ code $4D }
    (	str:'sbw';	kind:IRIM8	),	{ code $4E }
    (	str:'sbw';	kind:IRIM8	),	{ code $4F }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $50 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $51 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $52 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $53 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $54 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $55 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $56 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $57 }
    (	str:'rtn';	kind:NONE	),	{ code $58 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $59 }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $5A }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $5B }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $5C }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $5D }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $5E }
    (	str:'ijmp';	kind:CCINDIR	),	{ code $5F }
    (	str:'cal';	kind:CCIM16	),	{ code $60 }
    (	str:'cal';	kind:CCIM16	),	{ code $61 }
    (	str:'cal';	kind:CCIM16	),	{ code $62 }
    (	str:'cal';	kind:CCIM16	),	{ code $63 }
    (	str:'cal';	kind:CCIM16	),	{ code $64 }
    (	str:'cal';	kind:CCIM16	),	{ code $65 }
    (	str:'cal';	kind:CCIM16	),	{ code $66 }
    (	str:'cal';	kind:CCIM16	),	{ code $67 }
    (	str:'cani';	kind:NONE	),	{ code $68 }
    (	str:'cal';	kind:CCIM16	),	{ code $69 }
    (	str:'cal';	kind:CCIM16	),	{ code $6A }
    (	str:'cal';	kind:CCIM16	),	{ code $6B }
    (	str:'cal';	kind:CCIM16	),	{ code $6C }
    (	str:'cal';	kind:CCIM16	),	{ code $6D }
    (	str:'cal';	kind:CCIM16	),	{ code $6E }
    (	str:'cal';	kind:CCIM16	),	{ code $6F }
    (	str:'jmp';	kind:CCIM16	),	{ code $70 }
    (	str:'jmp';	kind:CCIM16	),	{ code $71 }
    (	str:'jmp';	kind:CCIM16	),	{ code $72 }
    (	str:'jmp';	kind:CCIM16	),	{ code $73 }
    (	str:'jmp';	kind:CCIM16	),	{ code $74 }
    (	str:'jmp';	kind:CCIM16	),	{ code $75 }
    (	str:'jmp';	kind:CCIM16	),	{ code $76 }
    (	str:'jmp';	kind:CCIM16	),	{ code $77 }
    (	str:'rti';	kind:NONE	),	{ code $78 }
    (	str:'jmp';	kind:CCIM16	),	{ code $79 }
    (	str:'jmp';	kind:CCIM16	),	{ code $7A }
    (	str:'jmp';	kind:CCIM16	),	{ code $7B }
    (	str:'jmp';	kind:CCIM16	),	{ code $7C }
    (	str:'jmp';	kind:CCIM16	),	{ code $7D }
    (	str:'jmp';	kind:CCIM16	),	{ code $7E }
    (	str:'jmp';	kind:CCIM16	),	{ code $7F }
    (	str:'st';	kind:MEMOREG	),	{ code $80 }
    (	str:'st';	kind:MEMOREG	),	{ code $81 }
    (	str:'st';	kind:MEMOREG	),	{ code $82 }
    (	str:'st';	kind:MEMOREG	),	{ code $83 }
    (	str:'st';	kind:MEMOREG	),	{ code $84 }
    (	str:'st';	kind:MEMOREG	),	{ code $85 }
    (	str:'st';	kind:MEMOREG	),	{ code $86 }
    (	str:'st';	kind:MEMOREG	),	{ code $87 }
    (	str:'st';	kind:MEMOREG	),	{ code $88 }
    (	str:'st';	kind:MEMOREG	),	{ code $89 }
    (	str:'st';	kind:MEMOREG	),	{ code $8A }
    (	str:'st';	kind:MEMOREG	),	{ code $8B }
    (	str:'st';	kind:MEMOREG	),	{ code $8C }
    (	str:'st';	kind:MEMOREG	),	{ code $8D }
    (	str:'st';	kind:MEMOREG	),	{ code $8E }
    (	str:'st';	kind:MEMOREG	),	{ code $8F }
    (	str:'stm';	kind:MEMOARY	),	{ code $90 }
    (	str:'stm';	kind:MEMOARY	),	{ code $91 }
    (	str:'stm';	kind:MEMOARY	),	{ code $92 }
    (	str:'stm';	kind:MEMOARY	),	{ code $93 }
    (	str:'stm';	kind:MEMOARY	),	{ code $94 }
    (	str:'stm';	kind:MEMOARY	),	{ code $95 }
    (	str:'stm';	kind:MEMOARY	),	{ code $96 }
    (	str:'stm';	kind:MEMOARY	),	{ code $97 }
    (	str:'stm';	kind:MEMOARY	),	{ code $98 }
    (	str:'stm';	kind:MEMOARY	),	{ code $99 }
    (	str:'stm';	kind:MEMOARY	),	{ code $9A }
    (	str:'stm';	kind:MEMOARY	),	{ code $9B }
    (	str:'stm';	kind:MEMOARY	),	{ code $9C }
    (	str:'stm';	kind:MEMOARY	),	{ code $9D }
    (	str:'stm';	kind:MEMOARY	),	{ code $9E }
    (	str:'stm';	kind:MEMOARY	),	{ code $9F }
    (	str:'ld';	kind:REGMEMO	),	{ code $A0 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A1 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A2 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A3 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A4 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A5 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A6 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A7 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A8 }
    (	str:'ld';	kind:REGMEMO	),	{ code $A9 }
    (	str:'ld';	kind:REGMEMO	),	{ code $AA }
    (	str:'ld';	kind:REGMEMO	),	{ code $AB }
    (	str:'ld';	kind:REGMEMO	),	{ code $AC }
    (	str:'ld';	kind:REGMEMO	),	{ code $AD }
    (	str:'ld';	kind:REGMEMO	),	{ code $AE }
    (	str:'ld';	kind:REGMEMO	),	{ code $AF }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B0 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B1 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B2 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B3 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B4 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B5 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B6 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B7 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B8 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $B9 }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $BA }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $BB }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $BC }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $BD }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $BE }
    (	str:'ldm';	kind:ARYMEMO	),	{ code $BF }
    (	str:'pst';	kind:STIM8	),	{ code $C0 }
    (	str:'pst';	kind:STIM8	),	{ code $C1 }
    (	str:'pst';	kind:STIM8	),	{ code $C2 }
    (	str:'pst';	kind:STIM8	),	{ code $C3 }
    (	str:'pst';	kind:STREG	),	{ code $C4 }
    (	str:'pst';	kind:STREG	),	{ code $C5 }
    (	str:'pst';	kind:STREG	),	{ code $C6 }
    (	str:'pst';	kind:STREG	),	{ code $C7 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $C8 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $C9 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $CA }
    (	str:'st';	kind:SMEMOIM8	),	{ code $CB }
    (	str:'pst';	kind:STREG	),	{ code $CC }
    (	str:'pst';	kind:STREG	),	{ code $CD }
    (	str:'pst';	kind:STREG	),	{ code $CE }
    (	str:'pst';	kind:STREG	),	{ code $CF }
    (	str:'st';	kind:MEMOIM8	),	{ code $D0 }
    (	str:'st';	kind:MEMOIM8	),	{ code $D1 }
    (	str:'st';	kind:MEMOIM8	),	{ code $D2 }
    (	str:'st';	kind:MEMOIM8	),	{ code $D3 }
    (	str:'gst';	kind:REGST	),	{ code $D4 }
    (	str:'gst';	kind:REGST	),	{ code $D5 }
    (	str:'gst';	kind:REGST	),	{ code $D6 }
    (	str:'gst';	kind:REGST	),	{ code $D7 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $D8 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $D9 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $DA }
    (	str:'st';	kind:SMEMOIM8	),	{ code $DB }
    (	str:'gst';	kind:REGST	),	{ code $DC }
    (	str:'gst';	kind:REGST	),	{ code $DD }
    (	str:'gst';	kind:REGST	),	{ code $DE }
    (	str:'gst';	kind:REGST	),	{ code $DF }
    (	str:'cmp';	kind:REG	),	{ code $E0 }
    (	str:'cmpm';	kind:ARY	),	{ code $E1 }
    (	str:'bup';	kind:NONE	),	{ code $E2 }
    (	str:'byum';	kind:ARY	),	{ code $E3 }
    (	str:'cmpb';	kind:REG	),	{ code $E4 }
    (	str:'dium';	kind:ARY	),	{ code $E5 }
    (	str:'rou';	kind:REG	),	{ code $E6 }
    (	str:'roum';	kind:ARY	),	{ code $E7 }
    (	str:'cmp';	kind:REG	),	{ code $E8 }
    (	str:'cmpm';	kind:ARY	),	{ code $E9 }
    (	str:'[EA00]';	kind:YREG	),	{ code $EA }
    (	str:'byum';	kind:ARY	),	{ code $EB }
    (	str:'cmpb';	kind:REG	),	{ code $EC }
    (	str:'dium';	kind:ARY	),	{ code $ED }
    (	str:'rou';	kind:REG	),	{ code $EE }
    (	str:'roum';	kind:ARY	),	{ code $EF }
    (	str:'inv';	kind:REG	),	{ code $F0 }
    (	str:'invm';	kind:ARY	),	{ code $F1 }
    (	str:'bdn';	kind:NONE	),	{ code $F2 }
    (	str:'bydm';	kind:ARY	),	{ code $F3 }
    (	str:'cmpbm';	kind:ARYMTBM	),	{ code $F4 }
    (	str:'didm';	kind:ARY	),	{ code $F5 }
    (	str:'rod';	kind:REG	),	{ code $F6 }
    (	str:'rodm';	kind:ARY	),	{ code $F7 }
    (	str:'inv';	kind:REG	),	{ code $F8 }
    (	str:'invm';	kind:ARY	),	{ code $F9 }
    (	str:'nop';	kind:NONE	),	{ code $FA }
    (	str:'bydm';	kind:ARY	),	{ code $FB }
    (	str:'cmpbm';	kind:ARYMTBM	),	{ code $FC }
    (	str:'didm';	kind:ARY	),	{ code $FD }
    (	str:'rod';	kind:REG	),	{ code $FE }
    (	str:'rodm';	kind:ARY	),	{ code $FF }
{ bit 7 of the second byte set }
    (	str:'adb';	kind:REGIM8	),	{ code $00 }
    (	str:'sbb';	kind:REGIM8	),	{ code $01 }
    (	str:'ad';	kind:REGIM8	),	{ code $02 }
    (	str:'sb';	kind:REGIM8	),	{ code $03 }
    (	str:'an';	kind:REGIM8	),	{ code $04 }
    (	str:'na';	kind:REGIM8	),	{ code $05 }
    (	str:'or';	kind:REGIM8	),	{ code $06 }
    (	str:'xr';	kind:REGIM8	),	{ code $07 }
    (	str:'adb';	kind:REGIM8	),	{ code $08 }
    (	str:'sbb';	kind:REGIM8	),	{ code $09 }
    (	str:'ad';	kind:REGIM8	),	{ code $0A }
    (	str:'sb';	kind:REGIM8	),	{ code $0B }
    (	str:'an';	kind:REGIM8	),	{ code $0C }
    (	str:'na';	kind:REGIM8	),	{ code $0D }
    (	str:'or';	kind:REGIM8	),	{ code $0E }
    (	str:'xr';	kind:REGIM8	),	{ code $0F }
    (	str:'adbm';	kind:ARYIM6	),	{ code $10 }
    (	str:'sbbm';	kind:ARYIM6	),	{ code $11 }
    (	str:'adm';	kind:ARYIM6	),	{ code $12 }
    (	str:'sbm';	kind:ARYIM6	),	{ code $13 }
    (	str:'swp';	kind:XREG	),	{ code $14 }
    (	str:'swpm';	kind:XARY	),	{ code $15 }
    (	str:'xcls';	kind:XREG	),	{ code $16 }
    (	str:'xclsm';	kind:XARY	),	{ code $17 }
    (	str:'adbm';	kind:ARYIM6	),	{ code $18 }
    (	str:'sbbm';	kind:ARYIM6	),	{ code $19 }
    (	str:'adm';	kind:ARYIM6	),	{ code $1A }
    (	str:'sbm';	kind:ARYIM6	),	{ code $1B }
    (	str:'xc';	kind:XREG	),	{ code $1C }
    (	str:'xcm';	kind:XARY	),	{ code $1D }
    (	str:'xchs';	kind:XREG	),	{ code $1E }
    (	str:'xchsm';	kind:XARY	),	{ code $1F }
    (	str:'tadb';	kind:REGIM8	),	{ code $20 }
    (	str:'tsbb';	kind:REGIM8	),	{ code $21 }
    (	str:'tad';	kind:REGIM8	),	{ code $22 }
    (	str:'tsb';	kind:REGIM8	),	{ code $23 }
    (	str:'tan';	kind:REGIM8	),	{ code $24 }
    (	str:'tna';	kind:REGIM8	),	{ code $25 }
    (	str:'tor';	kind:REGIM8	),	{ code $26 }
    (	str:'txr';	kind:REGIM8	),	{ code $27 }
    (	str:'tadb';	kind:REGIM8	),	{ code $28 }
    (	str:'tsbb';	kind:REGIM8	),	{ code $29 }
    (	str:'tad';	kind:REGIM8	),	{ code $2A }
    (	str:'tsb';	kind:REGIM8	),	{ code $2B }
    (	str:'tan';	kind:REGIM8	),	{ code $2C }
    (	str:'tna';	kind:REGIM8	),	{ code $2D }
    (	str:'tor';	kind:REGIM8	),	{ code $2E }
    (	str:'txr';	kind:REGIM8	),	{ code $2F }
    (	str:'tadbm';	kind:ARYIM6	),	{ code $30 }
    (	str:'tsbbm';	kind:ARYIM6	),	{ code $31 }
    (	str:'tadm';	kind:ARYIM6	),	{ code $32 }
    (	str:'tsbm';	kind:ARYIM6	),	{ code $33 }
    (	str:'sup';	kind:REG	),	{ code $34 }
    (	str:'[3580]';	kind:DATA1	),	{ code $35 }
    (	str:'[3680]';	kind:DATA1	),	{ code $36 }
    (	str:'[3780]';	kind:DATA1	),	{ code $37 }
    (	str:'tadbm';	kind:ARYIM6	),	{ code $38 }
    (	str:'tsbbm';	kind:ARYIM6	),	{ code $39 }
    (	str:'tadm';	kind:ARYIM6	),	{ code $3A }
    (	str:'tsbm';	kind:ARYIM6	),	{ code $3B }
    (	str:'sdn';	kind:REG	),	{ code $3C }
    (	str:'[3D80]';	kind:DATA1	),	{ code $3D }
    (	str:'[3E80]';	kind:DATA1	),	{ code $3E }
    (	str:'[3F80]';	kind:DATA1	),	{ code $3F }
    (	str:'ldw';	kind:WRIM16	),	{ code $40 }
    (	str:'ldw';	kind:WRIM16	),	{ code $41 }
    (	str:'ldw';	kind:WRIM16	),	{ code $42 }
    (	str:'ldw';	kind:WRIM16	),	{ code $43 }
    (	str:'ldw';	kind:WRIM16	),	{ code $44 }
    (	str:'ldw';	kind:WRIM16	),	{ code $45 }
    (	str:'ldw';	kind:WRIM16	),	{ code $46 }
    (	str:'ldw';	kind:WRIM16	),	{ code $47 }
    (	str:'adw';	kind:IRIM8	),	{ code $48 }
    (	str:'adw';	kind:IRIM8	),	{ code $49 }
    (	str:'adw';	kind:IRIM8	),	{ code $4A }
    (	str:'adw';	kind:IRIM8	),	{ code $4B }
    (	str:'sbw';	kind:IRIM8	),	{ code $4C }
    (	str:'sbw';	kind:IRIM8	),	{ code $4D }
    (	str:'sbw';	kind:IRIM8	),	{ code $4E }
    (	str:'sbw';	kind:IRIM8	),	{ code $4F }
    (	str:'off';	kind:DROP1	),	{ code $50 }
    (	str:'gst';	kind:REGKI	),	{ code $51 }
    (	str:'bit';	kind:REGIM8	),	{ code $52 }
    (	str:'bit';	kind:XREG	),	{ code $53 }
    (	str:'ld';	kind:XREG	),	{ code $54 }
    (	str:'ldm';	kind:XARY	),	{ code $55 }
    (	str:'ld';	kind:REGIM8	),	{ code $56 }
    (	str:'ldm';	kind:ARYIM6	),	{ code $57 }
    (	str:'rtn';	kind:NONE	),	{ code $58 }
    (	str:'gst';	kind:REGKI	),	{ code $59 }
    (	str:'bit';	kind:REGIM8	),	{ code $5A }
    (	str:'bit';	kind:YREG	),	{ code $5B }
    (	str:'ld';	kind:YREG	),	{ code $5C }
    (	str:'ldm';	kind:YARY	),	{ code $5D }
    (	str:'ld';	kind:REGIM8	),	{ code $5E }
    (	str:'ldm';	kind:ARYIM6	),	{ code $5F }
    (	str:'cal';	kind:CCIM16	),	{ code $60 }
    (	str:'cal';	kind:CCIM16	),	{ code $61 }
    (	str:'cal';	kind:CCIM16	),	{ code $62 }
    (	str:'cal';	kind:CCIM16	),	{ code $63 }
    (	str:'cal';	kind:CCIM16	),	{ code $64 }
    (	str:'cal';	kind:CCIM16	),	{ code $65 }
    (	str:'cal';	kind:CCIM16	),	{ code $66 }
    (	str:'cal';	kind:CCIM16	),	{ code $67 }
    (	str:'cani';	kind:NONE	),	{ code $68 }
    (	str:'cal';	kind:CCIM16	),	{ code $69 }
    (	str:'cal';	kind:CCIM16	),	{ code $6A }
    (	str:'cal';	kind:CCIM16	),	{ code $6B }
    (	str:'cal';	kind:CCIM16	),	{ code $6C }
    (	str:'cal';	kind:CCIM16	),	{ code $6D }
    (	str:'cal';	kind:CCIM16	),	{ code $6E }
    (	str:'cal';	kind:CCIM16	),	{ code $6F }
    (	str:'jmp';	kind:CCIM16	),	{ code $70 }
    (	str:'jmp';	kind:CCIM16	),	{ code $71 }
    (	str:'jmp';	kind:CCIM16	),	{ code $72 }
    (	str:'jmp';	kind:CCIM16	),	{ code $73 }
    (	str:'jmp';	kind:CCIM16	),	{ code $74 }
    (	str:'jmp';	kind:CCIM16	),	{ code $75 }
    (	str:'jmp';	kind:CCIM16	),	{ code $76 }
    (	str:'jmp';	kind:CCIM16	),	{ code $77 }
    (	str:'rti';	kind:NONE	),	{ code $78 }
    (	str:'jmp';	kind:CCIM16	),	{ code $79 }
    (	str:'jmp';	kind:CCIM16	),	{ code $7A }
    (	str:'jmp';	kind:CCIM16	),	{ code $7B }
    (	str:'jmp';	kind:CCIM16	),	{ code $7C }
    (	str:'jmp';	kind:CCIM16	),	{ code $7D }
    (	str:'jmp';	kind:CCIM16	),	{ code $7E }
    (	str:'jmp';	kind:CCIM16	),	{ code $7F }
    (	str:'st';	kind:OFFSREG	),	{ code $80 }
    (	str:'st';	kind:OFFSREG	),	{ code $81 }
    (	str:'st';	kind:OFFSREG	),	{ code $82 }
    (	str:'st';	kind:OFFSREG	),	{ code $83 }
    (	str:'st';	kind:OFFSREG	),	{ code $84 }
    (	str:'st';	kind:OFFSREG	),	{ code $85 }
    (	str:'st';	kind:OFFSREG	),	{ code $86 }
    (	str:'st';	kind:OFFSREG	),	{ code $87 }
    (	str:'st';	kind:OFFSREG	),	{ code $88 }
    (	str:'st';	kind:OFFSREG	),	{ code $89 }
    (	str:'st';	kind:OFFSREG	),	{ code $8A }
    (	str:'st';	kind:OFFSREG	),	{ code $8B }
    (	str:'st';	kind:OFFSREG	),	{ code $8C }
    (	str:'st';	kind:OFFSREG	),	{ code $8D }
    (	str:'st';	kind:OFFSREG	),	{ code $8E }
    (	str:'st';	kind:OFFSREG	),	{ code $8F }
    (	str:'stm';	kind:OFFSARY	),	{ code $90 }
    (	str:'stm';	kind:OFFSARY	),	{ code $91 }
    (	str:'stm';	kind:OFFSARY	),	{ code $92 }
    (	str:'stm';	kind:OFFSARY	),	{ code $93 }
    (	str:'stm';	kind:OFFSARY	),	{ code $94 }
    (	str:'stm';	kind:OFFSARY	),	{ code $95 }
    (	str:'stm';	kind:OFFSARY	),	{ code $96 }
    (	str:'stm';	kind:OFFSARY	),	{ code $97 }
    (	str:'stm';	kind:OFFSARY	),	{ code $98 }
    (	str:'stm';	kind:OFFSARY	),	{ code $99 }
    (	str:'stm';	kind:OFFSARY	),	{ code $9A }
    (	str:'stm';	kind:OFFSARY	),	{ code $9B }
    (	str:'stm';	kind:OFFSARY	),	{ code $9C }
    (	str:'stm';	kind:OFFSARY	),	{ code $9D }
    (	str:'stm';	kind:OFFSARY	),	{ code $9E }
    (	str:'stm';	kind:OFFSARY	),	{ code $9F }
    (	str:'ld';	kind:REGOFFS	),	{ code $A0 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A1 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A2 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A3 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A4 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A5 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A6 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A7 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A8 }
    (	str:'ld';	kind:REGOFFS	),	{ code $A9 }
    (	str:'ld';	kind:REGOFFS	),	{ code $AA }
    (	str:'ld';	kind:REGOFFS	),	{ code $AB }
    (	str:'ld';	kind:REGOFFS	),	{ code $AC }
    (	str:'ld';	kind:REGOFFS	),	{ code $AD }
    (	str:'ld';	kind:REGOFFS	),	{ code $AE }
    (	str:'ld';	kind:REGOFFS	),	{ code $AF }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B0 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B1 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B2 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B3 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B4 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B5 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B6 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B7 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B8 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $B9 }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $BA }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $BB }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $BC }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $BD }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $BE }
    (	str:'ldm';	kind:ARYOFFS	),	{ code $BF }
    (	str:'pst';	kind:STIM8	),	{ code $C0 }
    (	str:'pst';	kind:STIM8	),	{ code $C1 }
    (	str:'pst';	kind:STIM8	),	{ code $C2 }
    (	str:'pst';	kind:STIM8	),	{ code $C3 }
    (	str:'ldle';	kind:REGIM8	),	{ code $C4 }
    (	str:'ldlo';	kind:REGIM8	),	{ code $C5 }
    (	str:'stle';	kind:REGIM8	),	{ code $C6 }
    (	str:'stlo';	kind:REGIM8	),	{ code $C7 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $C8 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $C9 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $CA }
    (	str:'st';	kind:SMEMOIM8	),	{ code $CB }
    (	str:'ldle';	kind:REGIM8	),	{ code $CC }
    (	str:'ldlo';	kind:REGIM8	),	{ code $CD }
    (	str:'stle';	kind:REGIM8	),	{ code $CE }
    (	str:'stlo';	kind:REGIM8	),	{ code $CF }
    (	str:'st';	kind:MEMOIM8	),	{ code $D0 }
    (	str:'st';	kind:MEMOIM8	),	{ code $D1 }
    (	str:'st';	kind:MEMOIM8	),	{ code $D2 }
    (	str:'st';	kind:MEMOIM8	),	{ code $D3 }
    (	str:'ldlem';	kind:ARYIM6	),	{ code $D4 }
    (	str:'ldlom';	kind:ARYIM6	),	{ code $D5 }
    (	str:'stlem';	kind:ARYIM6	),	{ code $D6 }
    (	str:'stlom';	kind:ARYIM6	),	{ code $D7 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $D8 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $D9 }
    (	str:'st';	kind:SMEMOIM8	),	{ code $DA }
    (	str:'st';	kind:SMEMOIM8	),	{ code $DB }
    (	str:'ldlem';	kind:ARYIM6	),	{ code $DC }
    (	str:'ldlom';	kind:ARYIM6	),	{ code $DD }
    (	str:'stlem';	kind:ARYIM6	),	{ code $DE }
    (	str:'stlom';	kind:ARYIM6	),	{ code $DF }
    (	str:'swp';	kind:REG	),	{ code $E0 }
    (	str:'[E180]';	kind:DATA1	),	{ code $E1 }
    (	str:'bup';	kind:NONE	),	{ code $E2 }
    (	str:'byu';	kind:REG	),	{ code $E3 }
    (	str:'biu';	kind:REG	),	{ code $E4 }
    (	str:'diu';	kind:REG	),	{ code $E5 }
    (	str:'mtb';	kind:REG	),	{ code $E6 }
    (	str:'trp';	kind:DROP1	),	{ code $E7 }
    (	str:'swp';	kind:REG	),	{ code $E8 }
    (	str:'[E980]';	kind:DATA1	),	{ code $E9 }
    (	str:'[EA80]';	kind:YREG	),	{ code $EA }
    (	str:'byu';	kind:REG	),	{ code $EB }
    (	str:'biu';	kind:REG	),	{ code $EC }
    (	str:'diu';	kind:REG	),	{ code $ED }
    (	str:'mtb';	kind:REG	),	{ code $EE }
    (	str:'trp';	kind:DROP1	),	{ code $EF }
    (	str:'bnusm';	kind:ARY	),	{ code $F0 }
    (	str:'gst';	kind:REGIF	),	{ code $F1 }
    (	str:'bdn';	kind:NONE	),	{ code $F2 }
    (	str:'byd';	kind:REG	),	{ code $F3 }
    (	str:'bium';	kind:ARY	),	{ code $F4 }
    (	str:'did';	kind:REG	),	{ code $F5 }
    (	str:'mtbm';	kind:ARYMTBM	),	{ code $F6 }
    (	str:'trp';	kind:DROP1	),	{ code $F7 }
    (	str:'bnusm';	kind:ARY	),	{ code $F8 }
    (	str:'gst';	kind:REGIF	),	{ code $F9 }
    (	str:'nop';	kind:NONE	),	{ code $FA }
    (	str:'byd';	kind:REG	),	{ code $FB }
    (	str:'bium';	kind:ARY	),	{ code $FC }
    (	str:'did';	kind:REG	),	{ code $FD }
    (	str:'mtbm';	kind:ARYMTBM	),	{ code $FE }
    (	str:'trp';	kind:DROP1	)	{ code $FF }
  );


  ir_tab: array[0..3] of string[2] = (
	'ix', 'iy', 'iz', 'sp' );

  wr_tab: array[0..7] of string[2] = (
	'ix', 'iy', 'iz', 'v3', 'v2', 'v1', 'v0', 'sp' );

  cc_tab: array[0..15] of string[4] = (
	'', 'k,', 'lz,',  'uz,',  'nz,', 'v,',  'h,',  'c,',
	'*,', 'nk,', 'nlz,', 'nuz,', 'z,',  'nv,', 'nh,', 'nc,' );

  st_tab: array[0..7] of string[2] = (
	'ko', 'if', 'as', 'ie',		{ immediate value }
	'ko', 'f', 'as', 'ie' );	{ register }



function CaHexB (x: byte) : string;
begin
  CaHexB := '&H' + IntToHex(x,2);
end {CaHexB};


function Reg1 (x: byte) : byte;
begin
  Reg1 := ((x shr 1) and $38) or (x and $07);
end {Reg1};


function Reg2 (y: byte) : byte;
begin
  Reg2 := ((y shr 5) and $07) or ((y shl 3) and $38);
end {Reg2};


function Rl1 (x, y: byte) : byte;
begin
  Rl1 := ((x shr 1) and $38) or ((y shr 5) and $07);
end {Rl1};


function Rf2 (x, y: byte) : byte;
begin
  Rf2 := (x and $07) or ((y shl 3) and $38);
end {Rf2};


function Im6 (x, y: byte) : byte;
begin
  Im6 := (y and $1F) or ((not x shl 2) and $20);
end {Im6};


{ returns the mnemonic }
function Mnemonic (kod: word) : string;
begin
  Mnemonic := mnem[kod].str;
end {Mnemonic};


{ returns the arguments }
function Arguments (kod: word) : string;
var
  x, y, z : byte;
  sign, name : char;
begin
  if (kod and $04) = 0 then sign := '+' else sign := '-';
  z := byte(not kod shl 3) and $40;

  case mnem[kod].kind of

    NONE:
      Result := '';

    DROP1:
      begin
        FetchByte;
        Result := '';
      end;

    CCIM16:
      begin
        x := FetchByte;
        y := FetchByte;
        Result := cc_tab[kod and $0F] + CaHexB(x) + IntToHex(y,2);
      end;

    WRIM16:
      begin
        x := FetchByte;
        y := FetchByte;
        Result := wr_tab[kod and $07] + ',' + CaHexB(x) + IntToHex(y,2);
      end;

    IRIM8:
      Result := ir_tab[kod and $03] + ',' + CaHexB(FetchByte);

    SMEMOIM8:
      begin
        if (kod and $10) = 0 then sign := '+' else sign := '-';
        Result := sign + '(' + ir_tab[kod and $03] + '),' + CaHexB(FetchByte);
      end;

    MEMOIM8:
      Result := '(' + ir_tab[kod and $03] + '),' + CaHexB(FetchByte);

    XREG:
      begin
        x := FetchByte;
        y := FetchByte;
        if (y and $10) <> 0 then name := 'i' else name := 'r';
        Result := name + IntToStr(Reg2(y) or $40) + ',r' + IntToStr(Reg1(x));
      end;

    YREG:
      begin
        x := FetchByte;
        y := FetchByte;
        if (y and $10) <> 0 then name := 'i' else name := 'r';
        Result := 'r' + IntToStr(Reg1(x)) + ',' +
		name + IntToStr(Reg2(y) or $40);
      end;

    REGIM8:
      begin
        x := FetchByte;
        y := FetchByte;
        Result := 'r' + IntToStr(Reg1(x) or z) + ',' + CaHexB(y);
      end;

    XARY:
      begin
        x := FetchByte;
        y := FetchByte;
        if (y and $10) <> 0 then name := 'i' else name := 'r';
        Result := name + IntToStr(Rf2(x,y) or $40) + '..' +
		name + IntToStr(Reg2(y) or $40) +
		',r' + IntToStr(Reg1(x)) + '..r' + IntToStr(Rl1(x,y));
      end;

    YARY:
      begin
        x := FetchByte;
        y := FetchByte;
        if (y and $10) <> 0 then name := 'i' else name := 'r';
        Result := 'r' + IntToStr(Reg1(x)) + '..r' + IntToStr(Rl1(x,y)) +
		',' + name + IntToStr(Rf2(x,y) or $40) +
		'..' + name + IntToStr(Reg2(y) or $40);
      end;

    ARYIM6:
      begin
        x := FetchByte;
        y := FetchByte;
        Result := 'r' + IntToStr(Reg1(x) or z) +
		'..r' + IntToStr(Rl1(x,y) or z) + ',' + CaHexB(Im6(x,y));
      end;

    MEMOREG:
      Result := sign + '(' + ir_tab[kod and $03] + '),r' +
		IntToStr(Reg1(FetchByte) or z);

    REG:
      Result := 'r' + IntToStr(Reg1(FetchByte) or z);

    OFFSREG:
      begin
        x := FetchByte;
        y := FetchByte;
        if (y and $10) <> 0 then name := 'i' else name := 'r';
        Result := '(' + ir_tab[kod and $03] + sign;
        if (kod and $08) = 0 then
          Result := Result + 'r' + IntToStr(Reg1(x)) + '),' +
		name + IntToStr(Reg2(y) or $40)
        else
          Result := Result + CaHexB(y) + '),r' + IntToStr(Reg1(x));
      end;

    REGMEMO:
      begin
        x := FetchByte;
        Result := 'r' + IntToStr(Reg1(x) or z) +
		',(' + ir_tab[kod and $03] + ')' + sign;
      end;

    REGOFFS:
      begin
        x := FetchByte;
        y := FetchByte;
        if (y and $10) <> 0 then name := 'i' else name := 'r';
        if (kod and $08) = 0 then
          Result := name + IntToStr(Reg2(y) or $40) + ',(' +
		ir_tab[kod and $03] + sign + 'r' + IntToStr(Reg1(x)) + ')'
        else
          Result := 'r' + IntToStr(Reg1(x)) + ',(' +
		ir_tab[kod and $03] + sign + CaHexB(y) + ')';
      end;

    MEMOARY:
      begin
        x := FetchByte;
        y := FetchByte;
        Result := sign + '(' + ir_tab[kod and $03] + '),r' +
		IntToStr(Reg1(x) or z) + '..r' + IntToStr(Rl1(x,y) or z);
      end;

    OFFSARY:
      begin
        x := FetchByte;
        y := FetchByte;
  { indirect is illegal }
        if (y and $10) <> 0 then name := '?' else name := 'r';
        Result := '(' + ir_tab[kod and $03] + sign;
        if (kod and $08) = 0 then
          Result := Result + 'r' + IntToStr(Reg1(x)) + '),' +
		name + IntToStr(Rf2(x,y) or $40) + '..' +
		name + IntToStr(Reg2(y) or $40)
        else
          Result := Result + CaHexB(Im6(x,y)) + '),r' +
		IntToStr(Reg1(x)) + '..r' + IntToStr(Rl1(x,y));
      end;

    ARY:
      begin
        x := FetchByte;
        y := FetchByte;
        if (y and $10) <> 0 then name := 'i' else name := 'r';
        if (kod and $08) = 0 then
          Result := name + IntToStr(Rf2(x,y) or $40) + '..' +
		name + IntToStr(Reg2(y) or $40)
        else
          Result := 'r' + IntToStr(Reg1(x)) + '..r' + IntToStr(Rl1(x,y));
      end;

    ARYMTBM:
      begin
        x := FetchByte;
        y := FetchByte;
        Result := 'r' + IntToStr(Reg1(x) or z) +
		'..r' + IntToStr(Rl1(x,y) or z);
      end;

    ARYMEMO:
      begin
        x := FetchByte;
        y := FetchByte;
        Result := 'r' + IntToStr(Reg1(x) or z) +
		'..r' + IntToStr(Rl1(x,y) or z) +
		',(' + ir_tab[kod and $03] + ')' + sign;
      end;

    ARYOFFS:
      begin
        x := FetchByte;
        y := FetchByte;
  { indirect is illegal }
        if (y and $10) <> 0 then name := '?' else name := 'r';
        if (kod and $08) = 0 then
          Result := name + IntToStr(Rf2(x,y) or $40) + '..' +
		name + IntToStr(Reg2(y) or $40) + ',(' + ir_tab[kod and $03] +
		sign + 'r' + IntToStr(Reg1(x)) + ')'
        else
          Result := 'r' + IntToStr(Reg1(x)) + '..r' + IntToStr(Rl1(x,y)) +
		',(' + ir_tab[kod and $03] + sign + CaHexB(Im6(x,y)) + ')';
      end;

    STREG:
      Result := st_tab[kod and $07] + ',r' + IntToStr(Reg1(FetchByte) or z);

    STIM8:
      Result := st_tab[kod and $07] + ',' + CaHexB(FetchByte);

    REGST:
      Result := 'r' + IntToStr(Reg1(FetchByte) or z) + ',' +
		st_tab[kod and $07];

    REGIF:
      Result := 'r' + IntToStr(Reg1(FetchByte) or z) + ',if';

    REGKI:
      Result := 'r' + IntToStr(Reg1(FetchByte) or z) + ',ki';

    CCINDIR:
      begin
        Result := cc_tab[kod and $0F];
        x := FetchByte;
        y := FetchByte;
        if (y and $10) <> 0 then name := 'i' else name := 'r';
        if (Reg1(x) = Reg2(y)) and ((y and $1F) = $07) and (name = 'r') then
          Result := Result + wr_tab[x and $07]
        else
          Result := Result + name + IntToStr(Reg2(y) or $40) +
		',r' + IntToStr(Reg1(x));
      end;

    DATA1:
      Result := '[' + IntToHex(FetchByte,2) + ']';

  else
    Result := '';

  end {case};
end {Arguments};


end.
