{ NM-320 instruction execution }

unit Exec;


interface

  var
    regbank: byte;	{ $00 when RB0, $40 when RB1 }
    regstep: byte;	{ determines the order of processed registers
			  in a register array,
			  +1 if the register index is incremented (default),
			  -1 if the register index is decremented } 


  function Wreg (x, y: byte) : word;
  function PreIncw (x, y: byte) : word;
  function PreDecw (x, y: byte) : word;
  function PostIncw (x, y: byte) : word;
  function PostDecw (x, y: byte) : word;
  procedure PlusOffset (x: byte);
  procedure MinusOffset (x: byte);
  procedure UnReg (op2: pointer);
  procedure RotReg (op2: pointer);
  procedure UnAry (op2: pointer);
  procedure RotAry (op2: pointer);
  procedure Mtbm (op2: pointer);
  procedure Xreg (op2: pointer);
  procedure Yreg (op2: pointer);
  procedure TXreg (op2: pointer);
  procedure TYreg (op2: pointer);
  procedure ExchReg (op2: pointer);
  procedure RegIm8 (op2: pointer);
  procedure TRegIm8 (op2: pointer);
  procedure Xary (op2: pointer);
  procedure Yary (op2: pointer);
  procedure TXary (op2: pointer);
  procedure TYary (op2: pointer);
  procedure ExchAry (op2: pointer);
  procedure AryIm6 (op2: pointer);
  procedure TAryIm6 (op2: pointer);
  procedure Ldw (op2: pointer);
  procedure AdwSbw (op2: pointer);
  procedure Cond (op2: pointer);
  procedure NotCond (op2: pointer);
  procedure KeyCond (op2: pointer);
  procedure NotKeyCond (op2: pointer);
  procedure Jump (x1, x2: byte);
  procedure Call (x1, x2: byte);
  procedure Trap (op2: pointer);
  procedure Ijmp (x1, x2: byte);
  procedure Rtn (op2: pointer);
  procedure Cani (op2: pointer);
  procedure Rti (op2: pointer);
  procedure Nop (op2: pointer);
  procedure BlockCopy (op2: pointer);
  procedure BlockSearch (op2: pointer);
  procedure StMemoReg (op2: pointer);
  procedure StMemoIm8 (op2: pointer);
  procedure StmMemoAry (op2: pointer);
  procedure StImOffsReg (op2: pointer);
  procedure StRegOffsReg (op2: pointer);
  procedure StmImOffsAry (op2: pointer);
  procedure StmRegOffsAry (op2: pointer);
  procedure LdRegMemo (op2: pointer);
  procedure LdmAryMemo (op2: pointer);
  procedure LdRegImOffs (op2: pointer);
  procedure LdRegRegOffs (op2: pointer);
  procedure LdmAryImOffs (op2: pointer);
  procedure LdmAryRegOffs (op2: pointer);
  procedure PstIm8 (op2: pointer);
  procedure PstReg (op2: pointer);
  procedure Gst (op2: pointer);
  procedure Off (op2: pointer);

  function OpAd (x, y: byte) : byte;
  function OpSb (x, y: byte) : byte;
  function OpAdb (x, y: byte) : byte;
  function OpSbb (x, y: byte) : byte;
  function OpAn (x, y: byte) : byte;
  function OpBit (x, y: byte) : byte;
  function OpXr (x, y: byte) : byte;
  function OpNa (x, y: byte) : byte;
  function OpOr (x, y: byte) : byte;
  function OpLd (x, y: byte) : byte;

  function OpRod (var x: byte; y: byte) : byte;
  function OpRou (var x: byte; y: byte) : byte;
  function OpMtb (var x: byte; y: byte) : byte;
  function OpInv (var x: byte; y: byte) : byte;
  function OpCmp (var x: byte; y: byte) : byte;
  function OpCmpb (var x: byte; y: byte) : byte;
  function OpSwp1 (var x: byte; y: byte) : byte;
  function OpDiu (var x: byte; y: byte) : byte;
  function OpDid (var x: byte; y: byte) : byte;
  function OpByu (var x: byte; y: byte) : byte;
  function OpByd (var x: byte; y: byte) : byte;
  function OpBnus (var x: byte; y: byte) : byte;

  procedure OpXc (var x, y: byte);
  procedure OpXcls (var x, y: byte);
  procedure OpXchs (var x, y: byte);
  procedure OpSwp2 (var x, y: byte);

  procedure OpKo (x: byte);
  procedure OpIf (x: byte);
  procedure OpAs (x: byte);
  procedure OpIe (x: byte);
  procedure OpFl (x: byte);

  procedure Ldle (op2: pointer);
  procedure Ldlo (op2: pointer);
  procedure Stle (op2: pointer);
  procedure Stlo (op2: pointer);
  procedure Ldlem (op2: pointer);
  procedure Ldlom (op2: pointer);
  procedure Stlem (op2: pointer);
  procedure Stlom (op2: pointer);


implementation

  uses Def, Lcd, Keyboard;


  type

    Proc3 = procedure (x, y: byte);
    Proc4 = procedure (x: byte);
    Proc5 = procedure (var x, y: byte);
    Func2 = function (x: byte) : byte;
    Func3 = function (x, y: byte) : byte;
    Func4 = function (x, y: byte) : word;
    Func5 = function (var x: byte; y: byte) : byte;


  const

    IX = 0;
    IY = 1;
    IZ = 2;
    V3 = 3;
    V2 = 4;
    V1 = 5;
    V0 = 6;
    SP = 7;

    cc: array[0..7] of byte = (
	$00, $00, LZ_bit, UZ_bit, NZ_bit, V_bit, H_bit, C_bit );

    NONE_REG = 255;	{ index of a non-existing general purpose register }

  var

    ea: word;		{ temporary pointer used in the indexed addressing
			  mode }

function Reg1 (x: byte) : byte;
begin
  Reg1 := ((x shr 1) and $38) or (x and $07);
end {Reg1};


{ expects the third opcode byte (specification of a general purpose register
  in the bank 1, direct or indirect),
  returns the address (index) of the general purpose register }
function Reg2 (x: byte) : byte;
begin
  if (x and $10) = 0 then	{ directly specified register }
    x := (x shr 5) or ((x shl 3) and $38)
  else if (asreg and $01) = 0 then
    x := (asreg shr 1) and $3F
  else
    x := ((asreg shr 1) and $07) or ((x shl 3) and $38);
  Reg2 := x or $40;
end {Reg2};


{ expects the second and third opcode byte (specification of a general purpose
  register in the bank 1, direct or indirect),
  returns the first register of an array in the bank 1 }
function Reg3 (x, y: byte) : byte;
begin
  if (y and $10) = 0 then	{ directly specified array }
    x := (x and $07) or ((y shl 3) and $38)
  else if (asreg and $01) = 0 then
    x := (asreg shr 1) and $3F
  else
    x := ((asreg shr 1) and $07) or ((y shl 3) and $38);
  Reg3 := x or $40;
end {Reg3};


{ optional last register of an indirectly specified array }
function AsLimit (y: byte) : byte;
begin
  if (y and $10) = 0 then
    AsLimit := NONE_REG	{ directly specified array, no limit }
  else if (asreg and $01) = 0 then
    AsLimit := ((asreg shr 1) and $3F) or $47
  else
    AsLimit := ((asreg shr 4) and $07) or ((y shl 3) and $38) or $40;
end {AsLimit};


function Rl1 (x, y: byte) : byte;
begin
  Rl1 := ((x shr 1) and $38) or (y shr 5);
end {Rl1};


function Im6 (x, y: byte) : byte;
begin
  Im6 := (y and $1F) or ((not x shl 2) and $20);
end {Im6};


function Ireg : byte;
begin
  Result := opcode[0] and $03;
  if Result > IZ then Result := SP;
end {Ireg};


{ next processed register in an array }
procedure NextReg (var x: byte);
begin
  x := (x and $F8) or ((x + regstep) and $07);
end {NextReg};


{ returns the contents of a 16-bit register 'x',
  'y' is a dummy variable }
function Wreg (x, y: byte) : word;
begin
  Wreg := (word(mr[x or $78]) shl 8) or word(mr[x or $38]);
end {Wreg};


{ return the contents of the 16-bit register 'x',
  then add 8-bit value 'y' to this 16-bit register }
function PostIncw (x, y: byte) : word;
var
  z: word;
begin
  Result := Wreg(x,0);
  z := Result + word(y);
  mr[x or $38] := Lo (z);
  mr[x or $78] := Hi (z);
end {PostIncw};


{ return the contents of the 16-bit register 'x',
  then subtract 8-bit value 'y' from this 16-bit register }
function PostDecw (x, y: byte) : word;
var
  z: word;
begin
  regstep := byte (-1);
  Result := Wreg(x,0);
  z := Result - word(y);
  mr[x or $38] := Lo (z);
  mr[x or $78] := Hi (z);
end {PostDecw};


{ add 8-bit value 'y' to the 16-bit register 'x',
  then return the contents of this 16-bit register }
function PreIncw (x, y: byte) : word;
begin
  Result := Wreg(x,0) + word(y);
  mr[x or $38] := Lo (Result);
  mr[x or $78] := Hi (Result);
end {PreIncw};


{ subtract 8-bit value 'y' from the 16-bit register 'x',
  then return the contents of this 16-bit register }
function PreDecw (x, y: byte) : word;
begin
  regstep := byte (-1);
  Result := Wreg(x,0) - word(y);
  mr[x or $38] := Lo (Result);
  mr[x or $78] := Hi (Result);
end {PreDecw};


{ add the 8-bit value 'x' to the 16-bit variable 'ea' }
procedure PlusOffset (x: byte);
begin
  Inc (ea, word(x));
end {PlusOffset};


{ subtract the 8-bit value 'x' from the 16-bit variable 'ea' }
procedure MinusOffset (x: byte);
begin
  regstep := byte (-1);
  Dec (ea, word(x));
end {MinusOffset};


{ unary operation on a register }
procedure UnReg (op2: pointer);
begin
  if op2 <> @OpSwp1 then flag := 0;
  Func5 (op2) (mr[regbank or Reg1 (FetchByte)], 0);
end {UnReg};


{ register rotation through Carry }
procedure RotReg (op2: pointer);
begin
  flag := flag and C_bit;
  Func5 (op2) (mr[regbank or Reg1 (FetchByte)], 0);
end {RotReg};


{ unary operation on an array }
procedure UnAry (op2: pointer);
var
  x, y, z, dstf, dstl: byte;
begin
  x := FetchByte;
  y := FetchByte;
  if regbank <> 0 then
  begin
    dstf := Reg3 (x, y);
    dstl := AsLimit (y); 
  end
  else
  begin
    dstf := Reg1 (x);
    dstl := NONE_REG;
  end {if};
  x := x and $07;
  y := y shr 5;
  if op2 = @OpBnus then z := mr[dstf] else
  begin
    flag := 0;
    z := 0;
  end {if};
  repeat
    z := Func5 (op2) (mr[dstf], z);
    Inc (cycles, 4);
    if x = y then Exit;
    if dstf = dstl xor ((regstep shr 1) and $07) then
    repeat
      NextReg (x);
      dstf := 0;
      z := Func5 (op2) (dstf, z);
      Inc (cycles, 4);
      if x = y then Exit;
    until FALSE;
    NextReg (x);
    NextReg (dstf);
  until FALSE;
end {UnAry};


{ array rotation through Carry }
procedure RotAry (op2: pointer);
var
  x, y, z, dstf, dstl: byte;
begin
  flag := flag and C_bit;
  x := FetchByte;
  y := FetchByte;
  if regbank <> 0 then
  begin
    dstf := Reg3 (x, y);
    dstl := AsLimit (y); 
  end
  else
  begin
    dstf := Reg1 (x);
    dstl := NONE_REG;
  end {if};
  x := x and $07;
  y := y shr 5;
  z := 0;
  repeat
    z := Func5 (op2) (mr[dstf], z);
    Inc (cycles, 4);
    if x = y then Exit;
    if dstf = dstl xor ((regstep shr 1) and $07) then
    repeat
      NextReg (x);
      dstf := 0;
      z := Func5 (op2) (dstf, z);
      Inc (cycles, 4);
      if x = y then Exit;
    until FALSE;
    NextReg (x);
    NextReg (dstf);
  until FALSE;
end {RotAry};


procedure Mtbm (op2: pointer);
var
  x, y, z: byte;
begin
  flag := 0;
  x := FetchByte;
  y := FetchByte;
  y := regbank or Rl1(x,y);
  x := regbank or Reg1(x);
  z := 0;
  repeat
    z := Func5 (op2) (mr[x], z);
    Inc (cycles, 4);
    if x = y then Break;
    NextReg (x);
  until FALSE;
end {Mtbm};


{ destination_register <- destination_register op2 source_register
  source register in the bank 0, destination register in the bank 1 }
procedure Xreg (op2: pointer);
var
  src, dst: byte;
begin
  if op2 <> @OpLd then flag := 0;
  src := Reg1 (FetchByte);	{ index of the source register }
  dst := Reg2 (FetchByte);	{ index of the destination register }
  mr[dst] := Func3 (op2) (mr[dst], mr[src]);
  Inc (cycles, 4);
end {Xreg};


{ destination_register <- destination_register op2 source_register
  source register in the bank 1, destination register in the bank 0 }
procedure Yreg (op2: pointer);
var
  src, dst: byte;
begin
  if op2 <> @OpLd then flag := 0;
  dst := Reg1 (FetchByte);	{ index of the destination register }
  src := Reg2 (FetchByte);	{ index of the source register }
  mr[dst] := Func3 (op2) (mr[dst], mr[src]);
  Inc (cycles, 4);
end {Yreg};


{ check flags of the operation: destination_register op2 source_register
  source register in the bank 0, destination register in the bank 1 }
procedure TXreg (op2: pointer);
var
  src, dst: byte;
begin
  flag := 0;
  src := Reg1 (FetchByte);	{ index of the source register }
  dst := Reg2 (FetchByte);	{ index of the destination register }
  Func3 (op2) (mr[dst], mr[src]);
  Inc (cycles, 4);
end {TXreg};


{ check flags of the operation: destination_register op2 source_register
  source register in the bank 1, destination register in the bank 0 }
procedure TYreg (op2: pointer);
var
  src, dst: byte;
begin
  flag := 0;
  dst := Reg1 (FetchByte);	{ index of the destination register }
  src := Reg2 (FetchByte);	{ index of the source register }
  Func3 (op2) (mr[dst], mr[src]);
  Inc (cycles, 4);
end {TYreg};


procedure ExchReg (op2: pointer);
var
  x, y: byte;
begin
  y := Reg1 (FetchByte);
  x := Reg2 (FetchByte);
  Proc5 (op2) (mr[x], mr[y]);
  Inc (cycles, 4);
end {ExchReg};


{ register <- register op2 immediate_byte }
procedure RegIm8 (op2: pointer);
var
  x: byte;
begin
  if op2 <> @OpLd then flag := 0;
  x := regbank or Reg1 (FetchByte);	{ index of the register }
  mr[x] := Func3 (op2) (mr[x], FetchByte);
  Inc (cycles, 4);
end {RegIm8};


{ check flag of the operation: register op2 immediate_byte }
procedure TRegIm8 (op2: pointer);
var
  x: byte;
begin
  flag := 0;
  x := regbank or Reg1 (FetchByte);	{ index of the register }
  Func3 (op2) (mr[x], FetchByte);
  Inc (cycles, 4);
end {TRegIm8};


{ destination_array <- destination_array op2 source_array
  source array in the bank 0, destination array in the bank 1 }
procedure Xary (op2: pointer);
var
  x1, x2, srcf, srcl, dstf, dstl: byte;
begin
  if op2 <> @OpLd then flag := 0;
  x1 := FetchByte;
  x2 := FetchByte;
  srcf := Reg1 (x1);		{ index of the first source register }
  srcl := Rl1 (x1, x2);		{ index of the last source register }
  dstf := Reg3 (x1, x2);	{ index of the first destination register }
  dstl := AsLimit (x2);		{ index of the last destination register }
  repeat
    mr[dstf] := Func3 (op2) (mr[dstf], mr[srcf]);
    Inc (cycles, 4);
    if srcf = srcl then Exit;
    if dstf = dstl then
    repeat
      NextReg (srcf);
      Func3 (op2) (0, mr[srcf]);
      Inc (cycles, 4);
      if srcf = srcl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {Xary};


{ destination_array <- destination_array op2 source_array
  source array in the bank 1, destination array in the bank 0 }
procedure Yary (op2: pointer);
var
  x1, x2, srcf, srcl, dstf, dstl: byte;
begin
  if op2 <> @OpLd then flag := 0;
  x1 := FetchByte;
  x2 := FetchByte;
  dstf := Reg1 (x1);		{ index of the first destination register }
  dstl := Rl1 (x1, x2);		{ index of the last destination register }
  srcf := Reg3 (x1, x2);	{ index of the first source register }
  srcl := AsLimit (x2);		{ index of the last source register }
  repeat
    mr[dstf] := Func3 (op2) (mr[dstf], mr[srcf]);
    Inc (cycles, 4);
    if dstf = dstl then Exit;
    if srcf = srcl then
    repeat
      NextReg (dstf);
      mr[dstf] := Func3 (op2) (mr[dstf], 0);
      Inc (cycles, 4);
      if dstf = dstl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {Yary};


{ destination_array <- destination_array op2 source_array
  source array in the bank 0, destination array in the bank 1 }
procedure TXary (op2: pointer);
var
  x1, x2, srcf, srcl, dstf, dstl: byte;
begin
  flag := 0;
  x1 := FetchByte;
  x2 := FetchByte;
  srcf := Reg1 (x1);		{ index of the first source register }
  srcl := Rl1 (x1, x2);		{ index of the last source register }
  dstf := Reg3 (x1, x2);	{ index of the first destination register }
  dstl := AsLimit (x2);		{ index of the last destination register }
  repeat
    Func3 (op2) (mr[dstf], mr[srcf]);
    Inc (cycles, 4);
    if srcf = srcl then Exit;
    if dstf = dstl then
    repeat
      NextReg (srcf);
      Func3 (op2) (0, mr[srcf]);
      Inc (cycles, 4);
      if srcf = srcl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {TXary};


{ destination_array <- destination_array op2 source_array
  source array in the bank 1, destination array in the bank 0 }
procedure TYary (op2: pointer);
var
  x1, x2, srcf, srcl, dstf, dstl: byte;
begin
  flag := 0;
  x1 := FetchByte;
  x2 := FetchByte;
  dstf := Reg1 (x1);		{ index of the first destination register }
  dstl := Rl1 (x1, x2);		{ index of the last destination register }
  srcf := Reg3 (x1, x2);	{ index of the first source register }
  srcl := AsLimit (x2);		{ index of the last source register }
  repeat
    Func3 (op2) (mr[dstf], mr[srcf]);
    Inc (cycles, 4);
    if dstf = dstl then Exit;
    if srcf = srcl then
    repeat
      NextReg (dstf);
      Func3 (op2) (mr[dstf], 0);
      Inc (cycles, 4);
      if dstf = dstl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {TYary};


procedure ExchAry (op2: pointer);
var
  x1, x2, srcf, srcl, dstf, dstl: byte;
begin
  x1 := FetchByte;
  x2 := FetchByte;
  srcf := Reg1 (x1);		{ index of the first source register }
  srcl := Rl1 (x1, x2);		{ index of the last source register }
  dstf := Reg3 (x1, x2);	{ index of the first destination register }
  dstl := AsLimit (x2);		{ index of the last destination register }
  repeat
    Proc5 (op2) (mr[dstf], mr[srcf]);
    Inc (cycles, 4);
    if srcf = srcl then Exit;
    if dstf = dstl then
    repeat
      NextReg (srcf);
      dstf := 0;
      Proc5 (op2) (dstf, mr[srcf]);
      Inc (cycles, 4);
      if srcf = srcl then Exit;
    until FALSE;
    NextReg (srcf);
    NextReg (dstf);
  until FALSE;
end {ExchAry};


{ array <- array op2 immediate_value }
procedure AryIm6 (op2: pointer);
var
  x, y, z: byte;
begin
  if op2 <> @OpLd then flag := 0;
  x := FetchByte;
  y := FetchByte;
  z := Im6 (x,y);
  y := regbank or Rl1 (x,y);	{ index of the last processed register }
  x := regbank or Reg1 (x);	{ index of the first processed register }
  repeat
    mr[x] := Func3 (op2) (mr[x], z);
    Inc (cycles, 4);
    if x = y then Break;
    NextReg (x);
    if op2 <> @OpLd then z := 0;
  until FALSE;
end {AryIm6};


{ check flags of the operation: array op2 immediate_value }
procedure TAryIm6 (op2: pointer);
var
  x, y, z: byte;
begin
  flag := 0;
  x := FetchByte;
  y := FetchByte;
  z := Im6 (x,y);
  y := regbank or Rl1 (x,y);	{ index of the last processed register }
  x := regbank or Reg1 (x);	{ index of the first processed register }
  repeat
    Func3 (op2) (mr[x], z);
    Inc (cycles, 4);
    if x = y then Break;
    NextReg (x);
    z := 0;
  until FALSE;
end {TAryIm6};


procedure Ldw (op2: pointer {dummy});
var
  dst: byte;
begin
  dst := (opcode[0] and $07) or $38;
  mr[dst or $40] := FetchByte;
  mr[dst] := FetchByte;
  Inc (cycles, 4);
end {Ldw};


procedure AdwSbw (op2: pointer);
begin
  Proc3 (op2) (Ireg, FetchByte);
  Inc (cycles, 6);
end {Adw};


{ conditional execution (when true) of 'op2' }
procedure Cond (op2: pointer);
var
  x1, x2: byte;
begin
  x1 := FetchByte;
  x2 := FetchByte;
  if (flag and cc[opcode[0] and $07]) <> 0 then Proc3 (op2) (x1, x2);
  Inc (cycles, 4);
end {Cond};


{ conditional execution (when false) of 'op2' }
procedure NotCond (op2: pointer);
var
  x1, x2: byte;
begin
  x1 := FetchByte;
  x2 := FetchByte;
  if (flag and cc[opcode[0] and $07]) = 0 then Proc3 (op2) (x1, x2);
  Inc (cycles, 4);
end {NotCond};


{ conditional execution (when key pressed) of 'op2' }
procedure KeyCond (op2: pointer);
var
  x1, x2: byte;
begin
  x1 := FetchByte;
  x2 := FetchByte;
  if ki <> 0 then Proc3 (op2) (x1, x2);
  Inc (cycles, 4);
end {KeyCond};


{ conditional execution (when key not pressed) of 'op2' }
procedure NotKeyCond (op2: pointer);
var
  x1, x2: byte;
begin
  x1 := FetchByte;
  x2 := FetchByte;
  if ki = 0 then Proc3 (op2) (x1, x2);
  Inc (cycles, 4);
end {NotKeyCond};


procedure Jump (x1, x2: byte);
begin
  pc := (word(x1) shl 8) or word(x2);
{ Alternatively, the above statement could be replaced with following ones:
  ptrb(PChar(@pc)+1)^ := x1;
  ptrb(@pc)^ := x2;
  Perhaps they would be more effective, especially if the left sides would
  evaluate to constants. Unfortunately, the pointer to the 'pc' variable is
  probably unknown at the compile state. Also they have a disadvantage to be
  less portable, because little endian order of bytes is assumed. }
end {Jump};


procedure Call (x1, x2: byte);
var
  saveie: byte;
begin
  saveie := ie;
  ie := 0;
  DstPtr(PreDecw(SP,1))^ := Lo (pc);
  DstPtr(PreDecw(SP,1))^ := Hi (pc);
  ie := saveie;
  pc := (word(x1) shl 8) or word(x2);
  Inc (cycles, 6);
end {Call};


procedure Trap (op2: pointer {dummy});
begin
  FetchByte;
  Call (mr[V3 or $78], mr[V3 or $38]);
  Inc (cycles, 6);
end {Trap};


{ for illegal operands the emulation may differ from the actual hardware }
procedure Ijmp (x1, x2: byte);
begin
  pc := (word(mr[Reg2(x2)]) shl 8) or word(mr[Reg1(x1)]);
  Inc (cycles, 2);
end {Ijmp};


procedure Rtn (op2: pointer {dummy});
var
  x1, x2, saveie: byte;
begin
  saveie := ie;
  ie := 0;
  x1 := SrcPtr(PostIncw(SP,1))^;
  x2 := SrcPtr(PostIncw(SP,1))^;
  ie := saveie;
  pc := (word(x1) shl 8) or word(x2);
  Inc (cycles, 10);
end {Rtn};


procedure Cani (op2: pointer {dummy});
var
  i: integer;
begin
  for i:=0 to 2 do
  begin
    if irqcnt[i] <> 0 then
    begin
      ifreg := ifreg and not INT_serv[i];
      irqcnt[i] := 0;
      Break;
    end {if};
  end {for};
  Inc (cycles, 4);
end {Cani};


procedure Rti (op2: pointer {dummy});
begin
  Cani (nil);
  Rtn (nil);
  Dec (cycles, 4);
end {Rti};


procedure Nop (op2: pointer {dummy});
begin
  Inc (cycles, 4);
end {Nop};


procedure BlockCopy (op2: pointer);
begin
  DstPtr(Func4 (op2) (IZ,1))^ := SrcPtr(Func4 (op2) (IX,1))^;
  if (mr[IX+56] <> mr[IY+56]) or (mr[IX+120] <> mr[IY+120]) then pc := savepc;
  Inc (cycles, 14);
end {BlockCopy};


procedure BlockSearch (op2: pointer);
begin
  if (DstPtr(Func4 (op2) (Ireg,1))^ <> mr[regbank or Reg1(FetchByte)]) then
  begin
    if Wreg(Ireg,0) <> Wreg(IY,0) then pc := savepc;
  end {if};
  Inc (cycles, 16);
end {BlockSearch};


procedure StMemoReg (op2: pointer);
begin
  DstPtr(Func4 (op2) (Ireg,1))^ := mr[regbank or Reg1(FetchByte)];
  Inc (cycles, 12);
end {StMemoReg};


procedure StMemoIm8 (op2: pointer);
begin
  DstPtr(Func4 (op2) (Ireg,1))^ := FetchByte;
  Inc (cycles, 6);
end {StMemoIm8};


procedure StmMemoAry (op2: pointer);
var
  dst, x, y: byte;
begin
  dst := Ireg;
  x := FetchByte;
  y := FetchByte;
  y := regbank or Rl1 (x, y);	{ index of the last processed register }
  x := regbank or Reg1 (x);	{ index of the first processed register }
  repeat
    DstPtr(Func4 (op2) (dst,1))^ := mr[x];
    Inc (cycles, 4);
    if x = y then Break;
    NextReg (x);
  until FALSE;
  Inc (cycles, 8);
end {StmMemoAry};


procedure StImOffsReg (op2: pointer);
var
  x: byte;
begin
  ea := Wreg (Ireg,0);
  x := Reg1 (FetchByte);
  Proc4 (op2) (FetchByte);
  DstPtr(ea)^ := mr[x];
  Inc (cycles, 12);
end {StImOffsReg};


procedure StRegOffsReg (op2: pointer);
var
  x: byte;
begin
  ea := Wreg (Ireg,0);
  Proc4 (op2) (mr[Reg1 (FetchByte)]);
  x := Reg2 (FetchByte);
  DstPtr(ea)^ := mr[x];
  Inc (cycles, 12);
end {StRegOffsReg};


procedure StmImOffsAry (op2: pointer);
var
  x, y: byte;
begin
  ea := Wreg (Ireg,0);
  x := FetchByte;
  y := FetchByte;
  Proc4 (op2) (Im6 (x,y));
  y := Rl1 (x, y);		{ index of the last processed register }
  x := Reg1 (x);		{ index of the first processed register }
  repeat
    DstPtr(ea)^ := mr[x];
    Inc (cycles, 4);
    if x = y then Break;
    NextReg (x);
    Proc4 (op2) (1);
  until FALSE;
  Inc (cycles, 8);
end {StmImOffsAry};


procedure StmRegOffsAry (op2: pointer);
var
  x, y, first, last: byte;
begin
  ea := Wreg (Ireg,0);
  x := FetchByte;
  y := FetchByte;
  Proc4 (op2) (mr[Reg1 (x)]);
  first := Reg1 (x);
  last := Rl1 (x, y);
  x := Reg3 (x, y);		{ index of the first processed register }
  repeat
    DstPtr(ea)^ := mr[x];
    Inc (cycles, 4);
    if first = last then Break;
    NextReg (first);
    NextReg (x);
    Proc4 (op2) (1);
  until FALSE;
  Inc (cycles, 8);
end {StmRegOffsAry};


procedure LdRegMemo (op2: pointer);
begin
  mr[regbank or Reg1(FetchByte)] := SrcPtr(Func4 (op2) (Ireg,1))^;
  Inc (cycles, 12);
end {LdRegMemo};


procedure LdmAryMemo (op2: pointer);
var
  src, x, y: byte;
begin
  src := Ireg;
  x := FetchByte;
  y := FetchByte;
  y := regbank or Rl1 (x, y);	{ index of the last processed register }
  x := regbank or Reg1 (x);	{ index of the first processed register }
  repeat
    mr[x] := SrcPtr(Func4 (op2) (src,1))^;
    Inc (cycles, 4);
    if x = y then Break;
    NextReg (x);
  until FALSE;
  Inc (cycles, 8);
end {LdmAryMemo};


procedure LdRegImOffs (op2: pointer);
var
  x: byte;
begin
  ea := Wreg (Ireg,0);
  x := Reg1 (FetchByte);
  Proc4 (op2) (FetchByte);
  mr[x] := SrcPtr(ea)^;
  Inc (cycles, 12);
end {LdRegImOffs};


procedure LdRegRegOffs (op2: pointer);
var
  x: byte;
begin
  ea := Wreg (Ireg,0);
  Proc4 (op2) (mr[Reg1 (FetchByte)]);
  x := Reg2 (FetchByte);
  mr[x] := SrcPtr(ea)^;
  Inc (cycles, 12);
end {LdRegRegOffs};


procedure LdmAryImOffs (op2: pointer);
var
  x, y: byte;
begin
  ea := Wreg (Ireg,0);
  x := FetchByte;
  y := FetchByte;
  Proc4 (op2) (Im6 (x,y));
  y := Rl1 (x, y);		{ index of the last processed register }
  x := Reg1 (x);		{ index of the first processed register }
  repeat
    mr[x] := SrcPtr(ea)^;
    Inc (cycles, 4);
    if x = y then Break;
    NextReg (x);
    Proc4 (op2) (1);
  until FALSE;
  Inc (cycles, 8);
end {LdmAryImOffs};


procedure LdmAryRegOffs (op2: pointer);
var
  x, y, first, last: byte;
begin
  ea := Wreg (Ireg,0);
  x := FetchByte;
  y := FetchByte;
  Proc4 (op2) (mr[Reg1 (x)]);
  first := Reg1 (x);
  last := Rl1 (x, y);
  x := Reg3 (x, y);		{ index of the first processed register }
  repeat
    mr[x] := SrcPtr(ea)^;
    Inc (cycles, 4);
    if first = last then Break;
    NextReg (first);
    NextReg (x);
    Proc4 (op2) (1);
  until FALSE;
  Inc (cycles, 8);
end {LdmAryRegOffs};


procedure PstIm8 (op2: pointer);
begin
  Proc4 (op2) (FetchByte);
  Inc (cycles, 4);
end {PstIm8};


procedure PstReg (op2: pointer);
begin
  Proc4 (op2) (mr[regbank or Reg1(FetchByte)]);
  Inc (cycles, 4);
end {PstReg};


procedure Gst (op2: pointer);
begin
  mr[regbank or Reg1(FetchByte)] := ptrb(op2)^;
  Inc (cycles, 4);
end {Gst};


procedure Off (op2: pointer);
begin
  CpuSleep := True;
  pc := $0000;
  ie := 0;
  ifreg := $00;
  irqcnt[0] := 0;
  irqcnt[1] := 0;
  irqcnt[2] := 0;
  lcdctrl := 0;
  LcdInit;
  DoPorts;
  ko := $41;
  KeyHandle;
end {Gst};



{ ARITHMETICAL AND LOGICAL OPERATIONS }


procedure ZeroBits (x: byte);
begin
  if x <> 0 then flag := flag or NZ_bit;
  if (x and $F0) = 0 then flag := flag or UZ_bit;
  if (x and $0F) = 0 then flag := flag or LZ_bit;
end {ZeroBits};


{ binary operations }

{ addition with carry }
function OpAd (x, y: byte) : byte;
var
  in1, in2, temp, out: cardinal;
begin
  in1 := cardinal(x);
  in2 := cardinal(y);
  out := in1 + in2;
  if (flag and C_bit) <> 0 then Inc(out);
  temp := in1 xor in2 xor out;
  flag := flag and NZ_bit;
  if out > $FF then flag := flag or C_bit;
  if (temp and $10) <> 0 then flag := flag or H_bit;
  if (temp and $80) <> 0 then flag := flag or V_bit;
  ZeroBits (byte(out));
  OpAd := byte(out);
end {OpAd};


{ subtraction with borrow }
function OpSb (x, y: byte) : byte;
var
  in1, in2, temp, out: cardinal;
begin
  in1 := cardinal(x);
  in2 := cardinal(y);
  out := in1 - in2;
  if (flag and C_bit) <> 0 then Dec(out);
  temp := in1 xor in2 xor out;
  flag := flag and NZ_bit;
  if out > $FF then flag := flag or C_bit;
  if (temp and $10) <> 0 then flag := flag or H_bit;
  if (temp and $80) <> 0 then flag := flag or V_bit;
  ZeroBits (byte(out));
  OpSb := byte(out);
end {OpSb};


{ BCD addition with carry }
function OpAdb (x, y: byte) : byte;
var
  in1, in2, out: cardinal;
begin
  in1 := cardinal(x);
  in2 := cardinal(y);
{ lower nibble }
  out := (in1 and $0F) + (in2 and $0F);
  if (flag and C_bit) <> 0 then Inc(out);
  flag := flag and NZ_bit;
{ decimal adjustement }
  if out > $09 then
  begin
    out := ((out + $06) and $0F) or $10;
    if out > $1F then Dec(out,$10);
    flag := flag or H_bit;
  end {if};
{ upper nibble }
  Inc(out, (in1 and $F0) + (in2 and $F0));
  if ((in1 xor in2 xor out) and $80) <> 0 then flag := flag or V_bit;
{ decimal adjustement }
  if out > $9F then
  begin
    Inc(out,$60);
    flag := flag or C_bit;
  end {if};
  ZeroBits (byte(out));
  OpAdb := byte(out);
end {OpAdb};


{ BCD subtraction with borrow }
function OpSbb (x, y: byte) : byte;
var
  in1, in2, out: cardinal;
begin
  in1 := cardinal(x);
  in2 := cardinal(y);
{ lower nibble }
  out := (in1 and $0F) - (in2 and $0F);
  if (flag and C_bit) <> 0 then Dec(out);
  flag := flag and NZ_bit;
{ decimal adjustement }
  if out > $09 then
  begin
    out := (out - $06) or cardinal (-$10);
    flag := flag or H_bit;
  end {if};
{ upper nibble }
  Inc(out, (in1 and $F0) - (in2 and $F0));
  if ((in1 xor in2 xor out) and $80) <> 0 then flag := flag or V_bit;
{ decimal adjustement }
  if out > $9F then
  begin
    Dec(out,$60);
    flag := flag or C_bit;
  end {if};
  ZeroBits (byte(out));
  OpSbb := byte(out);
end {OpSbb};


function OpAn (x, y: byte) : byte;
begin
  Result := x and y;
  flag := flag and NZ_bit;
  ZeroBits (Result);
end {OpAn};


function OpBit (x, y: byte) : byte;
begin
  Result := not x and y;
  flag := flag and NZ_bit;
  ZeroBits (Result);
end {OpBit};


function OpXr (x, y: byte) : byte;
begin
  Result := x xor y;
  flag := flag and NZ_bit;
  ZeroBits (Result);
end {OpXr};


function OpNa (x, y: byte) : byte;
begin
  OpNa := not (x and y);
  flag := (flag and NZ_bit) or (C_bit or V_bit or H_bit);
end {OpNa};


function OpOr (x, y: byte) : byte;
begin
  Result := x or y;
  flag := (flag and NZ_bit) or (C_bit or V_bit or H_bit);
  ZeroBits (Result);
end {OpOr};


function OpLd (x, y: byte) : byte;
begin
  OpLd := y;
end {OpLd};


{ unary operations, dummy second operand and returned value }


function OpRod (var x: byte; y: byte) : byte;
var
  z: byte;
begin
  regstep := byte (-1);
  z := x;
  x := x shr 1;
  if (flag and C_bit) <> 0 then x := x or $80;
  flag := flag and NZ_bit;
  if (z and $01) <> 0 then flag := flag or C_bit;
  ZeroBits (x);
  OpRod := 0;
end {OpRod};


function OpRou (var x: byte; y: byte) : byte;
begin
  x := OpAd (x, x);
  OpRou := 0;
end {OpRou};


function OpMtb (var x: byte; y: byte) : byte;
begin
  x := OpAdb (x, x);
  OpMtb := 0;
end {OpMtb};


function OpInv (var x: byte; y: byte) : byte;
begin
  x := not x;
  flag := (flag and NZ_bit) or (C_bit or V_bit or H_bit);
  ZeroBits (x);
  OpInv := 0;
end {OpInv};


function OpCmp (var x: byte; y: byte) : byte;
begin
  x := OpSb (0, x);
  OpCmp := 0;
end {OpCmp};


function OpCmpb (var x: byte; y: byte) : byte;
begin
  x := OpSbb (0, x);
  OpCmpb := 0;
end {OpCmpb};


function OpSwp1 (var x: byte; y: byte) : byte;
begin
  x := (x shl 4) or (x shr 4);
  OpSwp1 := 0;
end {OpSwp1};


{ shifts by 4 and 8 bits }

function OpDiu (var x: byte; y: byte) : byte;
begin
  OpDiu := x shr 4;
  x := (x shl 4) or y;
  flag := flag and NZ_bit;
  ZeroBits (x);
end {OpDiu};


function OpDid (var x: byte; y: byte) : byte;
begin
  regstep := byte (-1);
  OpDid := x shl 4;
  x := (x shr 4) or y;
  flag := flag and NZ_bit;
  ZeroBits (x);
end {OpDid};


function OpByu (var x: byte; y: byte) : byte;
begin
  OpByu := x;
  x := y;
  flag := flag and NZ_bit;
  ZeroBits (x);
end {OpByu};


function OpByd (var x: byte; y: byte) : byte;
begin
  regstep := byte (-1);
  OpByd := x;
  x := y;
  flag := flag and NZ_bit;
  ZeroBits (x);
end {OpByu};


function OpBnus (var x: byte; y: byte) : byte;
begin
  OpBnus := x;
  x := (y shl 4) or (y shr 4);
end {OpBnus};


{ exchanges }

{ exchange of two bytes }
procedure OpXc (var x, y: byte);
var
  temp: byte;
begin
  temp := x;
  x := y;
  y := temp;
end {OpXc};


{ rotation of four digits (nibbles) counterclockwise }
procedure OpXcls (var x, y: byte);
var
  temp: byte;
begin
  temp := (y shr 4) or (x and $F0);
  x := (x shl 4) or (y and $0F);
  y := temp;
end {OpXcls};


{ rotation of four digits (nibbles) clockwise }
procedure OpXchs (var x, y: byte);
var
  temp: byte;
begin
  temp := (y shl 4) or (x and $0F);
  x := (x shr 4) or (y and $F0);
  y := temp;
end {OpXchs};


{ swap nibbles in two bytes }
procedure OpSwp2 (var x, y: byte);
begin
  x := (x shl 4) or (x shr 4);
  y := (y shl 4) or (y shr 4);
end {OpSwp2};



{ WRITING TO THE STATUS REGISTERS }


procedure OpKo (x: byte);
begin
  ko := x;
  KeyHandle;
end {OpKo};


procedure OpIf (x: byte);
begin
  ifreg := (ifreg and $EE) or (x and $11);
  DoPorts;
end {OpIf};


procedure OpAs (x: byte);
begin
  asreg := x;
end {OpAs};


procedure OpIe (x: byte);
var
  i: integer;
begin
  ie := x;
  for i:=0 to 2 do
  begin
    if (x and INT_enable[i]) = 0 then
    begin
      ifreg := ifreg and not INT_serv[i];
      irqcnt[i] := 0;
    end {if};
  end {for};
end {OpIe};


procedure OpFl (x: byte);
begin
  flag := x;
end {OpFl};



{ LCD TRANSFER }


procedure Ldle (op2: pointer);
var
  x: byte;
begin
  LcdSync;
  x := regbank or Reg1 (FetchByte);	{ index of the register }
  lcdctrl := (lcdctrl and not $3F) or (FetchByte and $3F);
  mr[x] := LcdTransfer (0);
  Inc (cycles, 8);
  mr[x] := mr[x] or (LcdTransfer (0) shl 4);
end {Ldle};


procedure Ldlo (op2: pointer);
var
  x: byte;
begin
  LcdSync;
  x := regbank or Reg1 (FetchByte);	{ index of the register }
  lcdctrl := (lcdctrl and not $3F) or (FetchByte and $3F);
  mr[x] := LcdTransfer (0);
  Inc (cycles, 8);
end {Ldlo};


procedure Stle (op2: pointer);
var
  x: byte;
begin
  LcdSync;
  x := regbank or Reg1 (FetchByte);	{ index of the register }
  lcdctrl := (lcdctrl and not $3F) or (FetchByte and $3F);
  LcdTransfer (mr[x]);
  Inc (cycles, 8);
  LcdTransfer (mr[x] shr 4);
end {Stle};


procedure Stlo (op2: pointer);
var
  x: byte;
begin
  LcdSync;
  x := regbank or Reg1 (FetchByte);	{ index of the register }
  lcdctrl := (lcdctrl and not $3F) or (FetchByte and $3F);
  LcdTransfer (mr[x]);
  Inc (cycles, 8);
end {Stlo};


procedure Ldlem (op2: pointer);
var
  x, y: byte;
begin
  LcdSync;
  x := FetchByte;
  y := FetchByte;
  lcdctrl := (lcdctrl and not $3F) or Im6 (x,y);
  y := regbank or Rl1 (x,y);	{ index of the last processed register }
  x := regbank or Reg1 (x);	{ index of the first processed register }
  repeat
    mr[x] := LcdTransfer (0);
    Inc (cycles, 8);
    mr[x] := mr[x] or (LcdTransfer (0) shl 4);
    if x = y then Break;
    NextReg (x);
  until FALSE;
end {Ldlem};


procedure Ldlom (op2: pointer);
var
  x, y: byte;
begin
  LcdSync;
  x := FetchByte;
  y := FetchByte;
  lcdctrl := (lcdctrl and not $3F) or Im6 (x,y);
  y := regbank or Rl1 (x,y);	{ index of the last processed register }
  x := regbank or Reg1 (x);	{ index of the first processed register }
  repeat
    mr[x] := LcdTransfer (0);
    Inc (cycles, 8);
    if x = y then Break;
    mr[x] := mr[x] or (LcdTransfer (0) shl 4);
    NextReg (x);
  until FALSE;
end {Ldlom};


procedure Stlem (op2: pointer);
var
  x, y: byte;
begin
  LcdSync;
  x := FetchByte;
  y := FetchByte;
  lcdctrl := (lcdctrl and not $3F) or Im6 (x,y);
  y := regbank or Rl1 (x,y);	{ index of the last processed register }
  x := regbank or Reg1 (x);	{ index of the first processed register }
  repeat
    LcdTransfer (mr[x]);
    Inc (cycles, 8);
    LcdTransfer (mr[x] shr 4);
    if x = y then Break;
    NextReg (x);
  until FALSE;
end {Stlem};


procedure Stlom (op2: pointer);
var
  x, y: byte;
begin
  LcdSync;
  x := FetchByte;
  y := FetchByte;
  lcdctrl := (lcdctrl and not $3F) or Im6 (x,y);
  y := regbank or Rl1 (x,y);	{ index of the last processed register }
  x := regbank or Reg1 (x);	{ index of the first processed register }
  repeat
    LcdTransfer (mr[x]);
    Inc (cycles, 8);
    if x = y then Break;
    LcdTransfer (mr[x] shr 4);
    NextReg (x);
  until FALSE;
end {Stlom};


end.
