/* HD61700 disassembler */

#include <stdio.h>

typedef unsigned int uint;

/* global variables */
uint loc;		/* location pointer */
#define INBUFSIZE 8
uint inbuf[INBUFSIZE];
uint head, tail;	/* indexes to the inbuf */
uint dsize;	/* 1 if 8-bit memory access, 2 if 16-bit memory access */


enum {
  ILLOP,	/* illop */
  NONE,		/* nop */
  CC,		/* rtn z */
  JRCC,		/* jr z, relative_address */
  JPCC,		/* jp z, absolute_address */
  JR,		/* jr relative_address */
  JP,		/* jp absolute_address */
  REGREGJR,	/* ld reg, reg, optional_jr */
  REGDIRJR,	/* ld reg, (reg), optional_jr */
  REGJR,	/* stl reg, optional_jr */
  REGIRR,	/* st reg, (IX+/-reg) */
  REGIRRIM3,	/* stm reg, (IX+/-reg), IM3 */
  REG,		/* phs reg */
  DIR,		/* jp (reg) */
  IRRREG,	/* adc (IX+/-reg), reg */
  REGIM8JR,	/* adc reg, IM8, optional_jr */
  IM8,		/* stl IM8 */
  IM8A,		/* ppo IM8 */
  R8IM8,	/* pst PE,IM8 */
  REGIRI,	/* st reg, (IX+/-IM8) */
  IRIREG,	/* adc (IX+/-IM8), reg */
  R8REGJR,	/* gst reg_8bit, reg, optional_jr */
  R16REGJR,	/* pre reg_16bit, reg, optional_jr */
  R16IM16,	/* pre reg_16bit, IM16 */
  IM8IND,	/* st IM8,($sir) */
  IM16IND,	/* stw IM16,($sir) */
  RRIM3JR,	/* adbm reg, reg, IM3, optional_jr */
  RIM5IM3JR,	/* adbm reg, IM5, IM3, optional_jr */
  REGIM8,	/* ld reg,IM8 without optional jr */
  REGIM16,	/* ldw reg,IM16 */
  REGIM3,	/* stlm reg,IM3 */
  SIRREGJR,	/* psr sir, reg, optional_jr */
  SIRREGIM3,	/* psr sir, reg, IM3 */
  SIRIM5	/* psr sir, IM5 */
};


struct tab {
  char *str;
  int kind;
};

struct tab mnem[] = {
  { "adc",	REGREGJR	},	/* code $00 */
  { "sbc",	REGREGJR	},	/* code $01 */
  { "ld",	REGREGJR	},	/* code $02 */
  { "ldc",	REGREGJR	},	/* code $03 */
  { "anc",	REGREGJR	},	/* code $04 */
  { "nac",	REGREGJR	},	/* code $05 */
  { "orc",	REGREGJR	},	/* code $06 */
  { "xrc",	REGREGJR	},	/* code $07 */
  { "ad",	REGREGJR	},	/* code $08 */
  { "sb",	REGREGJR	},	/* code $09 */
  { "adb",	REGREGJR	},	/* code $0A */
  { "sbb",	REGREGJR	},	/* code $0B */
  { "an",	REGREGJR	},	/* code $0C */
  { "na",	REGREGJR	},	/* code $0D */
  { "or",	REGREGJR	},	/* code $0E */
  { "xr",	REGREGJR	},	/* code $0F */
  { "st",	REGDIRJR	},	/* code $10 */
  { "ld",	REGDIRJR	},	/* code $11 */
  { "0",	256+0*4		},	/* code $12 */
  { "0",	256+18*4	},	/* code $13 */
  { "0",	256+1*4		},	/* code $14 */
  { "psr",	SIRREGJR	},	/* code $15 */
  { "pst",	R8REGJR		},	/* code $16 */
  { "pst",	R8REGJR		},	/* code $17 */
  { "0",	256+2*4		},	/* code $18 */
  { "****",	ILLOP		},	/* code $19 */
  { "0",	256+3*4		},	/* code $1A */
  { "0",	256+4*4		},	/* code $1B */
  { "0",	256+5*4		},	/* code $1C */
  { "gsr",	SIRREGJR	},	/* code $1D */
  { "gst",	R8REGJR		},	/* code $1E */
  { "gst",	R8REGJR		},	/* code $1F */
  { "st",	REGIRR		},	/* code $20 */
  { "st",	REGIRR		},	/* code $21 */
  { "sti",	REGIRR		},	/* code $22 */
  { "sti",	REGIRR		},	/* code $23 */
  { "std",	REGIRR		},	/* code $24 */
  { "std",	REGIRR		},	/* code $25 */
  { "phs",	REG		},	/* code $26 */
  { "phu",	REG		},	/* code $27 */
  { "ld",	REGIRR		},	/* code $28 */
  { "ld",	REGIRR		},	/* code $29 */
  { "ldi",	REGIRR		},	/* code $2A */
  { "ldi",	REGIRR		},	/* code $2B */
  { "ldd",	REGIRR		},	/* code $2C */
  { "ldd",	REGIRR		},	/* code $2D */
  { "pps",	REG		},	/* code $2E */
  { "ppu",	REG		},	/* code $2F */
  { "jp",	JPCC		},	/* code $30 */
  { "jp",	JPCC		},	/* code $31 */
  { "jp",	JPCC		},	/* code $32 */
  { "jp",	JPCC		},	/* code $33 */
  { "jp",	JPCC		},	/* code $34 */
  { "jp",	JPCC		},	/* code $35 */
  { "jp",	JPCC		},	/* code $36 */
  { "jp",	JP		},	/* code $37 */
  { "adc",	IRRREG		},	/* code $38 */
  { "adc",	IRRREG		},	/* code $39 */
  { "sbc",	IRRREG		},	/* code $3A */
  { "sbc",	IRRREG		},	/* code $3B */
  { "ad",	IRRREG		},	/* code $3C */
  { "ad",	IRRREG		},	/* code $3D */
  { "sb",	IRRREG		},	/* code $3E */
  { "sb",	IRRREG		},	/* code $3F */
  { "adc",	REGIM8JR	},	/* code $40 */
  { "sbc",	REGIM8JR	},	/* code $41 */
  { "ld",	REGIM8JR	},	/* code $42 */
  { "ldc",	REGIM8JR	},	/* code $43 */
  { "anc",	REGIM8JR	},	/* code $44 */
  { "nac",	REGIM8JR	},	/* code $45 */
  { "orc",	REGIM8JR	},	/* code $46 */
  { "xrc",	REGIM8JR	},	/* code $47 */
  { "ad",	REGIM8JR	},	/* code $48 */
  { "sb",	REGIM8JR	},	/* code $49 */
  { "adb",	REGIM8JR	},	/* code $4A */
  { "sbb",	REGIM8JR	},	/* code $4B */
  { "an",	REGIM8JR	},	/* code $4C */
  { "na",	REGIM8JR	},	/* code $4D */
  { "or",	REGIM8JR	},	/* code $4E */
  { "xr",	REGIM8JR	},	/* code $4F */
  { "st",	IM8IND		},	/* code $50 */
  { "0",	256+20*4	},	/* code $51 */
  { "stl",	IM8		},	/* code $52 */
  { "****",	ILLOP		},	/* code $53 */
  { "0",	256+6*4		},	/* code $54 */
  { "psr",	SIRIM5		},	/* code $55 */
  { "pst",	R8IM8		},	/* code $56 */
  { "pst",	R8IM8		},	/* code $57 */
  { "bups",	IM8		},	/* code $58 */
  { "bdns",	IM8		},	/* code $59 */
  { "****",	ILLOP		},	/* code $5A */
  { "****",	ILLOP		},	/* code $5B */
  { "sup",	IM8		},	/* code $5C */
  { "sdn",	IM8		},	/* code $5D */
  { "****",	ILLOP		},	/* code $5E */
  { "****",	ILLOP		},	/* code $5F */
  { "st",	REGIRI		},	/* code $60 */
  { "st",	REGIRI		},	/* code $61 */
  { "sti",	REGIRI		},	/* code $62 */
  { "sti",	REGIRI		},	/* code $63 */
  { "std",	REGIRI		},	/* code $64 */
  { "std",	REGIRI		},	/* code $65 */
  { "****",	ILLOP		},	/* code $66 */
  { "****",	ILLOP		},	/* code $67 */
  { "ld",	REGIRI		},	/* code $68 */
  { "ld",	REGIRI		},	/* code $69 */
  { "ldi",	REGIRI		},	/* code $6A */
  { "ldi",	REGIRI		},	/* code $6B */
  { "ldd",	REGIRI		},	/* code $6C */
  { "ldd",	REGIRI		},	/* code $6D */
  { "****",	ILLOP		},	/* code $6E */
  { "****",	ILLOP		},	/* code $6F */
  { "cal",	JPCC		},	/* code $70 */
  { "cal",	JPCC		},	/* code $71 */
  { "cal",	JPCC		},	/* code $72 */
  { "cal",	JPCC		},	/* code $73 */
  { "cal",	JPCC		},	/* code $74 */
  { "cal",	JPCC		},	/* code $75 */
  { "cal",	JPCC		},	/* code $76 */
  { "cal",	JP		},	/* code $77 */
  { "adc",	IRIREG		},	/* code $78 */
  { "adc",	IRIREG		},	/* code $79 */
  { "sbc",	IRIREG		},	/* code $7A */
  { "sbc",	IRIREG		},	/* code $7B */
  { "ad",	IRIREG		},	/* code $7C */
  { "ad",	IRIREG		},	/* code $7D */
  { "sb",	IRIREG		},	/* code $7E */
  { "sb",	IRIREG		},	/* code $7F */
  { "adcw",	REGREGJR	},	/* code $80 */
  { "sbcw",	REGREGJR	},	/* code $81 */
  { "ldw",	REGREGJR	},	/* code $82 */
  { "ldcw",	REGREGJR	},	/* code $83 */
  { "ancw",	REGREGJR	},	/* code $84 */
  { "nacw",	REGREGJR	},	/* code $85 */
  { "orcw",	REGREGJR	},	/* code $86 */
  { "xrcw",	REGREGJR	},	/* code $87 */
  { "adw",	REGREGJR	},	/* code $88 */
  { "sbw",	REGREGJR	},	/* code $89 */
  { "adbw",	REGREGJR	},	/* code $8A */
  { "sbbw",	REGREGJR	},	/* code $8B */
  { "anw",	REGREGJR	},	/* code $8C */
  { "naw",	REGREGJR	},	/* code $8D */
  { "orw",	REGREGJR	},	/* code $8E */
  { "xrw",	REGREGJR	},	/* code $8F */
  { "stw",	REGDIRJR	},	/* code $90 */
  { "ldw",	REGDIRJR	},	/* code $91 */
  { "0",	256+7*4		},	/* code $92 */
  { "0",	256+19*4	},	/* code $93 */
  { "0",	256+8*4		},	/* code $94 */
  { "psrw",	SIRREGJR	},	/* code $95 */
  { "pre",	R16REGJR	},	/* code $96 */
  { "pre",	R16REGJR	},	/* code $97 */
  { "0",	256+9*4		},	/* code $98 */
  { "****",	ILLOP		},	/* code $99 */
  { "0",	256+10*4	},	/* code $9A */
  { "0",	256+11*4	},	/* code $9B */
  { "0",	256+12*4	},	/* code $9C */
  { "gsrw",	SIRREGJR	},	/* code $9D */
  { "gre",	R16REGJR	},	/* code $9E */
  { "gre",	R16REGJR	},	/* code $9F */
  { "stw",	REGIRR		},	/* code $A0 */
  { "stw",	REGIRR		},	/* code $A1 */
  { "stiw",	REGIRR		},	/* code $A2 */
  { "stiw",	REGIRR		},	/* code $A3 */
  { "stdw",	REGIRR		},	/* code $A4 */
  { "stdw",	REGIRR		},	/* code $A5 */
  { "phsw",	REG		},	/* code $A6 */
  { "phuw",	REG		},	/* code $A7 */
  { "ldw",	REGIRR		},	/* code $A8 */
  { "ldw",	REGIRR		},	/* code $A9 */
  { "ldiw",	REGIRR		},	/* code $AA */
  { "ldiw",	REGIRR		},	/* code $AB */
  { "lddw",	REGIRR		},	/* code $AC */
  { "lddw",	REGIRR		},	/* code $AD */
  { "ppsw",	REG		},	/* code $AE */
  { "ppuw",	REG		},	/* code $AF */
  { "jr",	JRCC		},	/* code $B0 */
  { "jr",	JRCC		},	/* code $B1 */
  { "jr",	JRCC		},	/* code $B2 */
  { "jr",	JRCC		},	/* code $B3 */
  { "jr",	JRCC		},	/* code $B4 */
  { "jr",	JRCC		},	/* code $B5 */
  { "jr",	JRCC		},	/* code $B6 */
  { "jr",	JR		},	/* code $B7 */
  { "adcw",	IRRREG		},	/* code $B8 */
  { "adcw",	IRRREG		},	/* code $B9 */
  { "sbcw",	IRRREG		},	/* code $BA */
  { "sbcw",	IRRREG		},	/* code $BB */
  { "adw",	IRRREG		},	/* code $BC */
  { "adw",	IRRREG		},	/* code $BD */
  { "sbw",	IRRREG		},	/* code $BE */
  { "sbw",	IRRREG		},	/* code $BF */
  { "adbcm",	RRIM3JR		},	/* code $C0 */
  { "sbbcm",	RRIM3JR		},	/* code $C1 */
  { "ldm",	RRIM3JR		},	/* code $C2 */
  { "ldcm",	RRIM3JR		},	/* code $C3 */
  { "ancm",	RRIM3JR		},	/* code $C4 */
  { "nacm",	RRIM3JR		},	/* code $C5 */
  { "orcm",	RRIM3JR		},	/* code $C6 */
  { "xrcm",	RRIM3JR		},	/* code $C7 */
  { "adbm",	RRIM3JR		},	/* code $C8 */
  { "sbbm",	RRIM3JR		},	/* code $C9 */
  { "0",	256+13*4	},	/* code $CA */
  { "0",	256+14*4	},	/* code $CB */
  { "anm",	RRIM3JR		},	/* code $CC */
  { "nam",	RRIM3JR		},	/* code $CD */
  { "orm",	RRIM3JR		},	/* code $CE */
  { "xrm",	RRIM3JR		},	/* code $CF */
  { "stw",	IM16IND		},	/* code $D0 */
  { "0",	256+21*4	},	/* code $D1 */
  { "stlm",	REGIM3		},	/* code $D2 */
  { "0",	256+15*4	},	/* code $D3 */
  { "ppom",	REGIM3		},	/* code $D4 */
  { "psrm",	SIRREGIM3	},	/* code $D5 */
  { "pre",	R16IM16		},	/* code $D6 */
  { "pre",	R16IM16		},	/* code $D7 */
  { "bup",	NONE		},	/* code $D8 */
  { "bdn",	NONE		},	/* code $D9 */
  { "0",	256+16*4	},	/* code $DA */
  { "0",	256+17*4	},	/* code $DB */
  { "sup",	REG		},	/* code $DC */
  { "sdn",	REG		},	/* code $DD */
  { "jp",	REG		},	/* code $DE */
  { "jp",	DIR		},	/* code $DF */
  { "stm",	REGIRRIM3	},	/* code $E0 */
  { "stm",	REGIRRIM3	},	/* code $E1 */
  { "stim",	REGIRRIM3	},	/* code $E2 */
  { "stim",	REGIRRIM3	},	/* code $E3 */
  { "stdm",	REGIRRIM3	},	/* code $E4 */
  { "stdm",	REGIRRIM3	},	/* code $E5 */
  { "phsm",	REGIM3		},	/* code $E6 */
  { "phum",	REGIM3		},	/* code $E7 */
  { "ldm",	REGIRRIM3	},	/* code $E8 */
  { "ldm",	REGIRRIM3	},	/* code $E9 */
  { "ldim",	REGIRRIM3	},	/* code $EA */
  { "ldim",	REGIRRIM3	},	/* code $EB */
  { "lddm",	REGIRRIM3	},	/* code $EC */
  { "lddm",	REGIRRIM3	},	/* code $ED */
  { "ppsm",	REGIM3		},	/* code $EE */
  { "ppum",	REGIM3		},	/* code $EF */
  { "rtn",	CC		},	/* code $F0 */
  { "rtn",	CC		},	/* code $F1 */
  { "rtn",	CC		},	/* code $F2 */
  { "rtn",	CC		},	/* code $F3 */
  { "rtn",	CC		},	/* code $F4 */
  { "rtn",	CC		},	/* code $F5 */
  { "rtn",	CC		},	/* code $F6 */
  { "rtn",	NONE		},	/* code $F7 */
  { "nop",	NONE		},	/* code $F8 */
  { "clt",	NONE		},	/* code $F9 */
  { "fst",	NONE		},	/* code $FA */
  { "slw",	NONE		},	/* code $FB */
  { "cani",	NONE		},	/* code $FC */
  { "rtni",	NONE		},	/* code $FD */
  { "off",	NONE		},	/* code $FE */
  { "trp",	NONE		},	/* code $FF */
/* mnemonic variations selected by bits 6 and 5 of the second byte */
/* code $12, index 256+0*4 */
  { "stl",	REGJR		},	/* x00xxxxx */
  { "****",	ILLOP		},	/* x01xxxxx */
  { "****",	ILLOP		},	/* x10xxxxx */
  { "****",	ILLOP		},	/* x11xxxxx */
/* code $14, index 256+1*4 */
  { "ppo",	REGJR		},
  { "****",	ILLOP		},
  { "pfl",	REGJR		},
  { "****",	ILLOP		},
/* code $18, index 256+2*4 */
  { "rod",	REGJR		},
  { "rou",	REGJR		},
  { "bid",	REGJR		},
  { "biu",	REGJR		},
/* code $1A, index 256+3*4 */
  { "did",	REGJR		},
  { "diu",	REGJR		},
  { "byd",	REGJR		},
  { "byu",	REGJR		},
/* code $1B, index 256+4*4 */
  { "cmp",	REGJR		},
  { "****",	ILLOP		},
  { "inv",	REGJR		},
  { "****",	ILLOP		},
/* code $1C, index 256+5*4 */
  { "gpo",	REGJR		},
  { "****",	ILLOP		},
  { "gfl",	REGJR		},
  { "****",	ILLOP		},
/* code $54, index 256+6*4 */
  { "ppo",	IM8A		},
  { "****",	ILLOP		},
  { "pfl",	IM8A		},
  { "****",	ILLOP		},
/* code $92, index 256+7*4 */
  { "stlw",	REGJR		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
/* code $94, index 256+8*4 */
  { "ppow",	REGJR		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
/* code $98, index 256+9*4 */
  { "rodw",	REGJR		},
  { "rouw",	REGJR		},
  { "bidw",	REGJR		},
  { "biuw",	REGJR		},
/* code $9A, index 256+10*4 */
  { "didw",	REGJR		},
  { "diuw",	REGJR		},
  { "bydw",	REGJR		},
  { "byuw",	REGJR		},
/* code $9B, index 256+11*4 */
  { "cmpw",	REGJR		},
  { "****",	ILLOP		},
  { "invw",	REGJR		},
  { "****",	ILLOP		},
/* code $9C, index 256+12*4 */
  { "gpow",	REGJR		},
  { "****",	ILLOP		},
  { "gflw",	REGJR		},
  { "****",	ILLOP		},
/* code $CA, index 256+13*4 */
  { "adbm",	RIM5IM3JR	},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
/* code $CB, index 256+14*4 */
  { "sbbm",	RIM5IM3JR	},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
/* code $D3, index 256+15*4 */
  { "ldlm",	REGIM3		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
/* code $DA, index 256+16*4 */
  { "didm",	REGIM3		},
  { "dium",	REGIM3		},
  { "bydm",	REGIM3		},
  { "byum",	REGIM3		},
/* code $DB, index 256+17*4 */
  { "cmpm",	REGIM3		},
  { "****",	ILLOP		},
  { "invm",	REGIM3		},
  { "****",	ILLOP		},
/* code $13, index 256+18*4 */
  { "ldl",	REGJR		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
/* code $93, index 256+19*4 */
  { "ldlw",	REGJR		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
/* code $51, index 256+20*4 */
  { "ld",	REGIM8		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
/* code $D1, index 256+21*4 */
  { "ldw",	REGIM16		},
  { "****",	ILLOP		},
  { "****",	ILLOP		},
  { "****",	ILLOP		}
};


/* condition codes */
char *cctab[7] = { "z", "nc", "lz", "uz", "nz", "c", "nlz" };


/* 8-bit register names */
char *r8tab[2][4] = {
  { "pe", "pd", "ib", "ua" },
  { "ia", "ie", "??", "tm" }
};


/* 16-bit register names */
char *r16tab[2][4] = {
  { "ix", "iy", "iz", "us" },
  { "ss", "ky", "ky", "ky" }
};


/* specific register names */
char *sirtab[4] = { "sx", "sy", "sz", "??" };


uint FetchByte (void)
{
  uint x;
  x = head++;
  if ((head^x)/dsize != 0)
  {
    loc++;
  }
  return inbuf[x];
}


/* returns the index to the 'mnem' table */
uint ScanMnemTab (void)
{
  uint code;
  code = FetchByte();
  if (mnem[code].str[0] == '0')
  {
    code = mnem[code].kind + ((inbuf[head] >> 5) & 3);
  }
  return code;
}


void Imm3Arg (uint x)
{
  printf ("%u", ((x >> 5) & 7) + 1);
}


void Imm5Arg (uint x)
{
  printf ("&H%02X", x & 0x1F);
}


void Imm7Arg (void)
{
  uint x, y;
  y = loc;
  if (dsize > 1 && head == 2)
  {
    (void) FetchByte ();
  }
  x = FetchByte ();
  if ((x & 0x80) != 0)
    x = 0x80 - x;
  printf ("&H%04X", x + y);
}


void Imm8Arg (void)
{
  printf ("&H%02X", FetchByte ());
}


void Imm16Arg (void)
{
  uint x;
  x = FetchByte ();
  printf ("&H%02X%02X", FetchByte (), x);
}


void AbsArg (void)
{
  uint x;
  x = FetchByte ();
  if (dsize > 1)
  {
    (void) FetchByte ();
  }
  printf ("&H%02X%02X", FetchByte (), x);
}


void RegArg (uint x)
{
  printf ("$%u", x & 0x1F);
}


void SirArg (uint x)
{
  printf ("%s", sirtab[(x >> 5) & 3]);
}


void ShortRegArg (uint x)
{
  if ((x & 0x60) == 0x60)
  {
    RegArg(FetchByte ());
  }
  else
  {
    (void) putchar ('$');
    SirArg (x);
  }
}


void ShortRegAr1 (uint x, uint y)
{
  if ((x & 0x60) == 0x60)
  {
    RegArg (y);
  }
  else
  {
    (void) putchar ('$');
    SirArg (x);
  }
}


void IrArg (uint x)
{
  (void) putchar (((x & 1) == 0) ? 'x' : 'z');
}


void SignArg (uint x)
{
  (void) putchar (((x & 0x80) != 0) ? '-' : '+');
}


void OptionalJr (uint x)
{
  if ((x & 0x80) != 0)
  {
    printf (",jr ");
    Imm7Arg ();
  }
}


void Arguments (uint index)
{
  uint x, y;

  switch (mnem[index].kind)
  {
    case CC:
      printf ("%s", cctab[index & 7]);
      break;

    case JRCC:
      printf ("%s,", cctab[index & 7]);
      Imm7Arg ();
      break;

    case JPCC:
      printf ("%s,", cctab[index & 7]);
      AbsArg ();
      break;

    case JR:
      Imm7Arg ();
      break;

    case JP:
      AbsArg ();
      break;

    case REGREGJR:
      x = FetchByte ();
      RegArg (x);
      (void) putchar (',');
      ShortRegArg (x);
      OptionalJr (x);
      break;

    case REGDIRJR:
      x = FetchByte ();
      RegArg (x);
      printf (",(");
      ShortRegArg (x);
      (void) putchar (')');
      OptionalJr (x);
      break;

    case REGJR:
      x = FetchByte ();
      RegArg (x);
      OptionalJr (x);
      break;

    case REGIRR:
      x = FetchByte ();
      RegArg (x);
      printf (",(i");
      IrArg (index);
      SignArg (x);
      ShortRegArg (x);
      (void) putchar (')');
      break;

    case REGIRRIM3:
      x = FetchByte ();
      y = FetchByte ();
      RegArg (x);
      printf (",(i");
      IrArg (index);
      SignArg (x);
      ShortRegAr1 (x, y);
      printf ("),");
      Imm3Arg (y);
      break;

    case REG:
      RegArg (FetchByte ());
      break;

    case DIR:
      (void) putchar ('(');
      RegArg (FetchByte ());
      (void) putchar (')');
      break;

    case IRRREG:
      x = FetchByte ();
      printf ("(i");
      IrArg (index);
      SignArg (x);
      ShortRegArg (x);
      printf ("),");
      RegArg (x);
      break;

    case REGIM8JR:
      x = FetchByte ();
      RegArg (x);
      (void) putchar (',');
      Imm8Arg ();
      OptionalJr (x);
      break;

    case IM8:
      Imm8Arg ();
      break;

    case IM8A:
      (void) FetchByte();
      Imm8Arg ();
      break;

    case R8IM8:
      x = FetchByte ();
      printf ("%s,", r8tab[index & 1][(x >> 5) & 3]);
      Imm8Arg ();
      break;

    case REGIRI:
      x = FetchByte ();
      RegArg (x);
      printf (",(i");
      IrArg (index);
      SignArg (x);
      Imm8Arg ();
      (void) putchar (')');
      break;

    case IRIREG:
      x = FetchByte ();
      printf ("(i");
      IrArg (index);
      SignArg (x);
      Imm8Arg ();
      printf ("),");
      RegArg (x);
      break;

    case R8REGJR:
      x = FetchByte ();
      printf ("%s,", r8tab[index & 1][(x >> 5) & 3]);
      RegArg (x);
      OptionalJr (x);
      break;

    case R16REGJR:
      x = FetchByte ();
      printf ("%s,", r16tab[index & 1][(x >> 5) & 3]);
      RegArg (x);
      OptionalJr (x);
      break;

    case R16IM16:
      x = FetchByte ();
      printf ("%s,", r16tab[index & 1][(x >> 5) & 3]);
      Imm16Arg ();
      break;

    case IM8IND:
      x = FetchByte ();
      Imm8Arg ();
      printf (",($");
      SirArg (x);
      (void) putchar (')');
      break;

    case IM16IND:
      x = FetchByte ();
      Imm16Arg ();
      printf (",($");
      SirArg (x);
      (void) putchar (')');
      break;

    case RRIM3JR:
      x = FetchByte ();
      y = FetchByte ();
      RegArg (x);
      (void) putchar (',');
      ShortRegAr1 (x,y);
      (void) putchar (',');
      Imm3Arg (y);
      OptionalJr (x);
      break;

    case RIM5IM3JR:
      x = FetchByte ();
      y = FetchByte ();
      RegArg (x);
      (void) putchar (',');
      Imm5Arg (y);
      (void) putchar (',');
      Imm3Arg (y);
      OptionalJr (x);
      break;

    case REGIM8:
      x = FetchByte ();
      RegArg (x);
      (void) putchar (',');
      Imm8Arg ();
      break;

    case REGIM16:
      x = FetchByte ();
      RegArg (x);
      (void) putchar (',');
      Imm16Arg ();
      break;

    case REGIM3:
      x = FetchByte ();
      y = FetchByte ();
      RegArg (x);
      (void) putchar (',');
      Imm3Arg (y);
      break;

    case SIRREGJR:
      x = FetchByte ();
      SirArg (x);
      (void) putchar (',');
      RegArg (x);
      OptionalJr (x);
      break;

    case SIRREGIM3:
      x = FetchByte ();
      y = FetchByte ();
      SirArg (x);
      (void) putchar (',');
      RegArg (x);
      (void) putchar (',');
      Imm3Arg (y);
      break;

    case SIRIM5:
      x = FetchByte ();
      SirArg (x);
      printf (",%u", x & 0x1F);
      break;
 
    default:
      break;

  }
}


int main (int argc, char *argv[])
{
  int x;
  uint i;
  FILE *fp;

  if (argc < 2)
  {
    fprintf (stderr, "\nNo input file specified\n");
    return 1;
  }

  if ((fp = fopen(*++argv, "rb")) == NULL)
  {
    fprintf(stderr, "\nCannot open the input file %s\n", *argv);
    return 1;
  }

  loc = 0;
  dsize = 1;
  while (argc-->2)
  {
    argv++;
    if (isxdigit(**argv))
    {
      (void) sscanf(*argv, "%x", &loc);
    }
    else if (**argv == '/' || **argv == '-')
    {
      switch (*(*argv+1))
      {
        case 'b':
        case 'B':
          dsize = 1;
          break;
        case 'w':
        case 'W':
          dsize = 2;
          break;
        default:
          fprintf(stderr, "\nUnrecognised switch: %s\n", *argv+1);
          break;
      }
    }
    else
    {
      fprintf(stderr, "\nUnrecognised parameter: %s\n", *argv);
    }
  }
  head = 0;
  tail = 0;

  do {

/* align the "head" index */
    while (head % dsize != 0)
    {
      (void) FetchByte ();
    }
/* shift the bytes in the inbuf */
    i = 0;
    while (head<tail)
    {
      inbuf[i++] = inbuf[head++];
    }
    tail = i;
    head = 0;
/* fill the inbuf */
    while ((tail<INBUFSIZE) && ((x=getc(fp))!=EOF))
    {
      inbuf[tail++] = (uint) x;
    }
/* disassemble */
    printf ("%04X:\t", loc);
    i = ScanMnemTab ();
    printf ("%s", mnem[i].str);
    if (mnem[i].kind != NONE && mnem[i].kind != ILLOP)
    {
      (void) putchar ('\t');
    }
    Arguments (i);
    printf ("\n");

  } while (tail!=0);

  (void) fclose (fp);
  return 0;
}
