/* This program displays the contents of a tape image saved by the FX-700P
   calculator through the serial port. Only program and memo files supported.
*/

#include <stdio.h>
#include "bool.h"

/* error messages */
const char *err_msg[] = {
  "",						/* OK */
  "leader expected",				/* message #1 */
  "premature end of a header/data segment",	/* message #2 */
  "premature end of a file",			/* message #3 */
  "header segment expected",			/* message #4 */
  "unknown segment identifier",			/* message #5 */
  "unexpected program separator"		/* message #6 */
};

/* file types 0x9_ to 0xF_ */
const char *file_type[] = {
  "Memo",
  "Memo",		/* password protected */
  "All programs",	/* password protected */
  "Program",		/* password protected */
  "Program",
  "Variables",
  "All programs"
};

/* character codes 0x00 to 0x7F */
const char characters[] =
" +-*/^!\"#$> = < 0123456789. )( EABCDEFGHIJKLMNOPQRSTUVWXYZ      abcdefghijklmnopqrstuvwxyz  ?,;:o ' @x: <   u v>%  [&_'.]       ";

/* BASIC keywords 0x80 to 0xD1 */
const char *tokens[] = {
  "SIN ",     "COS ",     "TAN ",     "ASN ",
  "ACS ",     "ATN ",     "LOG ",     "LN ",
  "EXP ",     "SQR ",     "INT ",     "FRAC ",
  "ABS ",     "SGN ",     "RND(",     "RAN#",
  "LEN(",     "VAL(",     "MID(",     "KEY",
  "CSR ",     " TO ",     " STEP ",   " THEN ",
  "DEG(",     "HEX$(",    "STR$(",    "DM$(",
  "MID$(",    "ALL",      "BEEP",     "LET",
  "FOR ",     "NEXT ",    "GOTO ",    "GOSUB ",
  "RETURN",   "IF ",      "PRINT ",   "INPUT ",
  "MODE ",    "STOP",     "END",      "ON ",
  "READ ",    "RESTORE ", "DATA ",    "REM",
  "VAC",      "SET ",     "PUT ",     "GET ",
  "CLEAR ",   "",         "",         "",
  "POL(",     "HYP ",     "REC(",     "FACT ",
  "NPR(",     "EOX ",     "NCR(",     "EOY ",
  "DEFM ",    "SAVE ",    "LOAD ",    "VERIFY ",
  "LIST ",    "RUN ",     "NEW ",     "PASS ",
  "SRAM",     "LRAM",     "WRITE# ",  "STAT ",
  "",         "",         "",         "",
  "DIM ",     "ERASE "
};


void casioprint (int c)
{
  switch (c)
  {
    case 0x0B:
      printf (">=");
      break;
    case 0x0D:
      printf ("<=");        
      break;
    case 0x0F:
      printf ("<>");
      break;
    case 0x1B:
      printf ("PI");
      break;
    case 0x1E:
      printf ("E-");
      break;
    case 0xFE:
      printf (":");
      break;
    case 0xFF:
      printf ("\n");
      break;
    default:
      if (c <= 0x7F)
        printf ("%c", characters[c]);
      else if (c <= 0xD1)
        printf ("%s", tokens[c-0x80]);
      break;
  }
}


int bcd2bin (int c)
{
  int x;
  x = c/16;
  return c - 6*x;
}


/* returns 0 when OK or the error code when invalid data encountered */
int list (int c)
{
  static int skipped = 5;
  static int idle_counter = 0;
  static int data_counter = 0;
  static int line_counter = 0;
  static int line_number = 0;
  static int prog_number = 0;
  static int segment_id = 0;
  static int file_id = 0;
  static bool leader_expected = TRUE;
  static bool dataseg_expected = FALSE;
  static bool password_expected = FALSE;

/* skip first 5 bytes before the leader */
  if (skipped>0)
  {
    skipped--;
    return 0;
  }

/* handle the leader */
  if (c >= 0x100)	/* idle string (leader) */
  {
    if (idle_counter==0 && !leader_expected)
    {
      return 2;		/* "Premature end of a header/data segment" error */
    }
    if (idle_counter++ >= 12)	/* leader should contain at least 12 idle strings */
    {
      leader_expected = FALSE;	/* valid leader detected */
    }
    data_counter = 0;
    return 0;
  }

/* data byte received */
  idle_counter = 0;
  if (leader_expected)
  {
    return 1;		/* "Leader expected" error */
  }

/* handle the first segment byte */
  if (data_counter++ == 0)
  {
    segment_id = c;

    if (dataseg_expected)
    {
      if (c == 0x02)		/* data segment ? */
      {
        line_counter = 0;
        return 0;
      }
      return 3;			/* "Premature end of a file" error */
    }

/* header segment expected */
    if (c == 0x02)		/* data segment ? */
    {
      return 4;			/* "Header segment expected" error */
    }

    if ((file_id = c & 0xF0) >= 0x90)
    {
      prog_number = 0;
      printf ("\n\n%s: ", file_type[file_id/16 - 0x09]);
      password_expected = (file_id <= 0xC0 && file_id >= 0xA0);
      return 0;
    }

    return 5;			/* "Unknown segment identifier" error */
  }

/* handle subsequent segment bytes */

/* handle the data segment */
  if (segment_id == 0x02)	/* data segment ? */
  {
    line_counter++;

/* End Marker ? */
    if (c == 0xF0 || c == 0xF1)
    {
      leader_expected = TRUE;
      skipped = 5;
      dataseg_expected = (c == 0xF1);
      password_expected = FALSE;
      return 0;
    }

/* programs separator ? */
    if (c == 0xE0)
    {
      prog_number++;
      line_counter = 0;
      if ((file_id != 0xF0 && file_id != 0xB0) || prog_number > 10)
      {
        return 6;		/* "Unexpected program separator" error */
      }
      if (prog_number < 10)
      {
        printf ("\n P%d\n\n", prog_number);
      }
      return 0;
    }

/* password segment ? */
    if (password_expected)
    {
      if (data_counter == 1)
      {
        printf ("Password: ");
      }
      casioprint(c);
      return 0;
    }

/* memo file segment ? */
    if (file_id == 0x90 || file_id == 0xA0)
    {
      casioprint(c);
      return 0;
    }

/* don't display data segments containing variables */
    if (file_id == 0xE0)
    {
      return 0;
    }

/* the least significant byte of the line number ? */
    if (line_counter == 1)
    {
      line_number = bcd2bin(c);
      return 0;
    }

/* the most significant byte of the line number ? */
    if (line_counter == 2)
    {
      line_number += 100*bcd2bin(c);
      printf ("%d ", line_number);
      return 0;
    }

/* print remaining bytes of the data segment */
    if (c == 0xFF)
    {
      line_counter = 0;
    }
    casioprint(c);
    return 0;
  }

/* handle the header segment */

/* display the file name */
  if (data_counter <= ((segment_id+1) & 0x0F))
  {
    casioprint(c);
    return 0;
  }

/* skip remaining data */
  if (data_counter < 11)
  {
    return 0;
  }

/* last byte of the header segment */
  printf ("\n\n");
  if (file_id == 0xF0 || file_id == 0xB0)
  {
    printf (" P0\n\n");
  }
  leader_expected = TRUE;
  skipped = 5;
  dataseg_expected = TRUE;
  return 0;
}


int main(int argc, char *argv[])
{
  int c1;
  int counter;			/* character counter */
  int err_code;
  unsigned int x;		/* 12-bit word */
  FILE *infp;

  if (argc<=1)
  {
    fprintf (stderr, "\nMissing file name, program aborted\n");
    return 1;
  }

  if ((infp = fopen(*++argv, "rt")) == NULL)
  {
    fprintf (stderr, "\nCannot open the file %s\n",*argv);
    return 1;
  }

  x = 0;
  counter = 0;
  while ((c1 = getc(infp)) != EOF)
  {

/* skip invalid characters */
    if (c1<0x30 || c1>0x6F)
    {
      counter = 0;
    }

    else
    {
/* shift the received 6-bit data into the 12-bit word */
      x = (x>>6) & 0x003F;
      x |= (((unsigned int) (c1-0x30)) << 6);

      if (++counter == 2)	/* already 2 characters processed ? */
      {
        counter = 0;
        c1 = (int) ((x>>1) & 0xFF);	/* strip the start, stop and parity bits */
        if ((x & 0x01) != 0)
        {
          c1 = 0x100;		/* idle string (leader) */
        }

        if ((err_code = list(c1)) != 0)
        {
          printf ("\nInvalid data encountered - %s.\n", err_msg[err_code]);
          break;
        }
      }
    }
  }

  (void) fclose (infp);
  return 0;
}
