/* program for reading and writing MD-100 formatted disks */

#include <stdio.h>
#include <stdlib.h>	/* atexit */
#pragma inline

typedef unsigned char uchar;
typedef unsigned int uint;

/* disk parameters */
#define Cylinders 80
#define Heads 1
#define Sectors 16
#define SecBase 1
#define SecSize 256

/* disk commands */
#define ReadSector 2
#define WriteSector 3

/* sector buffer */
uchar buffer[SecSize];

/* error buffer */
#define MAXERRORS 5
uchar errtab[MAXERRORS][3];

#define MAXATTEMPTS 3

uchar oldsecsize;
uchar oldeot;
FILE *fp;
int errors;


/* change a disk parameter table entry */
uchar dpt (uint offset, uchar data)
{
  _AX = offset;
  _BL = data;
  asm {
	push	ds
	mov	cx,0
	mov	ds,cx
	mov	bp, word ptr ds:0078h
	add	bp,ax
	mov	ds, word ptr ds:007Ah
	mov	al, byte ptr ds:[BP]
	mov	byte ptr ds:[BP],bl
	pop	ds
  }
  return _AL;
}


/* issue a disk command */
uchar DiskCmd (uchar drive, uchar head, uchar cylinder, uchar sector, uchar cmd)
{
  _DL = drive;
  _DH = head;
  _CH = cylinder;
  _CL = sector;
  _AH = cmd;
  _AL = 1;		/* sector count */
  asm {
	mov	bx, offset buffer
	int	13h
  }
  return _AH;
}


/* display the list of errors, if any occured */
void statistics (void)
{
  int i;

  if (errors > 0)
  {
    printf ("\nFloppy disk access errors:\n");
    for (i=0; i<errors; i++)
    {
      printf ("T:%u\tH:%u\tS:%u\n", errtab[i][0], errtab[i][1], errtab[i][2]);
    }
  }
}


/* restore the disk parameter table entries */
void exit_fn (void)
{
  (void) dpt (3, oldsecsize);
  (void) dpt (4, oldeot);
  (void) fclose (fp);
  statistics ();
}


void usage ()
{
  printf ("\nThe program expects a command 'r' or 'w' and a file name.\n");
}


int main (int argc, char **argv)
{
  char cmd;
  int attempt;
  int status;
  uchar he, cy, se;

  if (argc < 3)
  {
    usage ();
    return 1;
  }

  cmd = **++argv;
  if (cmd != 'r' && cmd != 'w')
  {
    usage ();
    return 1;
  }

  if ((fp = fopen(*++argv, cmd == 'r' ? "wb" : "rb")) == NULL)
  {
    printf ("\nCannot open the file %s\n", *argv);
    return 1;
  }

  atexit (exit_fn);
  oldsecsize = dpt (3, 1);	/* change sector size to 256 */
  oldeot = dpt (4, SecBase+Sectors-1);

  errors = 0;
  printf ("Track: ");

  for (cy=0; cy<Cylinders; cy++)
  {
    printf ("%02d\b\b",cy);
    for (he=0; he<Heads; he++)
    {
      for (se=0; se<Sectors; se++)
      {

        if (cmd == 'w')
        {
          if (fread (buffer, sizeof(uchar), (size_t) SecSize, fp) !=
			(size_t) SecSize)
          {
            printf ("\nUnexpected end of file %s\n", *argv);
            return 1;
          }
        }

        for (attempt=0; attempt<MAXATTEMPTS; attempt++)
        {
          if ((status = (int) DiskCmd (0, he, cy, se+SecBase,
			cmd == 'r' ? ReadSector : WriteSector)) == 0)
            break;
        }

        if (status != 0)
        {
          errtab[errors][0] = cy;
          errtab[errors][1] = he;
          errtab[errors][2] = se;
          if (++errors >= MAXERRORS)
          {
            printf ("\nTo many errors, program aborted!");
            return 1;
          }
        }

        if (cmd == 'r')
        {
          if (fwrite (buffer, sizeof(uchar), (size_t) SecSize, fp) !=
			(size_t) SecSize)
          {
            printf ("\nFile %s write error\n", *argv);
            return 1;
          }
        }

      }
    }
  }

  return 0;
}
