; Replacement memory card for the MK-90 microcomputer
; osc. frequency 9.6MHz

; Fuses:
; BODLEVEL=01 - Brown out detector trigger level 2.7V
; RSTDISBL unprogrammed (1) - External Reset enabled
; WDTON unprogrammed (1) - WDT enabled by WDTCR
; CKDIV8 unprogrammed (1) - clock not divided
; SUT=10 - long start-up time 64ms
; CKSEL=10 - Internal Calibrated RC Oscillator 9.6MHz

.include "tn13def.inc"

; Port B pin functions:
.equ	DATA		=0	;card bus signal, bit number chosen intentionally
.equ	SELECT		=1	;INT0, card bus signal
.equ	CLOCK		=2	;PCINT2, card bus signal
.equ	SCL		=3	;FRAM bus signal
.equ	SDA		=4	;FRAM bus signal, bit number chosen intentionally

.def	Const2		=r2	;constant value 0x00
.def	Const3		=r3	;constant value (1<<TOV0)
.def	Const4		=r4	;constant value (1<<CS02)+(0<<CS01)+(0<<CS00)

.def	BufPtr1		=r12
.def	BufPtr2		=r13
.def	FramVector	=r14
.def	IntSregSave	=r15	;stores SREG
.def	IntTemp		=r16	;temporary variable used in interrupts
.def	Temp		=r17	;local temporary variable
.def	CardData	=r18	;byte transferred through the card bus
.def	DataByte	=r19
.def	PortbCopy	=r20

.def	Flags		=r21
; meaning of individual bits:
.equ	CMD_VALID	=7
.equ	CMD_COMPLETED	=6

.def	FramAddrL	=r22
.def	FramAddrH	=r23
.def	CardAddrL	=r24
.def	CardAddrH	=r25

; X register - pointer to buffer, used in FRAM bus routines
; Y register - pointer to buffer, used in card bus routines
; Z register - used to walk through the PCINT handle code


.DSEG

; circular data transfer buffer

.org	0x0080			;address chosen intentionally

.equ	BUFFER_SIZE	=16	;must be power of two
.equ	BUFFER_DISTANCE	=2	;minimal distance between head and tail,
				;must be >= 2

buffer:
.byte	BUFFER_SIZE


.CSEG


.MACRO	card_wrbit
	in	PortbCopy,PINB
	sbic	PINB,CLOCK
	reti
; falling CLOCK edge detected, sense DATA line
	in	IntSregSave,SREG
	ror	PortbCopy	;DATA bit goes to Carry
	rol	CardData
.ENDMACRO


.MACRO	card_rdbit
	sbic	PINB,CLOCK
	rjmp	PC+3
; falling CLOCK edge detected, change DATA line
	out	PORTB,PortbCopy
	reti
; rising CLOCK edge detected, prepare data to be sent to PORTB
	in	IntSregSave,SREG
	clr	PortbCopy
	lsl	CardData
.ENDMACRO


.org	0x0000

		rjmp	reset		;Reset Handler
		rjmp	ext_int0	;IRQ0 Handler
		ijmp			;PCINT0 Handler
		rjmp	tim0_ovf	;Timer0 Overflow Handler
		reti			;EEPROM Ready Handler
		reti			;Analog Comparator Handler
		reti			;Timer0 CompareA Handler
		reti			;Timer0 CompareB Handler
		reti			;Watchdog Interrupt Handler
		reti			;ADC Conversion Handler


; Part of code that has to stay within a single page of the program memory,
; so it can be addressed with only 8 bits of the Z register.

; these tables could be placed in the EEPROM
card_vectors:
.DB		low (card_cmd0), low (card_rddec)
.DB		low (card_wrdec), low (card_idle)
.DB		low (card_idle), low (card_idle)
.DB		low (card_idle), low (card_idle)
.DB		low (card_idle), low (card_idle)
.DB		low (card_wraddrh), low (card_rdaddr)
.DB		low (card_wrinc), low (card_rdinc)
.DB		low (card_wrdec), low (card_idle)

fram_vectors:
.DB		low (fram_idle), low (fram_rddec)
.DB		low (fram_wrdec), low (fram_idle)
.DB		low (fram_idle), low (fram_idle)
.DB		low (fram_idle), low (fram_idle)
.DB		low (fram_idle), low (fram_idle)
.DB		low (fram_wraddr), low (fram_idle)
.DB		low (fram_wrinc), low (fram_rdinc)
.DB		low (fram_wrdec), low (fram_idle)


fram_idle:	mov	ZL,Temp
		sei
		rjmp	main_loop

fram_rddec:	mov	ZL,Temp
		sei
		rjmp	main_loop

fram_wrdec:	mov	ZL,Temp
		sei
		mov	BufPtr2,XL
		rjmp	fram_wrdec1

fram_wraddr:	mov	ZL,Temp
		sei
fram_wraddr1:	sbrs	Flags,CMD_COMPLETED
		rjmp	fram_wraddr1
		movw	XH:XL,YH:YL
		movw	FramAddrH:FramAddrL,CardAddrH:CardAddrL
		rjmp	main_loop

fram_wrinc:	mov	ZL,Temp
		sei
		rjmp	fram_wrinc1

fram_rdinc:	mov	ZL,Temp
		sei
		rjmp	fram_rdinc1


; PCINT handle routines, used to handle the card bus traffic,
; called at each edge of the CLOCK pulse.

; idle
card_idle:	reti


; command sent to the card
card_cmd:	card_wrbit
		brcc	card_cmd_1
; 4 bits received
		ldi	ZL,low (fram_vectors<<1)
		add	ZL,CardData
		lpm	FramVector,z
		sbr	Flags,(1<<CMD_VALID)
		ldi	PortbCopy,4		;misused as bit counter
		ldi	ZL,low (card_skip)
card_cmd_1:	out	SREG,IntSregSave
		reti

; skip last 4 bits of the command sent to the card
card_skip:	sbic	PINB,CLOCK
		reti
; falling CLOCK edge detected
		in	IntSregSave,SREG
		dec	PortbCopy		;misused as bit counter
		brne	card_skip_1
; 4 bits skipped
		ldi	ZL,low (card_vectors<<1)
		add	ZL,CardData
		lpm	ZL,z
		ldi	CardData,0x01
card_skip_1:	out	SREG,IntSregSave
		reti


; address written to the card, most significant byte
card_wraddrh:	card_wrbit
		brcc	card_wraddrh_1
; 8 bits received
		mov	CardAddrH,CardData
		ldi	CardData,0x01
		ldi	ZL,low (card_wraddrl)
card_wraddrh_1:	out	SREG,IntSregSave
		reti

; address written to the card, least significant byte
card_wraddrl:	card_wrbit
		brcc	card_wraddrl_1
; 8 bits received
		mov	CardAddrL,CardData
		ldi	ZL,low (card_idle)
card_wraddrl_1:	out	SREG,IntSregSave
		reti


; data written to the card, postincrement address mode
card_wrinc:	card_wrbit
		brcc	card_wrinc_1
; 8 bits received
		st	y+,CardData
		andi	YL,low (buffer+BUFFER_SIZE-1)
		adiw	CardAddrL,1
		ldi	CardData,0x01
card_wrinc_1:	out	SREG,IntSregSave
		reti


; data written to the card, postdecrement address mode
card_wrdec:	card_wrbit
		brcc	card_wrdec_1
; 8 bits received
		st	y+,CardData
		andi	YL,low (buffer+BUFFER_SIZE-1)
		sbiw	CardAddrL,1
		ldi	CardData,0x01
card_wrdec_1:	out	SREG,IntSregSave
		reti


; data read from the card, postincrement address mode
; first rising CLOCK edge
card_rdinc:	in	IntSregSave,SREG
		ldi	ZL,low (card_rdinc_1)
		sbi	DDRB,DATA
		rjmp	card_rdinc_2

; all next CLOCK edges
card_rdinc_1:	card_rdbit
		brne	card_rdinc_3
		adiw	CardAddrL,1
; 8 bits sent, fetch next byte from buffer
card_rdinc_2:	ld	CardData,y+
		andi	YL,low (buffer+BUFFER_SIZE-1)
		sec
		rol	CardData
card_rdinc_3:	rol	PortbCopy	;Carry goes to DATA bit
		out	SREG,IntSregSave
		reti


; data read from the card, postdecrement address mode (not implemented)
card_rddec:	reti


; address read from the card
; first rising CLOCK edge
card_rdaddr:	in	IntSregSave,SREG
		ldi	ZL,low (card_rdaddr_1)
		mov	CardData,CardAddrH
		sbi	DDRB,DATA
		rjmp	card_rdaddr_2

; all next CLOCK edges
card_rdaddr_1:	card_rdbit
		brne	card_rdaddr_3
; 8 bits sent, next address byte
		mov	CardData,CardAddrL
card_rdaddr_2:	sec
		rol	CardData
card_rdaddr_3:	rol	PortbCopy	;Carry goes to DATA bit
		out	SREG,IntSregSave
		reti


; unknown command 0x00, returns 0x00
; first rising CLOCK edge
card_cmd0:	in	IntSregSave,SREG
		ldi	ZL,low (card_rdaddr_1)
		ldi	CardData,0x00
		sbi	DDRB,DATA
		rjmp	card_rdaddr_2

.IF	PC>0x00FF
.ERROR	"Page boundary crossed"
.ENDIF

; End of code that isn't allowed to cross a page boundary.


ext_int0:	sbis	PINB,SELECT
		rjmp	ext_int0_1
; high level on SELECT pin detected
		cbi	DDRB,DATA
		out	TCCR0B,Const4	;start the Timer0
		ldi	ZL,low (card_idle)
		in	IntSregSave,SREG
		sbr	Flags,(1<<CMD_COMPLETED)
		out	SREG,IntSregSave
		reti
; low level on SELECT pin detected
ext_int0_1:	ldi	CardData,0x10
		out	TCCR0B,Const2	;stop the Timer0
		out	TCNT0,Const2	;clear the time-out counter
		out	TIFR0,Const3	;clear pending interrupt flag
		ldi	ZL,low (card_cmd)
		reti


; the Timer0 overflow puts the microprocessor to sleep
tim0_ovf:	ldi	IntTemp,(1<<PUD)+(1<<SE)+(1<<SM1)+(0<<SM0)+(0<<ISC01)+(1<<ISC00)
		out	MCUCR,IntTemp
		ldi	IntTemp,(1<<PUD)+(0<<SE)+(1<<SM1)+(0<<SM0)+(0<<ISC01)+(1<<ISC00)
		sei
		sleep
		cli
		out	MCUCR,IntTemp
		reti


reset:		cli
		ldi	Temp,0x7F
		out	SPL,Temp
		ldi	Temp,0
		out	DDRB,Temp
		out	PORTB,Temp
		out	MCUSR,Temp
		out	WDTCR,Temp
		out	TCNT0,Temp
		mov	Const2,Temp
		ldi	Temp,(1<<TOV0)
		mov	Const3,Temp
		ldi	Temp,(1<<CS02)+(0<<CS01)+(0<<CS00)
		mov	Const4,Temp
		ldi	XL,low (buffer)
		ldi	XH,high (buffer)
		movw	YH:YL,XH:XL
		ldi	ZL,low (card_idle)
		ldi	ZH,high (card_idle)
		ldi	CardAddrL,0
		ldi	CardAddrH,0
		movw	FramAddrH:FramAddrL,CardAddrH:CardAddrL
		ldi	Flags,0

; pull-ups disabled, sleep mode: power down,
; any logical change on INT0 generates an interrupt request
		ldi	Temp,(1<<PUD)+(0<<SE)+(1<<SM1)+(0<<SM0)+(0<<ISC01)+(1<<ISC00)
		out	MCUCR,Temp
		ldi	Temp,(1<<CLOCK)
		out	PCMSK,Temp
		out	TCCR0B,Const4	;select prescaler 256
		ldi	Temp,(1<<INT0)+(1<<PCIE)
		out	GIMSK,Temp
		ldi	Temp,(1<<TOIE0)
		out	TIMSK0,Temp	;Timer0 overflow interrupt enable
		sei

; read a byte from the FRAM to the buffer in advance
main_loop:	rcall	fram_addr
		cbi	DDRB,SCL	;SCL=1
		rjmp	PC+1
		sbi	DDRB,SCL	;SCL=0
		nop
		cbi	DDRB,SDA	;SDA=1
		mov	DataByte,FramAddrH
		cbi	DDRB,SCL	;SCL=1
		swap	DataByte
		andi	DataByte,0x0C
		ori	DataByte,0xA1
		sbi	DDRB,SDA	;SDA=0 (start condition)
		rcall	fram_send
		sbi	DDRB,SCL	;SCL=0
		nop
		cbi	DDRB,SDA	;SDA=1 (receive ACK)
		nop
		cbi	DDRB,SCL	;SCL=1
		rjmp	PC+1
		sbi	DDRB,SCL	;SCL=0
		rcall	fram_read
		cbi	DDRB,SCL	;SCL=1 (send NOACK)
		st	x,DataByte
		rcall	fram_stop1

main_wait:	sbrs	Flags,CMD_VALID
		rjmp	main_wait
		cbr	Flags,(1<<CMD_VALID)+(1<<CMD_COMPLETED)
		cli
		mov	Temp,ZL
		mov	ZL,FramVector
		ijmp


; FRAM bus routines

fram_stop:
; send NACK or read ACK, expected SDA=1
		cbi	DDRB,SCL	;SCL=1
		rjmp	PC+1
; send stop condition
fram_stop1:	sbi	DDRB,SCL	;SCL=0
		nop
fram_stop2:	sbi	DDRB,SDA	;SDA=0
		nop
		cbi	DDRB,SCL	;SCL=1
		nop
		cbi	DDRB,SDA	;SDA=1
		ret


; send DataByte throgh the FRAM bus, returns with SCL=1
fram_send:	sec
		rol	DataByte
fram_send_loop:	sbi	DDRB,SCL	;SCL=0
		brcs	fram_send_1
		sbi	DDRB,SDA	;SDA=0
		rjmp	fram_send_2
fram_send_1:	cbi	DDRB,SDA	;SDA=1
fram_send_2:	cbi	DDRB,SCL	;SCL=1
		lsl	DataByte
		brne	fram_send_loop
		ret


; send FRAM device address and FramAddr bytes through the FRAM bus
fram_addr:	sbi	DDRB,SDA	;SDA=0 (start condition)
		mov	DataByte,FramAddrH
		swap	DataByte
		andi	DataByte,0x0E
		ori	DataByte,0xA0
		rcall	fram_send
		sbi	DDRB,SCL	;SCL=0
		nop
		cbi	DDRB,SDA	;SDA=1 (receive ACK)
		mov	DataByte,FramAddrH
		cbi	DDRB,SCL	;SCL=1
		rcall	fram_send
		sbi	DDRB,SCL	;SCL=0
		nop
		cbi	DDRB,SDA	;SDA=1 (receive ACK)
		mov	DataByte,FramAddrL
		cbi	DDRB,SCL	;SCL=1
		rcall	fram_send
		sbi	DDRB,SCL	;SCL=0
		cbi	DDRB,SDA	;SDA=1 (receive ACK)
		ret


; read DataByte from the FRAM bus, expects SCL=0 and SDA=1
fram_read:	ldi	DataByte,0x01
fram_read_loop:	cbi	DDRB,SCL	;SCL=1
; read SDA to Carry
		in	Temp,PINB
		swap	Temp		;SDA=Temp,0
		sbi	DDRB,SCL	;SCL=0
		ror	Temp		;SDA goes to Carry
		rol	DataByte
		brcc	fram_read_loop
		ret


; data read from the FRAM, postincrement address mode
fram_rdinc0:	rcall	fram_stop	;send NACK and the stop condition
fram_rdinc1:	rcall	fram_addr
		cbi	DDRB,SCL	;SCL=1
		rjmp	PC+1
		sbi	DDRB,SCL	;SCL=0
		nop
		cbi	DDRB,SDA	;SDA=1
		mov	DataByte,FramAddrH
		cbi	DDRB,SCL	;SCL=1
		swap	DataByte
		andi	DataByte,0x0E
		ori	DataByte,0xA1
		sbi	DDRB,SDA	;SDA=0 (start condition)
		rcall	fram_send
		sbi	DDRB,SCL	;SCL=0
		nop
		cbi	DDRB,SDA	;SDA=1 (input)
		nop
		cbi	DDRB,SCL	;SCL=1 (receive ACK)
		rjmp	PC+1
		sbi	DDRB,SCL	;SCL=0
fram_rdinc3:	rcall	fram_read

; stop filling the buffer when X is BUFFER_DISTANCE steps before Y
fram_rdinc4:	sbrc	Flags,CMD_COMPLETED
		rjmp	fram_rdinc5
		mov	Temp,YL
		sub	Temp,XL
		cpi	Temp,BUFFER_DISTANCE
		breq	fram_rdinc4
		cpi	Temp,256-(BUFFER_SIZE-BUFFER_DISTANCE)
		breq	fram_rdinc4

		st	x+,DataByte
		andi	XL,low (buffer+BUFFER_SIZE-1)
		subi	FramAddrL,0xFF
		sbci	FramAddrH,0xFF

; does FramAddr point to the same FRAM chip?
		mov	Temp,FramAddrH
		andi	Temp,0x1F
		or	Temp,FramAddrL
		breq	fram_rdinc0	;next FRAM chip
; still the same FRAM chip
		sbi	DDRB,SDA	;SDA=0 (send ACK)
		cbi	DDRB,SCL	;SCL=1
		rjmp	PC+1
		sbi	DDRB,SCL	;SCL=0
		cbi	DDRB,SDA	;SDA=1 (input)
		rjmp	fram_rdinc3

; command completed, terminate the FRAM transfer
fram_rdinc5:	movw	XH:XL,YH:YL
		movw	FramAddrH:FramAddrL,CardAddrH:CardAddrL
		rcall	fram_stop	;send NACK and the stop condition
		rjmp	main_loop


; data written to the FRAM, postincrement address mode
fram_wrinc0:	rcall	fram_stop2
fram_wrinc1:	rcall	fram_addr
		cbi	DDRB,SCL	;SCL=1 (receive ACK)
		rjmp	PC+1
		sbi	DDRB,SCL	;SCL=0
		rjmp	fram_wrinc4

fram_wrinc3:	sbrc	Flags,CMD_COMPLETED
		rjmp	fram_wrinc5
fram_wrinc4:	cp	XL,YL
		breq	fram_wrinc3

		ld	DataByte,x+
		rcall	fram_send
		sbi	DDRB,SCL	;SCL=0
		andi	XL,low (buffer+BUFFER_SIZE-1)
		subi	FramAddrL,0xFF
		cbi	DDRB,SDA	;SDA=1 (input)
		sbci	FramAddrH,0xFF
; does FramAddr point to the same FRAM chip?
		mov	Temp,FramAddrH
		cbi	DDRB,SCL	;SCL=1 (receive ACK)
		andi	Temp,0x1F
		or	Temp,FramAddrL
		sbi	DDRB,SCL	;SCL=0
		brne	fram_wrinc4	;still the same FRAM chip
		rjmp	fram_wrinc0	;next FRAM chip

; command completed, terminate the FRAM transfer
fram_wrinc5:	movw	XH:XL,YH:YL
		movw	FramAddrH:FramAddrL,CardAddrH:CardAddrL
		rcall	fram_stop2
		rjmp	main_loop


; data written to the FRAM, postdecrement address mode
fram_wrdec0:	rcall	fram_stop1
fram_wrdec1:	mov	BufPtr1,BufPtr2
		rjmp	fram_wrdec3

fram_wrdec2:	sbrc	Flags,CMD_COMPLETED
		rjmp	fram_wrinc5
fram_wrdec3:	cp	BufPtr2,YL
		breq	fram_wrdec2
		movw	XH:XL,YH:YL
		movw	FramAddrH:FramAddrL,CardAddrH:CardAddrL
		mov	BufPtr2,XL

fram_wrdec4:	rcall	fram_addr
		cbi	DDRB,SCL	;SCL=1 (receive ACK)
fram_wrdec5:	dec	XL
		andi	XL,low (BUFFER_SIZE-1)
		ori	XL,low (buffer)
		sbi	DDRB,SCL	;SCL=0
		ld	DataByte,x
		rcall	fram_send
		sbi	DDRB,SCL	;SCL=0
		subi	FramAddrL,0xFF
		sbci	FramAddrH,0xFF
		cbi	DDRB,SDA	;SDA=1 (input)
		cp	BufPtr1,XL
		cbi	DDRB,SCL	;SCL=1 (receive ACK)
		breq	fram_wrdec0	;no more data to write
; does FramAddr point to the same FRAM chip?
		mov	Temp,FramAddrH
		andi	Temp,0x1F
		or	Temp,FramAddrL
		brne	fram_wrdec5	;still the same FRAM chip
; next FRAM chip
		rcall	fram_stop1
		rjmp	fram_wrdec4
