/* disassembler Casio uPD1007 */

#include <stdio.h>
#include <ctype.h>	/* isxdigit */

typedef unsigned int uint;

/* global variables */
uint loc;		/* location pointer */
#define INBUFSIZE 8
uint inbuf[INBUFSIZE];
uint head, tail;	/* indexes to the inbuf */

typedef enum {
  NONE,		/* no arguments */
  DROP1,	/* two-byte instruction, no arguments */
  IM16,		/* 16-bit immediate value */
  WRIM16,	/* Word Register, 16-bit immediate value */
  IRIM8,	/* Index Register, 8-bit immediate value */
  CCIM16,	/* Condition code, 16-bit immediate value */
  MEMOIM8,	/* (IR), Immediate byte */
  SMEMOIM8,	/* +/-(IR), Immediate byte */
  XREG,
  YREG,
  REGIM8,	/* Rd,Im8 */
  XARY,
  YARY,
  ARYIM6,
  MEMOREG,	/* +/-(IR),Rs */
  REG,		/* Rd */
  OFFSREG,	/* (IR+/-Ri),Rs or (IR+/-nn),Rs */
  REGMEMO,	/* Rd,(IR)+/- */
  REGOFFS,	/* Rd,(IR+/-Ri) or Rd,(IR+/-nn) */
  MEMOARY,	/* +/-(IR),Rsf..Rsl */
  OFFSARY,	/* (IR+/-Ri),Rsf..Rsl or (IR+/-nn),Rsf..Rsl */
  ARY,		/* Rdf..Rdl */
  ARYMTBM,	/* CMPBM/MTBM Rdf..Rdl */
  ARYMEMO,	/* Rdf..Rdl,(IR)+/- */
  ARYOFFS,	/* Rdf..Rdl,(IR+/-Ri) or Rdf..Rdl,(IR+/-nn) */
  STREG,	/* status register, data register */
  STIM8,	/* status register, Im8 */
  REGST,	/* data register, status register */
  REGIF,	/* Rd,IF */
  REGKI,	/* Rd,KI */
  CCINDIR,
  INDIR,
  DATA1
} T_KINDS;

struct tab {
  char *str;
  T_KINDS kind;
};


struct tab mnem[] = {
/* bit 7 of the second byte cleared */
  { "adb",	XREG		},	/* code 0x00 */
  { "sbb",	XREG		},	/* code 0x01 */
  { "ad",	XREG		},	/* code 0x02 */
  { "sb",	XREG		},	/* code 0x03 */
  { "an",	XREG		},	/* code 0x04 */
  { "na",	XREG		},	/* code 0x05 */
  { "or",	XREG		},	/* code 0x06 */
  { "xr",	XREG		},	/* code 0x07 */
  { "adb",	YREG		},	/* code 0x08 */
  { "sbb",	YREG		},	/* code 0x09 */
  { "ad",	YREG		},	/* code 0x0A */
  { "sb",	YREG		},	/* code 0x0B */
  { "an",	YREG		},	/* code 0x0C */
  { "na",	YREG		},	/* code 0x0D */
  { "or",	YREG		},	/* code 0x0E */
  { "xr",	YREG		},	/* code 0x0F */
  { "adbm",	XARY		},	/* code 0x10 */
  { "sbbm",	XARY		},	/* code 0x11 */
  { "adm",	XARY		},	/* code 0x12 */
  { "sbm",	XARY		},	/* code 0x13 */
  { "anm",	XARY		},	/* code 0x14 */
  { "nam",	XARY		},	/* code 0x15 */
  { "orm",	XARY		},	/* code 0x16 */
  { "xrm",	XARY		},	/* code 0x17 */
  { "adbm",	YARY		},	/* code 0x18 */
  { "sbbm",	YARY		},	/* code 0x19 */
  { "adm",	YARY		},	/* code 0x1A */
  { "sbm",	YARY		},	/* code 0x1B */
  { "anm",	YARY		},	/* code 0x1C */
  { "nam",	YARY		},	/* code 0x1D */
  { "orm",	YARY		},	/* code 0x1E */
  { "xrm",	YARY		},	/* code 0x1F */
  { "tadb",	XREG		},	/* code 0x20 */
  { "tsbb",	XREG		},	/* code 0x21 */
  { "tad",	XREG		},	/* code 0x22 */
  { "tsb",	XREG		},	/* code 0x23 */
  { "tan",	XREG		},	/* code 0x24 */
  { "tna",	XREG		},	/* code 0x25 */
  { "tor",	XREG		},	/* code 0x26 */
  { "txr",	XREG		},	/* code 0x27 */
  { "tadb",	YREG		},	/* code 0x28 */
  { "tsbb",	YREG		},	/* code 0x29 */
  { "tad",	YREG		},	/* code 0x2A */
  { "tsb",	YREG		},	/* code 0x2B */
  { "tan",	YREG		},	/* code 0x2C */
  { "tna",	YREG		},	/* code 0x2D */
  { "tor",	YREG		},	/* code 0x2E */
  { "txr",	YREG		},	/* code 0x2F */
  { "tadbm",	XARY		},	/* code 0x30 */
  { "tsbbm",	XARY		},	/* code 0x31 */
  { "tadm",	XARY		},	/* code 0x32 */
  { "tsbm",	XARY		},	/* code 0x33 */
  { "tanm",	XARY		},	/* code 0x34 */
  { "tnam",	XARY		},	/* code 0x35 */
  { "torm",	XARY		},	/* code 0x36 */
  { "txrm",	XARY		},	/* code 0x37 */
  { "tadbm",	YARY		},	/* code 0x38 */
  { "tsbbm",	YARY		},	/* code 0x39 */
  { "tadm",	YARY		},	/* code 0x3A */
  { "tsbm",	YARY		},	/* code 0x3B */
  { "tanm",	YARY		},	/* code 0x3C */
  { "tnam",	YARY		},	/* code 0x3D */
  { "torm",	YARY		},	/* code 0x3E */
  { "txrm",	YARY		},	/* code 0x3F */
  { "ldw",	WRIM16		},	/* code 0x40 */
  { "ldw",	WRIM16		},	/* code 0x41 */
  { "ldw",	WRIM16		},	/* code 0x42 */
  { "ldw",	WRIM16		},	/* code 0x43 */
  { "ldw",	WRIM16		},	/* code 0x44 */
  { "ldw",	WRIM16		},	/* code 0x45 */
  { "ldw",	WRIM16		},	/* code 0x46 */
  { "ldw",	WRIM16		},	/* code 0x47 */
  { "adw",	IRIM8		},	/* code 0x48 */
  { "adw",	IRIM8		},	/* code 0x49 */
  { "adw",	IRIM8		},	/* code 0x4A */
  { "adw",	IRIM8		},	/* code 0x4B */
  { "sbw",	IRIM8		},	/* code 0x4C */
  { "sbw",	IRIM8		},	/* code 0x4D */
  { "sbw",	IRIM8		},	/* code 0x4E */
  { "sbw",	IRIM8		},	/* code 0x4F */
  { "ijmp",	INDIR		},	/* code 0x50 */
  { "ijmp",	CCINDIR		},	/* code 0x51 */
  { "ijmp",	CCINDIR		},	/* code 0x52 */
  { "ijmp",	CCINDIR		},	/* code 0x53 */
  { "ijmp",	CCINDIR		},	/* code 0x54 */
  { "ijmp",	CCINDIR		},	/* code 0x55 */
  { "ijmp",	CCINDIR		},	/* code 0x56 */
  { "ijmp",	CCINDIR		},	/* code 0x57 */
  { "rtn",	NONE		},	/* code 0x58 */
  { "ijmp",	CCINDIR		},	/* code 0x59 */
  { "ijmp",	CCINDIR		},	/* code 0x5A */
  { "ijmp",	CCINDIR		},	/* code 0x5B */
  { "ijmp",	CCINDIR		},	/* code 0x5C */
  { "ijmp",	CCINDIR		},	/* code 0x5D */
  { "ijmp",	CCINDIR		},	/* code 0x5E */
  { "ijmp",	CCINDIR		},	/* code 0x5F */
  { "cal",	IM16		},	/* code 0x60 */
  { "cal",	CCIM16		},	/* code 0x61 */
  { "cal",	CCIM16		},	/* code 0x62 */
  { "cal",	CCIM16		},	/* code 0x63 */
  { "cal",	CCIM16		},	/* code 0x64 */
  { "cal",	CCIM16		},	/* code 0x65 */
  { "cal",	CCIM16		},	/* code 0x66 */
  { "cal",	CCIM16		},	/* code 0x67 */
  { "cani",	NONE		},	/* code 0x68 */
  { "cal",	CCIM16		},	/* code 0x69 */
  { "cal",	CCIM16		},	/* code 0x6A */
  { "cal",	CCIM16		},	/* code 0x6B */
  { "cal",	CCIM16		},	/* code 0x6C */
  { "cal",	CCIM16		},	/* code 0x6D */
  { "cal",	CCIM16		},	/* code 0x6E */
  { "cal",	CCIM16		},	/* code 0x6F */
  { "jmp",	IM16		},	/* code 0x70 */
  { "jmp",	CCIM16		},	/* code 0x71 */
  { "jmp",	CCIM16		},	/* code 0x72 */
  { "jmp",	CCIM16		},	/* code 0x73 */
  { "jmp",	CCIM16		},	/* code 0x74 */
  { "jmp",	CCIM16		},	/* code 0x75 */
  { "jmp",	CCIM16		},	/* code 0x76 */
  { "jmp",	CCIM16		},	/* code 0x77 */
  { "rti",	NONE		},	/* code 0x78 */
  { "jmp",	CCIM16		},	/* code 0x79 */
  { "jmp",	CCIM16		},	/* code 0x7A */
  { "jmp",	CCIM16		},	/* code 0x7B */
  { "jmp",	CCIM16		},	/* code 0x7C */
  { "jmp",	CCIM16		},	/* code 0x7D */
  { "jmp",	CCIM16		},	/* code 0x7E */
  { "jmp",	CCIM16		},	/* code 0x7F */
  { "st",	MEMOREG		},	/* code 0x80 */
  { "st",	MEMOREG		},	/* code 0x81 */
  { "st",	MEMOREG		},	/* code 0x82 */
  { "st",	MEMOREG		},	/* code 0x83 */
  { "st",	MEMOREG		},	/* code 0x84 */
  { "st",	MEMOREG		},	/* code 0x85 */
  { "st",	MEMOREG		},	/* code 0x86 */
  { "st",	MEMOREG		},	/* code 0x87 */
  { "st",	MEMOREG		},	/* code 0x88 */
  { "st",	MEMOREG		},	/* code 0x89 */
  { "st",	MEMOREG		},	/* code 0x8A */
  { "st",	MEMOREG		},	/* code 0x8B */
  { "st",	MEMOREG		},	/* code 0x8C */
  { "st",	MEMOREG		},	/* code 0x8D */
  { "st",	MEMOREG		},	/* code 0x8E */
  { "st",	MEMOREG		},	/* code 0x8F */
  { "stm",	MEMOARY		},	/* code 0x90 */
  { "stm",	MEMOARY		},	/* code 0x91 */
  { "stm",	MEMOARY		},	/* code 0x92 */
  { "stm",	MEMOARY		},	/* code 0x93 */
  { "stm",	MEMOARY		},	/* code 0x94 */
  { "stm",	MEMOARY		},	/* code 0x95 */
  { "stm",	MEMOARY		},	/* code 0x96 */
  { "stm",	MEMOARY		},	/* code 0x97 */
  { "stm",	MEMOARY		},	/* code 0x98 */
  { "stm",	MEMOARY		},	/* code 0x99 */
  { "stm",	MEMOARY		},	/* code 0x9A */
  { "stm",	MEMOARY		},	/* code 0x9B */
  { "stm",	MEMOARY		},	/* code 0x9C */
  { "stm",	MEMOARY		},	/* code 0x9D */
  { "stm",	MEMOARY		},	/* code 0x9E */
  { "stm",	MEMOARY		},	/* code 0x9F */
  { "ld",	REGMEMO		},	/* code 0xA0 */
  { "ld",	REGMEMO		},	/* code 0xA1 */
  { "ld",	REGMEMO		},	/* code 0xA2 */
  { "ld",	REGMEMO		},	/* code 0xA3 */
  { "ld",	REGMEMO		},	/* code 0xA4 */
  { "ld",	REGMEMO		},	/* code 0xA5 */
  { "ld",	REGMEMO		},	/* code 0xA6 */
  { "ld",	REGMEMO		},	/* code 0xA7 */
  { "ld",	REGMEMO		},	/* code 0xA8 */
  { "ld",	REGMEMO		},	/* code 0xA9 */
  { "ld",	REGMEMO		},	/* code 0xAA */
  { "ld",	REGMEMO		},	/* code 0xAB */
  { "ld",	REGMEMO		},	/* code 0xAC */
  { "ld",	REGMEMO		},	/* code 0xAD */
  { "ld",	REGMEMO		},	/* code 0xAE */
  { "ld",	REGMEMO		},	/* code 0xAF */
  { "ldm",	ARYMEMO		},	/* code 0xB0 */
  { "ldm",	ARYMEMO		},	/* code 0xB1 */
  { "ldm",	ARYMEMO		},	/* code 0xB2 */
  { "ldm",	ARYMEMO		},	/* code 0xB3 */
  { "ldm",	ARYMEMO		},	/* code 0xB4 */
  { "ldm",	ARYMEMO		},	/* code 0xB5 */
  { "ldm",	ARYMEMO		},	/* code 0xB6 */
  { "ldm",	ARYMEMO		},	/* code 0xB7 */
  { "ldm",	ARYMEMO		},	/* code 0xB8 */
  { "ldm",	ARYMEMO		},	/* code 0xB9 */
  { "ldm",	ARYMEMO		},	/* code 0xBA */
  { "ldm",	ARYMEMO		},	/* code 0xBB */
  { "ldm",	ARYMEMO		},	/* code 0xBC */
  { "ldm",	ARYMEMO		},	/* code 0xBD */
  { "ldm",	ARYMEMO		},	/* code 0xBE */
  { "ldm",	ARYMEMO		},	/* code 0xBF */
  { "pst",	STIM8		},	/* code 0xC0 */
  { "pst",	STIM8		},	/* code 0xC1 */
  { "pst",	STIM8		},	/* code 0xC2 */
  { "pst",	STIM8		},	/* code 0xC3 */
  { "pst",	STREG		},	/* code 0xC4 */
  { "pst",	STREG		},	/* code 0xC5 */
  { "pst",	STREG		},	/* code 0xC6 */
  { "pst",	STREG		},	/* code 0xC7 */
  { "st",	SMEMOIM8	},	/* code 0xC8 */
  { "st",	SMEMOIM8	},	/* code 0xC9 */
  { "st",	SMEMOIM8	},	/* code 0xCA */
  { "st",	SMEMOIM8	},	/* code 0xCB */
  { "pst",	STREG		},	/* code 0xCC */
  { "pst",	STREG		},	/* code 0xCD */
  { "pst",	STREG		},	/* code 0xCE */
  { "pst",	STREG		},	/* code 0xCF */
  { "st",	MEMOIM8		},	/* code 0xD0 */
  { "st",	MEMOIM8		},	/* code 0xD1 */
  { "st",	MEMOIM8		},	/* code 0xD2 */
  { "st",	MEMOIM8		},	/* code 0xD3 */
  { "gst",	REGST		},	/* code 0xD4 */
  { "gst",	REGST		},	/* code 0xD5 */
  { "gst",	REGST		},	/* code 0xD6 */
  { "gst",	REGST		},	/* code 0xD7 */
  { "st",	SMEMOIM8	},	/* code 0xD8 */
  { "st",	SMEMOIM8	},	/* code 0xD9 */
  { "st",	SMEMOIM8	},	/* code 0xDA */
  { "st",	SMEMOIM8	},	/* code 0xDB */
  { "gst",	REGST		},	/* code 0xDC */
  { "gst",	REGST		},	/* code 0xDD */
  { "gst",	REGST		},	/* code 0xDE */
  { "gst",	REGST		},	/* code 0xDF */
  { "cmp",	REG		},	/* code 0xE0 */
  { "cmpm",	ARY		},	/* code 0xE1 */
  { "bup",	NONE		},	/* code 0xE2 */
  { "byum",	ARY		},	/* code 0xE3 */
  { "cmpb",	REG		},	/* code 0xE4 */
  { "dium",	ARY		},	/* code 0xE5 */
  { "rou",	REG		},	/* code 0xE6 */
  { "roum",	ARY		},	/* code 0xE7 */
  { "cmp",	REG		},	/* code 0xE8 */
  { "cmpm",	ARY		},	/* code 0xE9 */
  { "[EA00]",	YREG		},	/* code 0xEA */
  { "byum",	ARY		},	/* code 0xEB */
  { "cmpb",	REG		},	/* code 0xEC */
  { "dium",	ARY		},	/* code 0xED */
  { "rou",	REG		},	/* code 0xEE */
  { "roum",	ARY		},	/* code 0xEF */
  { "inv",	REG		},	/* code 0xF0 */
  { "invm",	ARY		},	/* code 0xF1 */
  { "bdn",	NONE		},	/* code 0xF2 */
  { "bydm",	ARY		},	/* code 0xF3 */
  { "cmpbm",	ARYMTBM		},	/* code 0xF4 */
  { "didm",	ARY		},	/* code 0xF5 */
  { "rod",	REG		},	/* code 0xF6 */
  { "rodm",	ARY		},	/* code 0xF7 */
  { "inv",	REG		},	/* code 0xF8 */
  { "invm",	ARY		},	/* code 0xF9 */
  { "nop",	NONE		},	/* code 0xFA */
  { "bydm",	ARY		},	/* code 0xFB */
  { "cmpbm",	ARYMTBM		},	/* code 0xFC */
  { "didm",	ARY		},	/* code 0xFD */
  { "rod",	REG		},	/* code 0xFE */
  { "rodm",	ARY		},	/* code 0xFF */
/* bit 7 of the second byte set */
  { "adb",	REGIM8		},	/* code 0x00 */
  { "sbb",	REGIM8		},	/* code 0x01 */
  { "ad",	REGIM8		},	/* code 0x02 */
  { "sb",	REGIM8		},	/* code 0x03 */
  { "an",	REGIM8		},	/* code 0x04 */
  { "na",	REGIM8		},	/* code 0x05 */
  { "or",	REGIM8		},	/* code 0x06 */
  { "xr",	REGIM8		},	/* code 0x07 */
  { "adb",	REGIM8		},	/* code 0x08 */
  { "sbb",	REGIM8		},	/* code 0x09 */
  { "ad",	REGIM8		},	/* code 0x0A */
  { "sb",	REGIM8		},	/* code 0x0B */
  { "an",	REGIM8		},	/* code 0x0C */
  { "na",	REGIM8		},	/* code 0x0D */
  { "or",	REGIM8		},	/* code 0x0E */
  { "xr",	REGIM8		},	/* code 0x0F */
  { "adbm",	ARYIM6		},	/* code 0x10 */
  { "sbbm",	ARYIM6		},	/* code 0x11 */
  { "adm",	ARYIM6		},	/* code 0x12 */
  { "sbm",	ARYIM6		},	/* code 0x13 */
  { "swp",	XREG		},	/* code 0x14 */
  { "swpm",	XARY		},	/* code 0x15 */
  { "xcls",	XREG		},	/* code 0x16 */
  { "xclsm",	XARY		},	/* code 0x17 */
  { "adbm",	ARYIM6		},	/* code 0x18 */
  { "sbbm",	ARYIM6		},	/* code 0x19 */
  { "adm",	ARYIM6		},	/* code 0x1A */
  { "sbm",	ARYIM6		},	/* code 0x1B */
  { "xc",	XREG		},	/* code 0x1C */
  { "xcm",	XARY		},	/* code 0x1D */
  { "xchs",	XREG		},	/* code 0x1E */
  { "xchsm",	XARY		},	/* code 0x1F */
  { "tadb",	REGIM8		},	/* code 0x20 */
  { "tsbb",	REGIM8		},	/* code 0x21 */
  { "tad",	REGIM8		},	/* code 0x22 */
  { "tsb",	REGIM8		},	/* code 0x23 */
  { "tan",	REGIM8		},	/* code 0x24 */
  { "tna",	REGIM8		},	/* code 0x25 */
  { "tor",	REGIM8		},	/* code 0x26 */
  { "txr",	REGIM8		},	/* code 0x27 */
  { "tadb",	REGIM8		},	/* code 0x28 */
  { "tsbb",	REGIM8		},	/* code 0x29 */
  { "tad",	REGIM8		},	/* code 0x2A */
  { "tsb",	REGIM8		},	/* code 0x2B */
  { "tan",	REGIM8		},	/* code 0x2C */
  { "tna",	REGIM8		},	/* code 0x2D */
  { "tor",	REGIM8		},	/* code 0x2E */
  { "txr",	REGIM8		},	/* code 0x2F */
  { "tadbm",	ARYIM6		},	/* code 0x30 */
  { "tsbbm",	ARYIM6		},	/* code 0x31 */
  { "tadm",	ARYIM6		},	/* code 0x32 */
  { "tsbm",	ARYIM6		},	/* code 0x33 */
  { "sup",	REG		},	/* code 0x34 */
  { "[3580]",	DATA1		},	/* code 0x35 */
  { "[3680]",	DATA1		},	/* code 0x36 */
  { "[3780]",	DATA1		},	/* code 0x37 */
  { "tadbm",	ARYIM6		},	/* code 0x38 */
  { "tsbbm",	ARYIM6		},	/* code 0x39 */
  { "tadm",	ARYIM6		},	/* code 0x3A */
  { "tsbm",	ARYIM6		},	/* code 0x3B */
  { "sdn",	REG		},	/* code 0x3C */
  { "[3D80]",	DATA1		},	/* code 0x3D */
  { "[3E80]",	DATA1		},	/* code 0x3E */
  { "[3F80]",	DATA1		},	/* code 0x3F */
  { "ldw",	WRIM16		},	/* code 0x40 */
  { "ldw",	WRIM16		},	/* code 0x41 */
  { "ldw",	WRIM16		},	/* code 0x42 */
  { "ldw",	WRIM16		},	/* code 0x43 */
  { "ldw",	WRIM16		},	/* code 0x44 */
  { "ldw",	WRIM16		},	/* code 0x45 */
  { "ldw",	WRIM16		},	/* code 0x46 */
  { "ldw",	WRIM16		},	/* code 0x47 */
  { "adw",	IRIM8		},	/* code 0x48 */
  { "adw",	IRIM8		},	/* code 0x49 */
  { "adw",	IRIM8		},	/* code 0x4A */
  { "adw",	IRIM8		},	/* code 0x4B */
  { "sbw",	IRIM8		},	/* code 0x4C */
  { "sbw",	IRIM8		},	/* code 0x4D */
  { "sbw",	IRIM8		},	/* code 0x4E */
  { "sbw",	IRIM8		},	/* code 0x4F */
  { "off",	DROP1		},	/* code 0x50 */
  { "gst",	REGKI		},	/* code 0x51 */
  { "bit",	REGIM8		},	/* code 0x52 */
  { "bit",	XREG		},	/* code 0x53 */
  { "ld",	XREG		},	/* code 0x54 */
  { "ldm",	XARY		},	/* code 0x55 */
  { "ld",	REGIM8		},	/* code 0x56 */
  { "ldm",	ARYIM6		},	/* code 0x57 */
  { "rtn",	NONE		},	/* code 0x58 */
  { "gst",	REGKI		},	/* code 0x59 */
  { "bit",	REGIM8		},	/* code 0x5A */
  { "bit",	YREG		},	/* code 0x5B */
  { "ld",	YREG		},	/* code 0x5C */
  { "ldm",	YARY		},	/* code 0x5D */
  { "ld",	REGIM8		},	/* code 0x5E */
  { "ldm",	ARYIM6		},	/* code 0x5F */
  { "cal",	IM16		},	/* code 0x60 */
  { "cal",	CCIM16		},	/* code 0x61 */
  { "cal",	CCIM16		},	/* code 0x62 */
  { "cal",	CCIM16		},	/* code 0x63 */
  { "cal",	CCIM16		},	/* code 0x64 */
  { "cal",	CCIM16		},	/* code 0x65 */
  { "cal",	CCIM16		},	/* code 0x66 */
  { "cal",	CCIM16		},	/* code 0x67 */
  { "cani",	NONE		},	/* code 0x68 */
  { "cal",	CCIM16		},	/* code 0x69 */
  { "cal",	CCIM16		},	/* code 0x6A */
  { "cal",	CCIM16		},	/* code 0x6B */
  { "cal",	CCIM16		},	/* code 0x6C */
  { "cal",	CCIM16		},	/* code 0x6D */
  { "cal",	CCIM16		},	/* code 0x6E */
  { "cal",	CCIM16		},	/* code 0x6F */
  { "jmp",	IM16		},	/* code 0x70 */
  { "jmp",	CCIM16		},	/* code 0x71 */
  { "jmp",	CCIM16		},	/* code 0x72 */
  { "jmp",	CCIM16		},	/* code 0x73 */
  { "jmp",	CCIM16		},	/* code 0x74 */
  { "jmp",	CCIM16		},	/* code 0x75 */
  { "jmp",	CCIM16		},	/* code 0x76 */
  { "jmp",	CCIM16		},	/* code 0x77 */
  { "rti",	NONE		},	/* code 0x78 */
  { "jmp",	CCIM16		},	/* code 0x79 */
  { "jmp",	CCIM16		},	/* code 0x7A */
  { "jmp",	CCIM16		},	/* code 0x7B */
  { "jmp",	CCIM16		},	/* code 0x7C */
  { "jmp",	CCIM16		},	/* code 0x7D */
  { "jmp",	CCIM16		},	/* code 0x7E */
  { "jmp",	CCIM16		},	/* code 0x7F */
  { "st",	OFFSREG		},	/* code 0x80 */
  { "st",	OFFSREG		},	/* code 0x81 */
  { "st",	OFFSREG		},	/* code 0x82 */
  { "st",	OFFSREG		},	/* code 0x83 */
  { "st",	OFFSREG		},	/* code 0x84 */
  { "st",	OFFSREG		},	/* code 0x85 */
  { "st",	OFFSREG		},	/* code 0x86 */
  { "st",	OFFSREG		},	/* code 0x87 */
  { "st",	OFFSREG		},	/* code 0x88 */
  { "st",	OFFSREG		},	/* code 0x89 */
  { "st",	OFFSREG		},	/* code 0x8A */
  { "st",	OFFSREG		},	/* code 0x8B */
  { "st",	OFFSREG		},	/* code 0x8C */
  { "st",	OFFSREG		},	/* code 0x8D */
  { "st",	OFFSREG		},	/* code 0x8E */
  { "st",	OFFSREG		},	/* code 0x8F */
  { "stm",	OFFSARY		},	/* code 0x90 */
  { "stm",	OFFSARY		},	/* code 0x91 */
  { "stm",	OFFSARY		},	/* code 0x92 */
  { "stm",	OFFSARY		},	/* code 0x93 */
  { "stm",	OFFSARY		},	/* code 0x94 */
  { "stm",	OFFSARY		},	/* code 0x95 */
  { "stm",	OFFSARY		},	/* code 0x96 */
  { "stm",	OFFSARY		},	/* code 0x97 */
  { "stm",	OFFSARY		},	/* code 0x98 */
  { "stm",	OFFSARY		},	/* code 0x99 */
  { "stm",	OFFSARY		},	/* code 0x9A */
  { "stm",	OFFSARY		},	/* code 0x9B */
  { "stm",	OFFSARY		},	/* code 0x9C */
  { "stm",	OFFSARY		},	/* code 0x9D */
  { "stm",	OFFSARY		},	/* code 0x9E */
  { "stm",	OFFSARY		},	/* code 0x9F */
  { "ld",	REGOFFS		},	/* code 0xA0 */
  { "ld",	REGOFFS		},	/* code 0xA1 */
  { "ld",	REGOFFS		},	/* code 0xA2 */
  { "ld",	REGOFFS		},	/* code 0xA3 */
  { "ld",	REGOFFS		},	/* code 0xA4 */
  { "ld",	REGOFFS		},	/* code 0xA5 */
  { "ld",	REGOFFS		},	/* code 0xA6 */
  { "ld",	REGOFFS		},	/* code 0xA7 */
  { "ld",	REGOFFS		},	/* code 0xA8 */
  { "ld",	REGOFFS		},	/* code 0xA9 */
  { "ld",	REGOFFS		},	/* code 0xAA */
  { "ld",	REGOFFS		},	/* code 0xAB */
  { "ld",	REGOFFS		},	/* code 0xAC */
  { "ld",	REGOFFS		},	/* code 0xAD */
  { "ld",	REGOFFS		},	/* code 0xAE */
  { "ld",	REGOFFS		},	/* code 0xAF */
  { "ldm",	ARYOFFS		},	/* code 0xB0 */
  { "ldm",	ARYOFFS		},	/* code 0xB1 */
  { "ldm",	ARYOFFS		},	/* code 0xB2 */
  { "ldm",	ARYOFFS		},	/* code 0xB3 */
  { "ldm",	ARYOFFS		},	/* code 0xB4 */
  { "ldm",	ARYOFFS		},	/* code 0xB5 */
  { "ldm",	ARYOFFS		},	/* code 0xB6 */
  { "ldm",	ARYOFFS		},	/* code 0xB7 */
  { "ldm",	ARYOFFS		},	/* code 0xB8 */
  { "ldm",	ARYOFFS		},	/* code 0xB9 */
  { "ldm",	ARYOFFS		},	/* code 0xBA */
  { "ldm",	ARYOFFS		},	/* code 0xBB */
  { "ldm",	ARYOFFS		},	/* code 0xBC */
  { "ldm",	ARYOFFS		},	/* code 0xBD */
  { "ldm",	ARYOFFS		},	/* code 0xBE */
  { "ldm",	ARYOFFS		},	/* code 0xBF */
  { "pst",	STIM8		},	/* code 0xC0 */
  { "pst",	STIM8		},	/* code 0xC1 */
  { "pst",	STIM8		},	/* code 0xC2 */
  { "pst",	STIM8		},	/* code 0xC3 */
  { "ldle",	REGIM8		},	/* code 0xC4 */
  { "ldlo",	REGIM8		},	/* code 0xC5 */
  { "stle",	REGIM8		},	/* code 0xC6 */
  { "stlo",	REGIM8		},	/* code 0xC7 */
  { "st",	SMEMOIM8	},	/* code 0xC8 */
  { "st",	SMEMOIM8	},	/* code 0xC9 */
  { "st",	SMEMOIM8	},	/* code 0xCA */
  { "st",	SMEMOIM8	},	/* code 0xCB */
  { "ldle",	REGIM8		},	/* code 0xCC */
  { "ldlo",	REGIM8		},	/* code 0xCD */
  { "stle",	REGIM8		},	/* code 0xCE */
  { "stlo",	REGIM8		},	/* code 0xCF */
  { "st",	MEMOIM8		},	/* code 0xD0 */
  { "st",	MEMOIM8		},	/* code 0xD1 */
  { "st",	MEMOIM8		},	/* code 0xD2 */
  { "st",	MEMOIM8		},	/* code 0xD3 */
  { "ldlem",	ARYIM6		},	/* code 0xD4 */
  { "ldlom",	ARYIM6		},	/* code 0xD5 */
  { "stlem",	ARYIM6		},	/* code 0xD6 */
  { "stlom",	ARYIM6		},	/* code 0xD7 */
  { "st",	SMEMOIM8	},	/* code 0xD8 */
  { "st",	SMEMOIM8	},	/* code 0xD9 */
  { "st",	SMEMOIM8	},	/* code 0xDA */
  { "st",	SMEMOIM8	},	/* code 0xDB */
  { "ldlem",	ARYIM6		},	/* code 0xDC */
  { "ldlom",	ARYIM6		},	/* code 0xDD */
  { "stlem",	ARYIM6		},	/* code 0xDE */
  { "stlom",	ARYIM6		},	/* code 0xDF */
  { "swp",	REG		},	/* code 0xE0 */
  { "[E180]",	DATA1		},	/* code 0xE1 */
  { "bup",	NONE		},	/* code 0xE2 */
  { "byu",	REG		},	/* code 0xE3 */
  { "biu",	REG		},	/* code 0xE4 */
  { "diu",	REG		},	/* code 0xE5 */
  { "mtb",	REG		},	/* code 0xE6 */
  { "trp",	DROP1		},	/* code 0xE7 */
  { "swp",	REG		},	/* code 0xE8 */
  { "[E980]",	DATA1		},	/* code 0xE9 */
  { "[EA80]",	YREG		},	/* code 0xEA */
  { "byu",	REG		},	/* code 0xEB */
  { "biu",	REG		},	/* code 0xEC */
  { "diu",	REG		},	/* code 0xED */
  { "mtb",	REG		},	/* code 0xEE */
  { "trp",	DROP1		},	/* code 0xEF */
  { "bnusm",	ARY		},	/* code 0xF0 */
  { "gst",	REGIF		},	/* code 0xF1 */
  { "bdn",	NONE		},	/* code 0xF2 */
  { "byd",	REG		},	/* code 0xF3 */
  { "bium",	ARY		},	/* code 0xF4 */
  { "did",	REG		},	/* code 0xF5 */
  { "mtbm",	ARYMTBM		},	/* code 0xF6 */
  { "trp",	DROP1		},	/* code 0xF7 */
  { "bnusm",	ARY		},	/* code 0xF8 */
  { "gst",	REGIF		},	/* code 0xF9 */
  { "nop",	NONE		},	/* code 0xFA */
  { "byd",	REG		},	/* code 0xFB */
  { "bium",	ARY		},	/* code 0xFC */
  { "did",	REG		},	/* code 0xFD */
  { "mtbm",	ARYMTBM		},	/* code 0xFE */
  { "trp",	DROP1		}	/* code 0xFF */
};

char *ir_tab[] = { "ix", "iy", "iz", "sp" };
char *wr_tab[] = { "ix", "iy", "iz", "v3", "v2", "v1", "v0", "sp" };
char *cc_tab[] = { "*", "k", "lz",  "uz",  "nz", "v",  "h",  "c",
                   "*", "nk", "nlz", "nuz", "z",  "nv", "nh", "nc" };
char *st_tab[] = { "ko", "if", "as", "ie",	/* immediate value */
                   "ko", "f", "as", "ie" };	/* register */


uint FetchByte (void)
{
  loc++;
  return inbuf[head++];
}


uint Reg1 (uint x)
{
  return ((x >> 1) & 0x38) | (x & 0x07);
}


uint Reg2 (uint y)
{
  return ((y >> 5) & 0x07) | ((y << 3) & 0x38);
}


uint Rl1 (uint x, uint y)
{
  return ((x >> 1) & 0x38) | ((y >> 5) & 0x07);
}


uint Rf2 (uint x, uint y)
{
  return (x & 0x07) | ((y << 3) & 0x38);
}


uint Im6 (uint x, uint y)
{
  return (y & 0x1F) | ((~x << 2) & 0x20);
}


void Arguments (uint opcode)
{
  uint x, y, z;
  char sign, name;

  sign = ((opcode & 0x04) == 0) ? '+' : '-';
  z = (~opcode << 3) & 0x40;

  switch (mnem[opcode].kind)
  {
    case NONE:
      break;

    case DROP1:
      (void) FetchByte();
      break;

    case CCIM16:
      printf ("%s,", cc_tab[opcode & 0x0F]);
/*@fallthrough@*/
    case IM16:
      x = FetchByte();
      y = FetchByte();
      printf ("&H%02X%02X", x, y);
      break;

    case WRIM16:
      x = FetchByte();
      y = FetchByte();
      printf ("%s,&H%02X%02X", wr_tab[opcode & 0x07], x, y);
      break;

    case IRIM8:
      printf ("%s,&H%02X", ir_tab[opcode & 0x03], FetchByte());
      break;

    case SMEMOIM8:
      (void) putchar ((opcode & 0x10) == 0 ? '+' : '-');
/*@fallthrough@*/
    case MEMOIM8:
      printf ("(%s),&H%02X", ir_tab[opcode & 0x03], FetchByte());
      break;

    case XREG:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? 'i' : 'r';
      printf ("%c%u,r%u", name, Reg2(y) | 0x40, Reg1(x));
      break;

    case YREG:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? 'i' : 'r';
      printf ("r%u,%c%u", Reg1(x), name, Reg2(y) | 0x40);
      break;

    case REGIM8:
      x = FetchByte();
      y = FetchByte();
      printf ("r%u,&H%02X", Reg1(x) | z, y);
      break;

    case XARY:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? 'i' : 'r';
      printf ("%c%u..%c%u,r%u..r%u",
	name, Rf2(x,y) | 0x40, name, Reg2(y) | 0x40,
	Reg1(x), Rl1(x,y));
      break;

    case YARY:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? 'i' : 'r';
      printf ("r%u..r%u,%c%u..%c%u",
	Reg1(x), Rl1(x,y),
	name, Rf2(x,y) | 0x40, name, Reg2(y) | 0x40);
      break;

    case ARYIM6:
      x = FetchByte();
      y = FetchByte();
      printf ("r%u..r%u,&H%02X", Reg1(x) | z, Rl1(x,y) | z, Im6(x,y));
      break;

    case MEMOREG:
      printf ("%c(%s),", sign, ir_tab[opcode & 0x03]);
/*@fallthrough@*/
    case REG:
      printf ("r%u", Reg1(FetchByte()) | z);
      break;

    case OFFSREG:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? 'i' : 'r';
      if ((opcode & 0x08) == 0)
      {
        printf ("(%s%cr%u),%c%u",
		ir_tab[opcode & 0x03], sign, Reg1(x), name, Reg2(y) | 0x40);
      }
      else
      {
        printf ("(%s%c&H%02X),r%u", ir_tab[opcode & 0x03], sign, y, Reg1(x));
      }
      break;

    case REGMEMO:
      x = FetchByte();
      printf ("r%u,(%s)%c", Reg1(x) | z, ir_tab[opcode & 0x03], sign);
      break;

    case REGOFFS:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? 'i' : 'r';
      if ((opcode & 0x08) == 0)
      {
        printf ("%c%u,(%s%cr%u)",
		name, Reg2(y) | 0x40, ir_tab[opcode & 0x03], sign, Reg1(x));
      }
      else
      {
        printf ("r%u,(%s%c&H%02X)", Reg1(x), ir_tab[opcode & 0x03], sign, y);
      }
      break;

    case MEMOARY:
      x = FetchByte();
      y = FetchByte();
      printf ("%c(%s),r%u..r%u",
		sign, ir_tab[opcode & 0x03], Reg1(x) | z, Rl1(x,y) | z);
      break;

    case OFFSARY:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? '?' : 'r';	/* indirect is illegal */
      if ((opcode & 0x08) == 0)
      {
        printf ("(%s%cr%u),%c%u..%c%u",
		ir_tab[opcode & 0x03], sign, Reg1(x),
		name, Rf2(x,y) | 0x40, name, Reg2(y) | 0x40);
      }
      else
      {
        printf ("(%s%c&H%02X),r%u..r%u",
		ir_tab[opcode & 0x03], sign, Im6(x,y),
		Reg1(x), Rl1(x,y));
      }
      break;

    case ARY:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? 'i' : 'r';
      if ((opcode & 0x08) == 0)
      {
        printf ("%c%u..%c%u", name, Rf2(x,y) | 0x40, name, Reg2(y) | 0x40);
      }
      else
      {
        printf ("r%u..r%u", Reg1(x), Rl1(x,y));
      }
      break;

    case ARYMTBM:
      x = FetchByte();
      y = FetchByte();
      printf ("r%u..r%u", Reg1(x) | z, Rl1(x,y) | z);
      break;

    case ARYMEMO:
      x = FetchByte();
      y = FetchByte();
      printf ("r%u..r%u,(%s)%c",
		Reg1(x) | z, Rl1(x,y) | z, ir_tab[opcode & 0x03], sign);
      break;

    case ARYOFFS:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? '?' : 'r';	/* indirect is illegal */
      if ((opcode & 0x08) == 0)
      {
        printf ("%c%u..%c%u,(%s%cr%u)",
		name, Rf2(x,y) | 0x40, name, Reg2(y) | 0x40,
		ir_tab[opcode & 0x03], sign, Reg1(x));
      }
      else
      {
        printf ("r%u..r%u,(%s%c&H%02X)",
		Reg1(x), Rl1(x,y),
		ir_tab[opcode & 0x03], sign, Im6(x,y));
      }
      break;

    case STREG:
      printf ("%s,r%u", st_tab[opcode & 0x07], Reg1(FetchByte()) | z);
      break;

    case STIM8:
      printf ("%s,&H%02X", st_tab[opcode & 0x07], FetchByte());
      break;

    case REGST:
      printf ("r%u,%s", Reg1(FetchByte()) | z, st_tab[opcode & 0x07]);
      break;

    case REGIF:
      printf ("r%u,if", Reg1(FetchByte()) | z);
      break;

    case REGKI:
      printf ("r%u,ki", Reg1(FetchByte()) | z);
      break;

    case DATA1:
      printf ("[%02X]", FetchByte());
      break;

    case CCINDIR:
      printf ("%s,", cc_tab[opcode & 0x0F]);
/*@fallthrough@*/
    case INDIR:
      x = FetchByte();
      y = FetchByte();
      name = ((y & 0x10) != 0) ? 'i' : 'r';
      if (Reg1(x) == Reg2(y) && (y & 0x1F) == 0x07 && name == 'r')
      {
        printf ("%s", wr_tab[x & 0x07]);
      }
      else
      {
        printf ("%c%u,r%u", name, Reg2(y) | 0x40, Reg1(x));
      }
      break;

    default:
      break;
  }
}


int main (int argc, char *argv[])
{
  int x;
  uint i;
  FILE *fp;

  if (argc < 2)
  {
    fprintf (stderr, "\nNo input file specified\n");
    return 1;
  }

  if ((fp = fopen(*++argv, "rb")) == NULL)
  {
    fprintf(stderr, "\nCannot open the input file %s\n", *argv);
    return 1;
  }

  loc = 0;

  while (argc-->2)
  {
    argv++;
    if (isxdigit(**argv))
    {
      (void) sscanf(*argv, "%x", &loc);
    }
    else if (**argv == '/' || **argv == '-')
    {
      switch (*(*argv+1))
      {
/* reserved for future expansions */
        default:
          fprintf(stderr, "\nUnrecognised switch: %s\n", *argv+1);
          break;
      }
    }
    else
    {
      fprintf(stderr, "\nUnrecognised parameter: %s\n", *argv);
    }
  }
  head = 0;
  tail = 0;

  do {

/* shift the bytes in the inbuf */
    i = 0;
    while (head<tail)
    {
      inbuf[i++] = inbuf[head++];
    }
    tail = i;
    head = 0;
/* fill the inbuf */
    while ((tail<INBUFSIZE) && ((x=getc(fp))!=EOF))
    {
      inbuf[tail++] = (uint) x;
    }
/* disassemble */
    printf ("%04X:\t", loc);
    i = FetchByte();
    i += (inbuf[head] << 1) & 0x0100;
    printf ("%s", mnem[i].str);
    if (mnem[i].kind != NONE)
    {
      (void) putchar ('\t');
    }
    Arguments (i);
    printf ("\n");

  } while (tail!=0);

  (void) fclose (fp);
  return 0;
}
