/*
 *  Utility to handle files in the ramdisk of the Casio PB-1000 emulator,
 *  written by Piotr Piatek <http://www.pisi.com.pl/piotr433/index.htm>
 *
 *  Contains large portions of the program md100 in version 30.03.2006 1.1,
 *  written by Marcus von Cube <http://www.mvcsys.de/>
 *
 *  Potential portability issues:
 *  - little endian order of bytes required
 *
 *  Updates:
 *  2009/01/22 - the program appends the EOF byte to a binary file copied to
 *               the PB-1000 ramdisk, and strips the EOF byte from a binary
 *               file copied to the PC
 */

#define DEBUG 0

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>

typedef unsigned short word;
typedef unsigned char byte;

/*
 *  Constants
 */
#define OK 0
#define NOT_OK -1

#define TRUE 1
#define FALSE 0

/*
 *  File types
 */
#define TYPE_M 0x0D           /* binary file */
#define TYPE_B 0x10           /* tokenized BASIC */
#define TYPE_S 0x24           /* sequential ASCII file */

/*
 *  File flags
 */
#define FLAG_NONE 0
#define FLAG_PRESET 1
#define FLAG_CBOOT 2

/*
 *  Structure of directory entry
 */
typedef struct _dir_entry {
    byte flag;		/* one of FLAG_* */
    word srcbegin;	/* starting address of the file in the RAM */
    word srcend;	/* = srcbegin + filesize */
    byte type;		/* one of TYPE_? */
    char name[8];	/* blank padded, lower case is preserved */
    char ext[3];	/* same here */
    byte pass[8];	/* password */
    word asmorg;	/* assembly address for machine code */
    word asmend;	/* = asmorg + filesize */
    word execaddr;	/* machine code starting address */
    byte unknown;	/* unknown meaning */
} DIR_ENTRY;

/*
 *  Some errors
 */
typedef enum _error {
    NO_ERROR = 0,
    FILE_EXISTS = 1,
    FILE_NOT_FOUND = 2,
    NO_ROOM = 3,
    IO_ERROR = 4,
} ERROR;

const char *Error[] = {
    "OK",
    "File exists",
    "File not found",
    "No room for file",
    "I/O error",
};

/*
 *  File information
 */
typedef struct _file_info {
    char name[ 13 ];	/* formatted filename */
    char typeS[ 3 ];	/* "M", "B", "S" or "xx" */
    word address;	/* starting address of the file in the RAM */
    word size;		/* size of file */
    byte type;		/* from directory */
    DIR_ENTRY *entry;	/* original directory entry */
    ERROR error;	/* Copy or Rename failed */
} FILE_INFO;

/*
 *  Option -l or -u
 */
typedef enum _set_case {
    AS_IS,
    LOWER,
    UPPER
} CASE;

/*
 *  Option -a or -b
 */
typedef enum _mode {
    AUTO,
    ASCII,
    BINARY
} MODE;

/*
 *  Options parsed from command line
 */
typedef struct _options {
    int           ignoreCase;  /* -i */
    CASE          setCase;     /* -l or -u */
    int           type;        /* -t */
    char          typeS[ 4 ];  /* -t as text */
    MODE          mode;        /* -a or -b */
    char         *destination; /* -d */
    int           noUpdate;    /* -n */
} OPTIONS;

#define DEFAULT_OPTIONS { FALSE, AS_IS, 0, "", AUTO, NULL, FALSE }


/*
 *  Types as Strings
 */
typedef struct _types {
    byte type;
    char typeName[ 2 ];
} TYPES;

TYPES Types[] = {
    { TYPE_B, "B" },
    { TYPE_M, "M" },
    { TYPE_S, "S" },
    { 0, "" }
};


/*
 *  Default file types other than TYPE_S
 */
typedef struct _default_file_types {
    char ext[ 4 ];
    byte type;
} DEFAULTFILETYPES;

DEFAULTFILETYPES dft[] = {
    { "EXE", TYPE_M },	/* Binary executable */
    { "BIN", TYPE_M },	/* Binary file */
    { "BAS", TYPE_B }	/* Tokenized BASIC */
};


/*
 *  BASIC keywords for PB-1000
 *  Double byte tokens from 0x40 to 0xCF with prefixes 4 to 7
 */
char *Tokens[ 4 ][ 0xd0 - 0x40 ] /*@null@*/ = 
{ 
    { /* Prefix 4 */
        NULL,       NULL,       NULL,       NULL,       /* 40 */  
        NULL,       NULL,       NULL,       NULL,       /* 44 */  
        NULL,       "GOTO ",    "GOSUB ",   "RETURN",   /* 48 */  
        "RESUME ",  "RESTORE ", "WRITE#",   NULL,       /* 4C */  
        NULL,       NULL,       "SYSTEM",   "PASS ",    /* 50 */  
        NULL,       "DELETE ",  "BSAVE ",   "LIST ",    /* 54 */  
        "LLIST ",   "LOAD ",    "MERGE ",   NULL,       /* 58 */  
        NULL,       "TRON",     NULL,       "TROFF",    /* 5C */  
        "VERIFY ",  "MON",      "CALL ",    "POKE ",    /* 60 */  
        NULL,       NULL,       NULL,       NULL,       /* 64 */  
        NULL,       "CHAIN ",   "CLEAR ",   "NEW ",     /* 68 */  
        "SAVE ",    "RUN ",     "ANGLE ",   "EDIT ",    /* 6C */  
        "BEEP ",    "CLS",      "CLOSE ",   NULL,       /* 70 */  
        NULL,       NULL,       "DEF ",     "DEFM ",    /* 74 */  
        "DEFSEG ",  NULL,       "VAC",      NULL,       /* 78 */  
        "DIM ",     "DRAW ",    NULL,       NULL,       /* 7C */  
        "DATA ",    "FOR ",     "NEXT ",    NULL,       /* 80 */  
        NULL,       "ERASE ",   "ERROR ",   "END",      /* 84 */  
        NULL,       NULL,       "FIELD ",   "FORMAT",   /* 88 */  
        "GET ",     "IF ",      NULL,       "LET ",     /* 8C */  
        "LINE ",    "LOCATE ",  NULL,       "LSET ",    /* 90 */  
        NULL,       NULL,       NULL,       "OPEN ",    /* 94 */  
        NULL,       "OUT ",     "ON ",      NULL,       /* 98 */  
        NULL,       NULL,       NULL,       "CALCJMP ", /* 9C */  
        "BLOAD ",   NULL,       "DRAWC ",   "PRINT ",   /* A0 */  
        "LPRINT ",  "PUT ",     NULL,       NULL,       /* A4 */  
        "READ ",    "REM ",     "RSET ",    NULL,       /* A8 */  
        "SET ",     "STAT",     "STOP",     NULL,       /* AC */  
        "MODE ",    NULL,       "VAR ",     "PBLOAD ",  /* B0 */  
        "PBGET ",   NULL,       NULL,       NULL,       /* B4 */  
        NULL,       NULL,       NULL,       NULL,       /* B8 */  
        NULL,       NULL,       NULL,       NULL,       /* BC */  
        NULL,       NULL,       NULL,       NULL,       /* C0 */  
        NULL,       NULL,       NULL,       NULL,       /* C4 */  
        NULL,       NULL,       NULL,       NULL,       /* C8 */  
        NULL,       NULL,       NULL,       NULL        /* CC */  
    },
    { /* prefix 5 */
        NULL,       NULL,       NULL,       NULL,       /* 40 */
        NULL,       NULL,       NULL,       NULL,       /* 44 */
        NULL,       NULL,       NULL,       NULL,       /* 48 */
        NULL,       NULL,       NULL,       "ERL",      /* 4C */
        "ERR",      "CNT",      "SUMX",     "SUMY",     /* 50 */
        "SUMX2",    "SUMY2",    "SUMXY",    "MEANX",    /* 54 */
        "MEANY",    "SDX",      "SDY",      "SDXN",     /* 58 */
        "SDYN",     "LRA",      "LRB",      "COR",      /* 5C */
        "PI",       NULL,       NULL,       "CUR ",     /* 60 */
        NULL,       NULL,       NULL,       "FACT ",    /* 64 */
        NULL,       "EOX ",     "EOY ",     "SIN ",     /* 68 */
        "COS ",     "TAN ",     "ASN ",     "ACS ",     /* 6C */
        "ATN ",     "HYPSIN ",  "HYPCOS ",  "HYPTAN ",  /* 70 */
        "HYPASN ",  "HYPACS ",  "HYPATN ",  "LN ",      /* 74 */
        "LOG ",     "EXP ",     "SQR ",     "ABS ",     /* 78 */
        "SGN ",     "INT ",     "FIX ",     "FRAC ",    /* 7C */
        "RND ",     NULL,       NULL,       NULL,       /* 80 */
        NULL,       NULL,       "PEEK ",    NULL,       /* 84 */
        NULL,       "LOF ",     "EOF ",     NULL,       /* 88 */
        NULL,       "FRE ",     NULL,       "POINT ",   /* 8C */
        "ROUND",    "RND",      "VALF",     "RAN#",     /* 90 */
        "ASC",      "LEN",      "VAL",      NULL,       /* 94 */
        NULL,       NULL,       NULL,       NULL,       /* 98 */
        "DEG",      NULL,       NULL,       NULL,       /* 9C */
        NULL,       NULL,       NULL,       NULL,       /* A0 */
        NULL,       NULL,       NULL,       "REC",      /* A4 */
        "POL",      NULL,       "NPR",      "NCR",      /* A8 */
        "HYP",      NULL,       NULL,       NULL,       /* AC */
        NULL,       NULL,       NULL,       NULL,       /* B0 */
        NULL,       NULL,       NULL,       NULL,       /* B4 */
        NULL,       NULL,       NULL,       NULL,       /* B8 */
        NULL,       NULL,       NULL,       NULL,       /* BC */
        NULL,       NULL,       NULL,       NULL,       /* C0 */
        NULL,       NULL,       NULL,       NULL,       /* C4 */
        NULL,       NULL,       NULL,       NULL,       /* C8 */
        NULL,       NULL,       NULL,       NULL        /* CC */
    },
    { /* prefix 6 */
        NULL,       NULL,       NULL,       NULL,       /* 40 */
        NULL,       NULL,       NULL,       NULL,       /* 44 */
        NULL,       NULL,       NULL,       NULL,       /* 48 */
        NULL,       NULL,       NULL,       NULL,       /* 4C */
        NULL,       NULL,       NULL,       NULL,       /* 50 */
        NULL,       NULL,       NULL,       NULL,       /* 54 */
        NULL,       NULL,       NULL,       NULL,       /* 58 */
        NULL,       NULL,       NULL,       NULL,       /* 5C */
        NULL,       NULL,       NULL,       NULL,       /* 60 */
        NULL,       NULL,       NULL,       NULL,       /* 64 */
        NULL,       NULL,       NULL,       NULL,       /* 68 */
        NULL,       NULL,       NULL,       NULL,       /* 6C */
        NULL,       NULL,       NULL,       NULL,       /* 70 */
        NULL,       NULL,       NULL,       NULL,       /* 74 */
        NULL,       NULL,       NULL,       NULL,       /* 78 */
        NULL,       NULL,       NULL,       NULL,       /* 7C */
        NULL,       NULL,       NULL,       NULL,       /* 80 */
        NULL,       NULL,       NULL,       NULL,       /* 84 */
        NULL,       NULL,       NULL,       NULL,       /* 88 */
        NULL,       NULL,       NULL,       NULL,       /* 8C */
        NULL,       NULL,       NULL,       NULL,       /* 90 */
        NULL,       NULL,       NULL,       "DMS$",     /* 94 */
        NULL,       NULL,       "MID",      "INPUT ",   /* 98 */
        "MID$",     "RIGHT$",   "LEFT$",    NULL,       /* 9C */
        "CHR$",     "STR$",     NULL,       "HEX$",     /* A0 */
        NULL,       NULL,       NULL,       NULL,       /* A4 */
        "INKEY$",   "KEY",      NULL,       "DATE$",    /* A8 */
        "TIME$",    "CALC$",    NULL,       NULL,       /* AC */
        NULL,       NULL,       NULL,       NULL,       /* B0 */
        NULL,       NULL,       NULL,       NULL,       /* B4 */
        NULL,       NULL,       NULL,       NULL,       /* B8 */
        NULL,       NULL,       NULL,       NULL,       /* BC */
        NULL,       NULL,       NULL,       NULL,       /* C0 */
        NULL,       NULL,       NULL,       NULL,       /* C4 */
        NULL,       NULL,       NULL,       NULL,       /* C8 */
        NULL,       NULL,       NULL,       NULL        /* CC */
    },
    { /* prefix 7 */
        NULL,       NULL,       NULL,       NULL,       /* 40 */
        NULL,       NULL,       NULL,       " THEN ",   /* 44 */
        "ELSE ",    NULL,       NULL,       NULL,       /* 48 */
        NULL,       NULL,       NULL,       NULL,       /* 4C */
        NULL,       NULL,       NULL,       NULL,       /* 50 */
        NULL,       NULL,       NULL,       NULL,       /* 54 */
        NULL,       NULL,       NULL,       NULL,       /* 58 */
        NULL,       NULL,       NULL,       NULL,       /* 5C */
        NULL,       NULL,       NULL,       NULL,       /* 60 */
        NULL,       NULL,       NULL,       NULL,       /* 64 */
        NULL,       NULL,       NULL,       NULL,       /* 68 */
        NULL,       NULL,       NULL,       NULL,       /* 6C */
        NULL,       NULL,       NULL,       NULL,       /* 70 */
        NULL,       NULL,       NULL,       NULL,       /* 74 */
        NULL,       NULL,       NULL,       NULL,       /* 78 */
        NULL,       NULL,       NULL,       NULL,       /* 7C */
        NULL,       NULL,       NULL,       NULL,       /* 80 */
        NULL,       NULL,       NULL,       NULL,       /* 84 */
        NULL,       NULL,       NULL,       NULL,       /* 88 */
        NULL,       NULL,       NULL,       NULL,       /* 8C */
        NULL,       NULL,       NULL,       NULL,       /* 90 */
        NULL,       NULL,       NULL,       NULL,       /* 94 */
        NULL,       NULL,       NULL,       NULL,       /* 98 */
        NULL,       NULL,       NULL,       NULL,       /* 9C */
        NULL,       NULL,       NULL,       NULL,       /* A0 */
        NULL,       NULL,       NULL,       NULL,       /* A4 */
        NULL,       NULL,       NULL,       NULL,       /* A8 */
        NULL,       NULL,       NULL,       NULL,       /* AC */
        NULL,       NULL,       NULL,       NULL,       /* B0 */
        NULL,       NULL,       "TAB ",     NULL,       /* B4 */
        "CSR ",     "REV ",     "NORM ",    "ALL ",     /* B8 */
        " AS ",     "APPEND ",  NULL,       "OFF",      /* BC */
        " STEP ",   " TO ",     "USING ",   "NOT ",     /* C0 */
        " AND ",    " OR ",     " XOR ",    " MOD ",    /* C4 */
        NULL,       NULL,       NULL,       NULL,       /* C8 */
        NULL,       NULL,       NULL,       NULL        /* CC */
    }
};


/*
 *  PB-1000 system variables
 */
#define HIMEM 0x6941	/* begin of BASIC program space */
#define BASEN 0x6943	/* end of BASIC program space */
#define MEMEN 0x6945	/* end of the data space */
#define DATDI 0x6947	/* directory for data files */
#define BASDI 0x6949	/* directory for BASIC programs */
#define DIREN 0x694B	/* end of directory space */
#define ONFIL0 0x6F47	/* Clock boot file address */
#define ONFIL1 0x6F49	/* Preset file address */
#define ONFIL2 0x6F4B	/* Preset file address */
#define ONFIL3 0x6F4D	/* Preset file address */
#define ONFIL4 0x6F4F	/* Preset file address */


/*
 *  Other global variables
 */
char *MyName = "ramtrans";
int MustUpdate = FALSE;


/* buffer for the RAM image (ram0.bin + ram1.bin) */
#define IMGSTART 0x6000
#define IMGSIZE 0xA000
byte image[IMGSIZE];


/* buffer for a signle text line */
#define MAXLINE 255
char textline[MAXLINE];


/*
 *  Local functions
 */
int usage( void );
int doCmd( int argc, char **argv, OPTIONS *options );
int parseOptions( int *argcp, char ***argvp, OPTIONS *options, char *opts );
int cmdDir(  int argc, char **argv, OPTIONS *options );
int cmdType( int argc, char **argv, OPTIONS *options );
int cmdGet(  int argc, char **argv, OPTIONS *options );
int cmdMget( int argc, char **argv, OPTIONS *options );
int cmdPut(  int argc, char **argv, OPTIONS *options );
int cmdMput( int argc, char **argv, OPTIONS *options );
int cmdDel(  int argc, char **argv, OPTIONS *options );
int cmdRen(  int argc, char **argv, OPTIONS *options );
char *getFile( FILE_INFO *info, char *dest, MODE mode );
FILE_INFO *putFile( char *source, char *dest, MODE mode,
                    CASE setCase, int type, int noUpdate );
void delFile( FILE_INFO *info );
FILE_INFO *renFile( FILE_INFO *info, char *newName, CASE setCase );
int printFile( FILE_INFO *info, MODE mode, FILE *out );
void printToken( int c, FILE *out );
FILE_INFO *fileInfo( DIR_ENTRY *dptr, FILE_INFO *info, CASE setCase );
DIR_ENTRY *findFile( char *pattern, int ignoreCase, int type );
int isWildcard( char *pattern );
void expandPattern( char *pattern, char expPattern[ 8 + 3 + 1 ],
                    CASE setCase );
word diskFree( void );
int loadFile ( char *name, int size, byte *buf );
int saveFile ( char *name, int size, byte *buf );
byte getByte( word address );
word getWord( word address );
void putWord( word address, word data );
void addWord( word address, word data );
DIR_ENTRY *getDirPtr( word sysvar );
int makeRoom( int type, word size );
void updateOnFile( void );
int checkRamDisk( void );


/*
 *  Main entry point
 */
int main( int argc, char **argv )
{
    int result;
    OPTIONS options = DEFAULT_OPTIONS;

/*    MyName = *argv; */

    ++argv;
    --argc;

    /*
     *  Parse option -n which is allowed before the command
     */
    if ( OK != parseOptions( &argc, &argv, &options, "n" ) ) {
        return usage();
    }

    if ( argc < 1 ) {
        return usage();
    }

    /*
     *  Load the RAM image
     */
    if ( OK != loadFile( "ram0.bin", 8192, image ) ) {
        return 2;
    }
    if ( OK != loadFile( "ram1.bin", 32768, image+8192 ) ) {
        return 2;
    }

    /*
     *  Test the memory integrity
     */
    if ( OK != checkRamDisk() ) {
        fprintf( stderr, "Incorrect data\n" );
        return 2;
    }

    /*
     *  Execute command
     */
    result = doCmd( argc, argv, &options );

    if ( result == OK && MustUpdate ) {
        /*
         *  Update the ONFILx system variables
         */
        updateOnFile();

        /*
         *  Save the RAM image
         */
        if ( OK != saveFile( "ram0.bin", 8192, image ) ) {
            return 2;
        }
        if ( OK != saveFile( "ram1.bin", 32768, image+8192 ) ) {
            return 2;
        }
    }

    if ( result != OK ) {
        fprintf( stderr, "\007Command failed\n" );
        return 2;
    }
    return 0;
}


/*
 *  Help
 */
int usage( void )
{
    fprintf( stderr,
             "usage: %s [-n] <cmd> <options> <parameters>\n"
             "       <cmd> <parameters> is one of:\n"
             "         dir  <options> <ramdisk-pattern>\n"
             "         type <options> <ramdisk-file>\n"
             "         get  <options> <ramdisk-file> [<pc-file>]\n"
             "         mget <options> <ramdisk-pattern>\n"
             "         put  <options> <pc-file> [<ramdisk-file>]\n"
             "         mput <options> <pc-files>\n"
             "         del  <options> <ramdisk-pattern>\n"
             "         ren  <options> <ramdisk-file> <new name>\n"
             "       <options> are:\n"
             "         -i   ignore the case of ramdisk file(s)\n"
             "         -l   make all files lowercase\n"
             "         -u   make all files uppercase\n"
             "         -tX  select type to X (B, M, S or hex)\n"
             "         -b   force binary transfer\n"
             "         -a   force ASCII transfer\n"
             "         -n   no updates are written to the image\n",
             MyName );
    return 2;
}


/*
 *  Parse the command argument and call routine
 */
int doCmd( int argc, char **argv, OPTIONS *options )
{
    static struct _cmds {
        char *cmd;
        int (*fn)( int, char **, OPTIONS * );
        char *opts;
    } cmds[] = {
        /* cmd    routine   options   */
        { "dir",  cmdDir,  "nt  ilu " },
        { "type", cmdType, "ntabi   " },
        { "get",  cmdGet,  "ntabilud" },
        { "mget", cmdMget, "ntabilud" },
        { "put",  cmdPut,  "ntab lu " },
        { "mput", cmdMput, "ntab lud" },
        { "del",  cmdDel,  "nt  i   " },
        { "ren",  cmdRen,  "nt  ilu " },
        { NULL,   NULL,    NULL        }
    };
    struct _cmds *cptr = cmds;
    char *cmd;
    int result;

    /*
     *  First argument is command name, defaults to "dir"
     */
    if ( argc == 0 ) {
        cmd = "dir";
    }
    else {
        cmd = argv[ 0 ];
        ++argv;
        --argc;
    }

    /*
     *  Find the command
     */
    while ( cptr->cmd != NULL && 0 != stricmp( cptr->cmd, cmd ) ) {
        ++cptr;
    }

    if ( cptr->fn == NULL ) {
        /*
         *  not found
         */
        fprintf( stderr, "%s: Command \"%s\" not found\n", MyName, cmd );
        return usage();
    }

    /*
     *  parse the standard options
     */
    if ( OK != parseOptions( &argc, &argv, options, cptr->opts ) ) {
        return usage();
    }

    /*
     *  Execute the command
     */
    result = (*cptr->fn)( argc, argv, options );

    if ( result != OK || options->noUpdate ) {
        /*
         *  Updates are disabled
         */
        MustUpdate = FALSE;
    }

    return result;
}


/*
 *  Parse options -i, -u, -l, -tX, -a, -b, -d, -n
 */
int parseOptions( int *argcp, char ***argvp, OPTIONS *options, char *opts )
{
    TYPES *tptr;

    while ( *argcp > 0 && ***argvp == '-' ) {
        /*
         *  All options are seperate and start with "-"
         */
        char *arg = **argvp;
        char opt = tolower( arg[ 1 ] );

        if ( opt == 'n' ) {
            /*
             *  No updates please!
             */
            options->noUpdate = TRUE;
        }
        else if ( opt == 'i' ) {
            /*
             *  Ignore case
             */
            options->ignoreCase = TRUE;
        }
        else if ( opt == 'l' ) {
            /*
             *  Make filenames lowercase
             */
            options->setCase = LOWER;
        }
        else if ( opt == 'u' ) {
            /*
             *  Make filenames uppercase
             */
            options->setCase = UPPER;
        }
        else if ( opt == 'a' ) {
            /*
             *  Transfers are ASCII
             */
            options->mode = ASCII;
        }
        else if ( opt == 'b' ) {
            /*
             *  Transfers are binary
             */
            options->mode = BINARY;
        }
        else if ( opt == 't' ) {
            /*
             *  Type
             */
            if ( arg[ 2 ] != '\0' ) {
                arg = arg + 2;
            }
            else if ( *argcp >= 2 ) {
                ++(*argvp);
                --(*argcp);
                arg = **argvp;
            }
            else {
                fprintf( stderr, "Missing value for -t option\n" );
                return NOT_OK;
            }
            if ( strlen( arg ) == 1 ) {
                /*
                 *  Convert type
                 */
                *arg = toupper( *arg );
                options->typeS[ 0 ] = *arg;

                for ( tptr = Types; tptr->type != 0; ++tptr ) {
                    if ( *arg == *tptr->typeName ) {
                        /*
                         *  Known type
                         */
                        options->type = tptr->type;
                        break;
                    }
                }
                if ( tptr->type == 0 ) {
                    /*
                     *  Unknown type
                     */
                    fprintf( stderr, "Invalid type for -t option\n" );
                    return NOT_OK;
                }
            }
            else {
                /*
                 *  Hex type
                 */
                int type;
                if ( 1 != sscanf( arg, "%2X", &type ) ) {
                    fprintf( stderr, "Invalid type for -t option\n" );
                    return NOT_OK;
                }
                options->type = (byte) type;
                sprintf( options->typeS, "%02X", type );
            }
        }
        else if ( opt == 'd' ) {
            /*
             *  Destination
             */
            if ( arg[ 2 ] != '\0' ) {
                arg = arg + 2;
            }
            else if ( *argcp >= 2 ) {
                ++(*argvp);
                --(*argcp);
                arg = **argvp;
            }
            else {
                fprintf( stderr, "Missing value for -d option\n" );
                return NOT_OK;
            }
            options->destination = arg;
        }
        else {
            /*
             *  unknown option
             */
            fprintf( stderr, "Unknown option %s\n", arg );
            return NOT_OK;
        }
        if ( opts != NULL && NULL == strchr( opts, opt ) ) {
            /*
             *  Option not allowed here
             */
            fprintf( stderr, "Option -%c not allowed\n", opt );
            return NOT_OK;
        }

        ++(*argvp);
        --(*argcp);
    }
    return OK;
}


/*
 *  dir <options> <ramdisk-pattern>
 */
int cmdDir( int argc, char **argv, OPTIONS *options )
{
    char *pattern, *p;
    FILE_INFO *info = NULL;
    word total = 0;
    int files = 0;
    int count;

    if ( argc == 0 ) {
        /*
         *  Default "*.*"
         */
        pattern = "*.*";
        argc = 1;
    }
    else {
        /*
         *  First pattern in list
         */
        pattern = argv[ 0 ];
    }

    printf( "\n Directory\n\n" );
    printf( "   Name     Type  Size\n" );

    /*
     *  List directory for all patterns on command line
     */
    while ( argc-- > 0 ) {
        /*
         *  Loop over all files
         */
        count = 0;
        p = pattern;
        while ( TRUE ) {
            info = fileInfo( findFile( p, options->ignoreCase, options->type),
                             info, options->setCase );
            if ( info == NULL ) {
                break;
            }
            p = NULL;
            ++count;
            ++files;

            printf( "%-12s %2s  %5u\n",
                    info->name, info->typeS, info->size );
            total += info->size;
        }
        if ( count == 0 ) {
            printf( "%-12s %2s  no files\n", pattern, options->typeS );
        }
        pattern = *++argv;
    }

    /*
     *  Statistics
     */
    printf( "\n Total:    %4d  %5u\n", files, total );
    printf( " Free:           %5u\n", diskFree() );

    return OK;
}


/*
 *  type <options> <ramdisk-pattern>
 */
int cmdType( int argc, char **argv, OPTIONS *options )
{
    char *pattern;
    FILE_INFO *info = NULL;
    int printNames;
    int count;
    char *p;

    if ( argc < 1 ) {
        return usage();
    }
    pattern = *argv;
    printNames = argc > 1 || isWildcard( pattern );

    /*
     *  Type all files for all patterns on command line
     */
    while ( argc-- > 0 ) {
        /*
         *  Loop over all files
         */
        count = 0;
        p = pattern;
        while ( TRUE ) {
            info = fileInfo( findFile( p, options->ignoreCase, options->type),
                             info, options->setCase );
            if ( info == NULL ) {
                break;
            }
            p = NULL;
            ++count;

            if ( printNames ) {
                printf( "\nFile %s, Type %s:\n\n", info->name, info->typeS );
            }
            if ( OK != printFile( info, options->mode, stdout ) ) {
                fprintf( stderr, "%s: Error reading file\n", info->name );
                return NOT_OK;
            }
        }
        if ( count == 0 ) {
            fprintf( stderr, "%s: file not found\n", pattern );
        }
        pattern = *++argv;
    }
    return OK;
}


/*
 *  get <options> <ramdisk-file> [<destination>]
 */
int cmdGet( int argc, char **argv, OPTIONS *options )
{
    char *source;
    char *dest;
    FILE_INFO *info;

    if ( argc < 1 || argc > 2 ) {
        return usage();
    }
    source = *argv;
    dest = argc == 2 ? argv[ 1 ] : options->destination;

    /*
     *  Copy a single file
     */
    info = fileInfo( findFile( source, options->ignoreCase, options->type ),
                     NULL, options->setCase );
    if ( info == NULL ) {
        fprintf( stderr, "%s: File not found\n", source );
        return NOT_OK;
    }
    dest = getFile( info, dest, options->mode );
    if ( dest == NULL ) {
        fprintf( stderr, "%s: Error copying file\n", info->name );
        return NOT_OK;
    }
    fprintf( stderr, "%-12s %2s copied to %s\n",
                     info->name, info->typeS, dest );
    return OK;
}


/*
 *  type <options> -d <destination> <ramdisk-pattern> 
 */
int cmdMget( int argc, char **argv, OPTIONS *options )
{
    char *pattern;
    FILE_INFO *info = NULL;
    int count;
    int total = 0;
    char *p;
    char *dest;

    if ( argc < 1 ) {
        return usage();
    }
    pattern = *argv;

    /*
     *  Type all files for all patterns on command line
     */
    while ( argc-- > 0 ) {
        /*
         *  Loop over all files
         */
        count = 0;
        p = pattern;
        while ( TRUE ) {
            info = fileInfo( findFile( p, options->ignoreCase, options->type),
                             info, options->setCase );
            if ( info == NULL ) {
                break;
            }
            p = NULL;
            ++count;

            dest = getFile( info, options->destination, options->mode );
            if ( dest == NULL ) {
                fprintf( stderr, "%s: Error copying file\n", info->name );
                return NOT_OK;
            }
            printf( "%-12s %2s copied to %s\n",
                    info->name, info->typeS, dest );
        }
        if ( count == 0 ) {
            fprintf( stderr, "%s: file not found\n", pattern );
        }
        total += count;
        pattern = *++argv;
    }
    if ( total > 0 ) {
        printf( "%d file%s copied\n", total, total == 1 ? "" : "s" );
    }
    else {
        printf( "No files copied\n" );
    }
    return OK;
}


/*
 *  put <options> <pc-file> [<ramdisk-file>]
 */
int cmdPut( int argc, char **argv, OPTIONS *options )
{
    char *source;
    char *dest;
    FILE_INFO *info;

    if ( argc < 1 || argc > 2 ) {
        return usage();
    }
    source = *argv;
    dest = argc == 2 ? argv[ 1 ] : NULL;

    /*
     *  Copy a single file
     */
    info = putFile( source, dest, options->mode,
                                  options->setCase,
                                  options->type,
                                  options->noUpdate );
    if ( info->error != NO_ERROR ) {
        fprintf( stderr, "%s: Error copying file: %s\n",
                         source, Error[ info->error ] );
        return NOT_OK;
    }
    fprintf( stderr, "%-12s %2s created from %s\n",
                     info->name, info->typeS, source );
    return OK;
}


/*
 *  mput <options> <pc-file> ...
 */
int cmdMput( int argc, char **argv, OPTIONS *options )
{
    char *source;
    char *dest;
    FILE_INFO *info;

    if ( argc < 1 ) {
        return usage();
    }
    dest = options->destination;

    /*
     *  Copy all files from command line
     */
    while ( argc-- > 0 ) {

        source = *argv++;

        info = putFile( source, dest, options->mode,
                                      options->setCase,
                                      options->type,
                                      options->noUpdate );
        if ( info->error != NO_ERROR ) {
            fprintf( stderr, "%s  Error copying file: %s\n",
                             source, Error[ info->error ] );
            return NOT_OK;
        }
        fprintf( stderr, "%-12s %2s created from %s\n",
                         info->name, info->typeS, source );
    }
    return OK;
}


/*
 *  del <options> <ramdisk-pattern>
 */
int cmdDel( int argc, char **argv, OPTIONS *options )
{
    char *pattern;
    FILE_INFO *info = NULL;
    int count;
    int total = 0;
    char *p;

    if ( argc < 1 ) {
        return usage();
    }
    pattern = *argv;

    /*
     *  Delete all files for all patterns on command line
     */
    while ( argc-- > 0 ) {
        /*
         *  Loop over all files
         */
        count = 0;
        p = pattern;
        while ( TRUE ) {
            info = fileInfo( findFile( p, options->ignoreCase, options->type),
                             info, options->setCase );
            if ( info == NULL ) {
                break;
            }
            ++count;

            delFile( info );
            printf( "%-12s %2s deleted\n", info->name, info->typeS );
        }
        if ( count == 0 ) {
            fprintf( stderr, "%s: file not found\n", pattern );
        }
        total += count;
        pattern = *++argv;
    }
    if ( total > 0 ) {
        printf( "%d file%s deleted\n", total, total == 1 ? "" : "s" );
    }
    else {
        printf( "No files deleted\n" );
        MustUpdate = FALSE;
    }
    return OK;
}


/*
 *  ren <options> <ramdisk-pattern> <new-name-pattern>
 */
int cmdRen( int argc, char **argv, OPTIONS *options )
{
    char *pattern;
    char *newName;
    FILE_INFO *info = NULL;
    FILE_INFO *newInfo;
    int count;
    int unchanged;
    int total = 0;
    char *p;

    if ( argc < 2 ) {
        return usage();
    }
    pattern = *argv;
    newName = argv[ --argc ];

    /*
     *  Rename all files for all but one patterns on the command line
     *  to the name given by the last pattern
     */
    while ( argc-- > 0 ) {
        /*
         *  Loop over all files
         */
        count = 0;
        unchanged = 0;
        p = pattern;
        while ( TRUE ) {
            info = fileInfo( findFile( p, options->ignoreCase, options->type),
                             info, AS_IS );
            if ( info == NULL ) {
                break;
            }
            p = NULL;

            newInfo = renFile( info, newName, options->setCase );
            if ( newInfo->error == FILE_EXISTS ) {
                printf( "%-12s %-2s duplicate: %-12s %-2s\n",
                        info->name, info->typeS,
                        newInfo->name, newInfo->typeS );
                return NOT_OK;
            }
            ++count;

            if ( 0 == strcmp( info->name, newInfo->name ) ) {
                ++unchanged;
                printf( "%-12s %-2s not changed\n", info->name, info->typeS );
            }
            else {
                printf( "%-12s %-2s renamed to %-12s %-2s\n",
                        info->name, info->typeS,
                        newInfo->name, newInfo->typeS );
            }
        }
        if ( count == 0 ) {
            fprintf( stderr, "%s: file not found\n", pattern );
        }
        total += count - unchanged;
        pattern = *++argv;
    }
    if ( total > 0 ) {
        printf( "%d file%s renamed\n", total, total == 1 ? "" : "s" );
    }
    else {
        printf( "No files renamed\n" );
        MustUpdate = FALSE;
    }
    return OK;
}


/*
 *  Expand a file-pattern, DOS like, "*" is equivalent to "*.*"
 */
void expandPattern( char *pattern, char expPattern[ 8 + 3 + 1 ], 
                    CASE setCase )
{
    int i;
    char c;

    expPattern[ 8 + 3 ] = '\0';

    if ( pattern == NULL || 0 == strcmp( pattern, "*" ) ) {
        /*
         *  "*": all files
         */
        memset( expPattern, '?', 8 + 3 );
        return;
    }

    /*
     *  Expand the pattern
     */
    i = 0;
    memset( expPattern, ' ', 8 + 3 );

    while ( *pattern != '\0' && i < 8 + 3 ) {
        if ( *pattern == '*' ) {
            if ( i < 8 ) {
                memset( expPattern + i, '?', 8 - i );
                i = 8;
            }
            else {
                memset( expPattern + i, '?', 8 + 3 - i );
                i = 8 + 3;
            }
        }
        else if ( *pattern == '.' ) {
            i = i < 8 ? 8 : 8 + 3;
        }
        else {
            expPattern[ i ] = setCase == UPPER ? toupper( *pattern )
                            : setCase == LOWER ? tolower( *pattern )
                                               : *pattern;
            ++i;
        }
        if ( i == 8 ) {
            while ( pattern[ 1 ] != '\0' && *pattern != '.' ) {
                ++pattern;
            }
        }
        ++pattern;
    }
#if DEBUG
    printf( "expandPattern: expanded pattern=\"%s\"\n", expPattern );
#endif
}


/*
 *  Copy a file to PC
 */
char *getFile( FILE_INFO *info, char *dest, MODE mode )
{
    int result;
    int l;
    FILE *out;
    int binary;
    word skip;
    
    /*
     *  Build destination filename
     */
    if ( dest == NULL ) {
        dest = "*";
    }
    l = strlen( dest );
    if ( dest[ l - 1 ] == '*'
      || dest[ l - 1 ] == '/'
      || dest[ l - 1 ] == '\\' )
    {
        /*
         *  Destination ends with directory seperator or "*"
         *  Concatenate with source filename
         */
        static char tmp[ FILENAME_MAX + 13 ];
        strncpy( tmp, dest, FILENAME_MAX );
        dest = tmp;
        dest[ FILENAME_MAX - 1 ] = '\0';
        l = strlen( dest );
        if ( dest[ l - 1 ] == '*' ) {
            --l;
        }
        strcpy( dest + l, info->name );

        /*
         *  Remove invalid characters
         */
        while ( dest[ l ] != '\0' ) {

            if ( NULL != strchr( ":\\/*?[]", dest[ l ] ) ) {
                dest[ l ] = '_';
            }
            ++l;
        }
    }
#if DEBUG
    printf( "Copy %s %s\n", info->name, dest );
#endif
    if ( mode == ASCII ) {
        binary = info->type == TYPE_M;
    }
    else if ( mode == BINARY ) {
        binary = TRUE;
    }
    else {
        binary = info->type != TYPE_S;
    }

    out = fopen( dest, binary ? "wb" : "wt" );
    if ( out == NULL ) {
        perror( dest );
        return NULL;
    }

    errno = 0;

    if ( binary ) {
        /*
         *  Transparent copy without the last EOF byte
         */
        skip = ( image[info->address + info->size - IMGSTART - 1] == '\032' )
            ? 1 : 0;
        if ( (size_t) (info->size - skip) !=
        fwrite( image - IMGSTART + info->address, sizeof( byte),
            (size_t) (info->size - skip), out ) ) {
            result = NOT_OK;
        }
    }
    else {
        /*
         *  "Print" to file
         */
        result = printFile( info, ASCII, out );
    }
    (void) fclose( out );

    if ( errno != 0 ) {
        /*
         *  Error
         */
        info->error = IO_ERROR;
        perror( dest );
        return NULL;
    }

    return result == NOT_OK ? NULL : dest;
}


/*
 *  Copy a file to the ramdisk
 */
FILE_INFO *putFile( char *source, char *dest, MODE mode,
                    CASE setCase, int type, int noUpdate )
{
    long length;
    FILE *in;
    int i;
    char *p;
    FILE_INFO *info;
    DIR_ENTRY newEntry;
    char ext[ 3 ];
    DIR_ENTRY *dptr;
    word count;

    MustUpdate = FALSE;

    /*
     *  Clear newEntry
     */
    memset ( &newEntry, 0, sizeof( DIR_ENTRY ) );

    /*
     *  Find base name of source
     */
    i = (int) strlen( source );
    while ( --i > 0 ) {
        if ( source[ i - 1 ] == '\\' || source[ i - 1 ] == '/' ) {
            break;
        }
    }

    /*
     *  Combine source and destination name
     */
    memset( &newEntry, '\0', sizeof( DIR_ENTRY ) );
    expandPattern( dest, newEntry.name, setCase );
    memcpy( ext, newEntry.ext, 3 );

    for ( p = source + i, i = 0; i < 8 + 3 && *p != '\0'; ++i, ++p ) {

        char c = setCase == UPPER ? toupper( *p )
               : setCase == LOWER ? tolower( *p )
               : *p;

        if ( c == '.' ) {
            /*
             *  Start of extension, restore pattern for last three bytes
             */
            memcpy( newEntry.name + 8, ext, 3 );
            i = 7;
            continue;
        }
        if ( newEntry.name[ i ] == '?' ) {
            /*
             *  Replace wildcard with character from source filename
             */
            newEntry.name[ i ] = c;
        }
    }

    /*
     *  Get rid of remaining wildcards
     */
    for ( i = 0; i < 8 + 3; ++i ) {
        if ( newEntry.name[ i ] == '?' ) {
            newEntry.name[ i ] = ' ';
        }
    }

    /*
     *  Set type
     */
    if ( type <= 0 ) {
        type = TYPE_S;	/* Default: Sequential file */
        for ( i = 0; i < sizeof( dft ) / sizeof( DEFAULTFILETYPES ); i++ ) {
            if ( 0 == memicmp( newEntry.ext, dft[ i ].ext, 3 ) ) {
               type = dft[ i ].type;
               break;
            }
        }
    }
    newEntry.type = type;

    /*
     *  Transfer mode
     */
    if ( mode == AUTO ) {

        if ( type == TYPE_M || type == TYPE_B ) {
            mode = BINARY;
        }
        else {
            mode = ASCII;
        }
    }

    /*
     *  Fill file information block with newly created directory entry
     *  This is preliminary to care for early exit (NO_ROOM)
     */
    info = fileInfo( &newEntry, NULL, AS_IS );

#if DEBUG
    printf( "Source: %s, New: %s\n", source, info->name );
#endif

    /*
     *  Open source file
     */
    in = fopen( source, mode == ASCII ? "rt" : "rb" );

    if ( in == NULL ) {
        perror( source );
        info->error = FILE_NOT_FOUND;
        return info;
    }

    /*
     *  Determine file size for a quick check
     */
    if ( fseek( in, 0, SEEK_END ) != 0 ||
         ( length = ftell( in ) ) < 0 ||
         length > 65535L )
    {
        perror( source );
        info->error = IO_ERROR;
        fclose( in );
        return info;
    }

    /*
     *  Check for entries with same name
     */
    for ( dptr = getDirPtr( DATDI );
          dptr < getDirPtr( DIREN );
          dptr++ ) {

        if ( 0 == memcmp( dptr->name, newEntry.name, 8 + 3 ) ) {
            /*
             *  Remove existing file of the same name
             */
            info = fileInfo( dptr, info, AS_IS );
#if DEBUG
            printf( "Found %-12s %-2s\n", info->name, info->typeS );
#endif
            delFile( info );
            break;
        }
    }

    /*
     *  Reposition source to start of file
     */
    if ( fseek( in, 0, SEEK_SET ) != 0 ) {
        perror( source );
        info->error = IO_ERROR;
        (void) fclose( in );
        return info;
    }

    newEntry.srcbegin = getWord( ( type == TYPE_B ) ? BASEN : MEMEN );
    newEntry.srcend = newEntry.srcbegin;
    newEntry.pass[0] = '\377';

    if ( mode == BINARY ) {

        count = (word) length;

        /*
         *  Make room for new file
         */
        if ( makeRoom( type, count ) != OK ) {
            info->error = NO_ROOM;
            fclose( in );
            return info;
        }

        /*
         *  Now copy the data
         */
        if ( fread( image - IMGSTART + newEntry.srcend,
             sizeof( byte ), (size_t) count, in ) !=
             (size_t) count ) {
            info->error = IO_ERROR;
            perror( source );
            (void) fclose( in );
            return info;
        }

        newEntry.srcend += count;

    }
    else {	/* ASCII mode */

        while ( fgets( textline, MAXLINE-1, in ) != NULL ) {

            /*
             *  Append CR,LF at the end of the textline
             */
            count = strlen( textline );
            while ( count > 0 && iscntrl( textline[ count - 1 ] ) ) {
                count--;
            }
            textline[ count++ ] = '\r';
            textline[ count++ ] = '\n';

            /*
             *  Make room for new line
             */
            if ( makeRoom( type, count ) != OK ) {
                info->error = NO_ROOM;
                fclose( in );
                return info;
            }

            /*
             *  Now copy the data
             */
            memcpy( image - IMGSTART + newEntry.srcend, textline,
                    (size_t) count );
            newEntry.srcend += count;

        }

    }

    /*
     *  Append Ctrl-Z at the end of the file regardless of the type
     */
    if ( makeRoom( type, 1 ) != OK ) {
        info->error = NO_ROOM;
        fclose( in );
        return info;
    }
    image[ - IMGSTART + newEntry.srcend ] = '\032';
    newEntry.srcend++;

    /*
     *  Check if there's enough free space for a new directory entry
     */
    if ( sizeof( DIR_ENTRY ) > diskFree() ) {
        info->error = NO_ROOM;
        fclose( in );
        return info;
    }

    /*
     *  Make room for new directory entry
     */
    if ( type == TYPE_B ) {
        dptr = getDirPtr( DATDI );
        memmove ( dptr-1, dptr,
            (size_t) ( getWord( BASDI ) - getWord( DATDI ) ) );
        addWord( BASDI, (word) ( - sizeof( DIR_ENTRY ) ) );
    }
    addWord( DATDI, (word) ( - sizeof( DIR_ENTRY ) ) );

    /*
     *  Update the directory
     */
    dptr = getDirPtr( ( type == TYPE_B ) ? BASDI : DATDI );
    info->entry = dptr;
    memcpy( dptr, &newEntry, sizeof( DIR_ENTRY ) );

    /*
     *  Done
     */
    (void) fclose( in );
    MustUpdate = ! noUpdate;
    return info;
}


/*
 *  Delete a file.
 */
void delFile( FILE_INFO *info )
{
    DIR_ENTRY *dptr;

    /*
     *  Collapse the memory block occupied by the file
     */
    memmove ( image - IMGSTART + info->address,
              image - IMGSTART + info->address + info->size,
              (size_t) ( getWord( MEMEN ) - info->address - info->size ) );
    if ( info->address < getWord( BASEN ) ) {
        addWord( BASEN, - info->size );
    }
    addWord( MEMEN, - info->size );

    /*
     *  Delete directory entry
     */
    dptr = getDirPtr( DATDI );
    addWord( DATDI, sizeof( DIR_ENTRY ) );
    if ( info->entry >= getDirPtr( BASDI ) ) {
        addWord( BASDI, sizeof( DIR_ENTRY ) );
    }
    memmove ( dptr+1, dptr,
        (size_t) ( ( info->entry - dptr ) * sizeof( DIR_ENTRY ) ) );

    /*
     *  Adjust the data position for remaining directory entries
     */
    for ( dptr = getDirPtr( DATDI );
          dptr < getDirPtr( DIREN );
          dptr++ ) {
        if ( dptr->srcbegin <= info->address ) {
            break;
        }
        dptr->srcbegin -= info->size;
        dptr->srcend -= info->size;
    }
    MustUpdate = TRUE;
}


/*
 *  Rename a file
 */
FILE_INFO *renFile( FILE_INFO *info, char *newName, CASE setCase )
{
    char newPattern[ 8 + 3 + 1 ];
    static FILE_INFO newInfo;
    word i;
    char c;
    int found;
    DIR_ENTRY *dptr;

    /*
     *  New name may contain wildcards
     */
    expandPattern( newName, newPattern, setCase );

    for ( i = 0; i < 8 + 3; ++i ) {
        /*
         *  Update each character in name
         */
        c = newPattern[ i ];
        if ( c == '?' ) {
            c = info->entry->name[ i ];
        }
        newPattern[ i ] = setCase == UPPER ? toupper( c )
                        : setCase == LOWER ? tolower( c )
                        : c;
    }

    /*
     *  Check if name exists
     */
    found = FALSE;
    for ( dptr = getDirPtr( DATDI );
          dptr < getDirPtr( DIREN );
          dptr++ ) {
        /*
         *  Find used entry with same name,
         *  but don't stumble over current entry!
         */
        found = dptr != info->entry
             && 0 == memcmp( dptr->name, newPattern, 8 + 3 );
        if ( found ) {
            break;
        }
    }

    if ( found ) {
        /*
         *  Duplicate, return information for existing file
         */
        fileInfo( dptr, &newInfo, AS_IS );
        newInfo.error = FILE_EXISTS;
    }
    else {
        /*
         *  Update the name
         */
        memcpy( info->entry->name, newPattern, 8 + 3 );
        fileInfo( info->entry, &newInfo, AS_IS );
        MustUpdate = TRUE;
    }
    return &newInfo;
}


/*
 *  Print a file
 */
int printFile( FILE_INFO *info, MODE mode, FILE *out )
{
    word start, count;
    word i, j, l;
    word addr = 0;
    byte c;
    int pos = 0;
    int length = 0;
    int line_nr = 0;

    start = info->address;
    count = info->size;

    switch ( mode == BINARY ? TYPE_M : info->type ) {
        
    case TYPE_S:
        /*
         *  Text file
         */
        for ( i = 0; i < count; ++i ) {
            c = getByte( start + i );
            if ( c == '\r' ) {
                continue;
            }
            fputc( c, out );
        }
        break;

    case TYPE_B:
        /*
         *  BASIC file
         */
        for ( i = 0; i < count; ++i ) {
            c = getByte( start + i );
            if ( --length == -1 ) {
                pos = 0;
            }

            switch ( pos++ ) {

            case 0:
                length = c;
                continue;

            case 1:
                line_nr = c;
                continue;

            case 2:
                line_nr += c * 256;
                fprintf( out, "%d ", line_nr );
                continue;

            case 3:
                if ( c == ' ' ) {
                     continue;
                }
                break;
            }
            printToken( c, out );
        }
        break;

    default:
        /*
         *  Binary file
         */
        for ( i = 0; i < count; i += 16 ) {
            fprintf( out, "%04X: ", addr );
            addr += 16;
            l = count - i < 16 ? count - i : 16;
            for ( j = 0; j < l; ++j ) {
                fprintf( out, "%02X ", getByte( start + i + j ) );
            }
            while ( j++ < 16 ) {
                fprintf( out, "   " );
            }
            fputc( ' ', out );
            for ( j = 0; j < l; ++j ) {
                char c = getByte( start + i + j );
                fputc( c < ' ' || c >= 0x7F ? ' ' : c, out );
            }
            fprintf( out, "\n" );
        }
        break;
    }

    return OK;
}


/*
 *  Print a character from a BASIC program, translate tokens
 */
void printToken( int c, FILE *out )
{
    static int insert_space = FALSE;
    static int pending_colon = FALSE;
    static int last = 0;
    static int quoted = FALSE;
    static int quoted_until_eol = FALSE;
    char *p = NULL;
    char buffer[] = "[x:xx]";
    static int prefix = 0;
    static int lsb = -1;

    if ( prefix == 3 ) {
        /*
         *  Binary line number
         */
        if ( lsb == -1 ) {
            lsb = c;
            c = -1;
        }
        else {
            sprintf( buffer, "%d", lsb + c * 256 );
            p = buffer;
            lsb = -1;
            prefix = 0;
        }
    }
    else {
        /*
         *  Text or token
         */
        switch ( c ) {

        case 0x00:
            p = "\n";
            break;

        case 0x01:
            pending_colon = TRUE;
            c = -1;
            break;

        case 0x02:
            p = "'";
            quoted_until_eol = TRUE;
            break;

        case 0x03:
            prefix = c;
            lsb = -1;
            c = -1;
            break;

        case 0x04:
        case 0x05:
        case 0x06:
        case 0x07:
            prefix = c;
            c = -1;
            break;

        default:
            if ( prefix != 0 ) {
                /*
                 *  Token
                 */
                if ( prefix < 4 || prefix > 7 || c < 0x40 || c > 0xCF ) {
                    p = NULL;
                }
                else {
                    p = Tokens[ prefix - 4 ][ c - 0x40 ];
                }
                if ( p == NULL ) {
                    sprintf( buffer, "[%d:%02X]", prefix,
                                     (byte) c );
                    p = buffer;
                }
            }
            prefix = 0;
        }
    }

    if ( c < 0 ) {
        /*
         *  Character is already handled
         */
    }
    else if ( c == ' ' ) {
        /*
         *  Literal space is always printed
         */
        if ( pending_colon ) {
            /*
             *  The colon had been postponed
             */
            fputc( ':', out );
            pending_colon = FALSE;
        }
        fputc( c, out );
        insert_space = FALSE;
        last = ' ';
    }
    else {
        if ( p == NULL ) {
            /*
             *  Save character as string and point to it
             */
            buffer[ 0 ] = (char) c;
            buffer[ 1 ] = '\0';
            p = buffer;
        }

        if ( *p == ' ' && last == ' ' ) {
            /*
             *  Collapse spaces in tokens
             */
            ++p;
        }

        if ( pending_colon ) {
            if ( strncmp( p, "ELSE", 4 ) != 0 ) {
                /*
                 *  Print the postponed colon except before ELSE
                 */
                fputc( ':', out );
            }
            pending_colon = FALSE;
        }

        /*
         *  Print the Text
         */
        while ( ( c = *p++ ) != '\0' ) {
            /*
             *  Print character, insert or drop spaces as appropriate
             */
            switch ( c ) {
            
            case '"':
                quoted = !quoted || quoted_until_eol;
                insert_space = FALSE;
                break;

            case '\n':
                quoted = quoted_until_eol = FALSE;
                insert_space = FALSE;
                break;
            }

            if ( quoted ) {
                /*
                 *  Print char as is, no further translation
                 */
                fputc( c, out );
                insert_space = FALSE;
            }
            else {
                /*
                 *  Check for spaces and unprintable characters
                 */
                if ( c < ' ' && c != '\n' ) {
                    fprintf( out, "[%02X]", (byte) c );
                }
                else if ( c == ' ' && *p == '\0' ) {
                    /*
                     *  Token ends with a space
                     */
                    insert_space = TRUE;
                }
                else {
                    /*
                     *  Printable character
                     */
                    if ( insert_space && ( c >= '0' && c <= '9' || c >= 'A' ) ) { 
                        fputc( ' ', out );
                    }
                    fputc( c, out );
                    insert_space = FALSE;
                }
            }
            last = c;
        }
    }
}


/*
 *  Get file information from directory entry
 */
FILE_INFO *fileInfo( DIR_ENTRY *dptr, FILE_INFO *info, CASE setCase )
{
    static FILE_INFO localInfo;
    int i, j, l;
    char c;
    TYPES *tptr;

    if ( dptr == NULL ) {
        return NULL;
    }

    if ( info == NULL ) {
        info = &localInfo;
    }
    memset( info, '\0', sizeof( FILE_INFO ) );

    /*
     *  Link to original entry in directory
     */
    info->entry = dptr;

    /*
     *  Collapse the filename
     */
    for ( l = 8; l > 0; --l ) {
        /*
         *  Find length of basename
         */
        if ( dptr->name[ l - 1 ] != ' ' ) {
            break;
        }
    }

    for ( i = 0, j = 0; i < l; ++i ) {
        /*
         *  Copy basename
         */
        c = dptr->name[ i ];
        info->name[ j++ ] = setCase == LOWER ? tolower( c )
                          : setCase == UPPER ? toupper( c )
                          : c;
    }

    for ( l = 3; l > 0; --l ) {
        /*
         *  Find length of extension
         */
        if ( dptr->ext[ l - 1 ] != ' ' ) {
            break;
        }
    }

    if ( l > 0 ) {
        /*
         *  Append extension
         */
        info->name[ j++ ] = '.';

        for ( i = 0; i < l; ++i ) {
            char c = dptr->ext[ i ];
            info->name[ j++ ] = setCase == LOWER ? tolower( c )
                              : setCase == UPPER ? toupper( c )
                              : c;
        }
    }

    /*
     *  Convert type
     */
    info->type = dptr->type;

    for ( tptr = Types; tptr->type != 0; ++tptr ) {
        if ( info->type == tptr->type ) {
            /*
             *  Known type
             */
            strcpy( info->typeS, tptr->typeName );
            break;
        }
    }

    if ( tptr->type == 0 ) {
        /*
         *  Unknown type
         */
        sprintf( info->typeS, "%02X", dptr->type );
    }

    /*
     *  Store the starting address and the file size
     */
    info->address = dptr->srcbegin;
    info->size = dptr->srcend - dptr->srcbegin;

    /*
     *  All info collected
     */
    return info;
}


/*
 *  Find a file in the directory
 */
DIR_ENTRY *findFile( char *pattern, int ignoreCase, int type )
{
    static char expPattern[ 8 + 3 + 1 ] = "";
    static DIR_ENTRY *dptr = NULL;
    int i;
    char c;
    int count;
    int found = FALSE;


    if ( pattern != NULL ) {
        /*
         *  expand the pattern DOS like
         */
        expandPattern( pattern, expPattern, ignoreCase ? UPPER : AS_IS );

        /*
         *  initialize the pointer
         */
        dptr = getDirPtr( DIREN );
    }

    while ( !found && dptr > getDirPtr( DATDI ) ) {
        /*
         *  find next entry
         */
        dptr--;

        if ( type != 0 && dptr->type != (byte) type ) {
            /*
             *  Type not matched
             */
            continue;
        }

        /*
         *  Compare
         */
        for ( i = 0; i < 8 + 3; ++i ) {
            if ( expPattern[ i ] == '?' ) {
                continue;
            }
            c = dptr->name[ i ];
            if ( expPattern[ i ] != ( ignoreCase ? toupper( c ) : c ) ) {
#if DEBUG_
                printf( "%c != %c\n", expPattern[ i ], c );
#endif
                break;
            }
        }
        found = ( i == 8 + 3 );
    }

    /*
     *  Return entry or NULL if no match
     */
    return found ? dptr : NULL;
}


/*
 *  Check for "?" and "*"
 */
int isWildcard( char *pattern )
{
    return strchr( pattern, '*' ) != NULL || strchr( pattern, '?' ) != NULL;
}


/*
 *  Compute free ramdisk space
 */
word diskFree( void )
{
    return getWord( DATDI ) - getWord( MEMEN );
}


/*
 *  Load a binary file into memory
 */
int loadFile ( char *name, int size, byte *buf )
{
    FILE *Handle;

    Handle = fopen( name, "rb" );
    if ( Handle == NULL ) {
        perror( name );
        return NOT_OK;
    }
    if ( fread( buf, sizeof( byte ), (size_t) size, Handle ) !=
    (size_t) size ) {
        perror( name );
        (void) fclose( Handle );
        return NOT_OK;
    }
    if ( 0 != fclose( Handle ) ) {
        perror( name );
        return NOT_OK;
    }
    return OK;
}


/*
 *  Save a binary file
 */
int saveFile ( char *name, int size, byte *buf )
{
    FILE *Handle;

    Handle = fopen( name, "wb" );
    if ( Handle == NULL ) {
        perror( name );
        return NOT_OK;
    }
    if ( fwrite( buf, sizeof( byte ), (size_t) size, Handle ) !=
    (size_t) size ) {
        perror( name );
        (void) fclose( Handle );
        return NOT_OK;
    }
    if ( 0 != fclose( Handle ) ) {
        perror( name );
        return NOT_OK;
    }
    return OK;
}


byte getByte( word address )
{
    return *(image - IMGSTART + address);
}


word getWord( word address )
{
    return *(word *) (image - IMGSTART + address);
}


void putWord( word address, word data )
{
    *(word *) (image - IMGSTART + address) = data;
}


void addWord( word address, word data )
{
    *(word *) (image - IMGSTART + address) += data;
}


/*
 *  Directory entry pointer for an address stored in one of the system
 *  variables DATDI, BASDI, DIREN
 */
DIR_ENTRY *getDirPtr( word sysvar )
{
    return (DIR_ENTRY *) (image - IMGSTART + getWord( sysvar ) );
}


/*
 *  Allocate memory in the data space
 */
int makeRoom( int type, word size )
{
    DIR_ENTRY *dptr;

    if ( size > diskFree() ) {
        return NOT_OK;
    }
    if ( type == TYPE_B ) {
        memmove ( image - IMGSTART + getWord( BASEN ) + size,
                  image - IMGSTART + getWord( BASEN ),
                  (size_t) ( getWord( MEMEN ) - getWord( BASEN ) ) );
        addWord( BASEN, size );
        /*
         *  Adjust the data position in the data directory
         */
        for ( dptr = getDirPtr( DATDI );
              dptr < getDirPtr( BASDI );
              dptr++ ) {
            dptr->srcbegin += size;
            dptr->srcend += size;
        }
    }
    addWord( MEMEN, size );
    return OK;
}


/*
 *  Update the ONFILx system variables
 */
void updateOnFile( void )
{
    word i, j;
    byte f;

    /*
     *  Clear the ONFILx system variables
     */
    for ( j = 0; j < 5; j++ ) {
        putWord( ONFIL0 + 2 * j++, 0 );
    }

    /*
     *  Copy marked directory entries to corresponding ONFILx system variables
     */
    j = 0;
    for ( i = getWord( BASDI );
          i < getWord( DIREN );
          i += sizeof( DIR_ENTRY ) ) {
        f = ( (DIR_ENTRY *) (image - IMGSTART + i ) ) -> flag;
        if ( f == FLAG_CBOOT ) {
            putWord( ONFIL0, i );
        }
        else if ( f == FLAG_PRESET && j < 4 ) {
            putWord( ONFIL1 + 2 * j++, i );
        }
    }
}


/*
 *  Test the memory integrity
 */
int checkRamDisk( void )
{
    DIR_ENTRY *dptr;

    if ( getWord( HIMEM ) < (word) 0x7000 ) return NOT_OK;
    if ( getWord( HIMEM ) > getWord( BASEN ) ) return NOT_OK;
    if ( getWord( BASEN ) > getWord( MEMEN ) ) return NOT_OK;
    if ( getWord( MEMEN ) > getWord( DATDI ) ) return NOT_OK;
    if ( getWord( DATDI ) > getWord( BASDI ) ) return NOT_OK;
    if ( getWord( BASDI ) > getWord( DIREN ) ) return NOT_OK;

    if ( ( getWord( DIREN ) - getWord( BASDI ) ) % sizeof( DIR_ENTRY ) != 0 )
        return NOT_OK;
    if ( ( getWord( BASDI ) - getWord( DATDI ) ) % sizeof( DIR_ENTRY ) != 0 )
        return NOT_OK;

    if ( getWord( DATDI ) == getWord( DIREN ) ) return OK;
    if ( getDirPtr( DATDI )->srcend != getWord( MEMEN ) ) return NOT_OK;
    if ( ( getDirPtr( DIREN ) - 1 )->srcbegin != getWord( HIMEM ) )
        return NOT_OK;
    for ( dptr = getDirPtr( DATDI );
          dptr < getDirPtr( DIREN ) - 1;
          dptr++ ) {
        if ( dptr->srcbegin != ( dptr + 1 )->srcend ) return NOT_OK;
    }

    if ( getWord( BASDI ) == getWord( DIREN ) ) return OK;
    if ( getDirPtr( BASDI )->srcend != getWord( BASEN ) ) return NOT_OK;

    return OK;
}
