; Forth dictionary

; absolute addresses
PUSH1:		EQU	&H7000
NEXT1:		EQU	&H7002
NEXT2:		EQU	&H7004
USER1:		EQU	&H700F
REAL_ADDR:	EQU	&H7016
SYSCALL:	EQU	&H7021
SYSJUMP:	EQU	&H7029

; relocatable code

		ORG	&H0000

ORIGIN_ADDR:
		DW	0
		DW	0
		DW	0
		DW	0
		DW	0
		DW	0
LAST_FORTH_DEF:	DW	LAST_DEFIN	;adr. ost. definicji w podsl. FORTH
BACKSPACE_KEY:	DW	&H08		;kod klawisza BACKSPACE
		DW	0
; tablica poczatkowych wartosci parametrow - kopiowane przez COLD do User Area
S0_INIT:	DW	&H6A4B
R0_INIT:	DW	&H6B4A
TIB_INIT:	DW	&H6A4B
WIDTH_INIT:	DW	31
WARNING_INIT:	DW	0
FENCE_INIT:	DW	HERE_ADDR
DP_INIT:	DW	HERE_ADDR
VOC_LINK_INIT:	DW	FORTH_LINK
		DW	0
		DW	0
USER_AREA_PTR:	DW	&H6952		;offset &H26
COLD_PFA_PTR:	DW	COLD_PFA	;offset &H28
QUIT_PFA_PTR:	DW	QUIT_PFA	;offset &H2A

; struktura User Area
;S0		offset &H00 - poczatkowy adres szczytu stosu parametrow
;R0		offset &H02 - poczatkowy adres szczytu stosu powrotow
;TIB		offset &H04 - terminal input buffer
;WIDTH		offset &H06
;WARNING	offset &H08 - okresla reakcje na bledy
;FENCE		offset &H0A
;DP		offset &H0C - adres pierwszego wolnego slowa slownika
;VOC-LINK	offset &H0E
;BLK		offset &H10 - directory entry for the interpreted file
;IN		offset &H12
;OUT		offset &H14
;STATE		offset &H1A - gdy 0, interpretator w stanie wykonywania
;BASE		offset &H1C
;DPL		offset &H1E
;FLD		offset &H20
;CSP		offset &H22
;R#		offset &H24
;HLD		offset &H26
;USE		offset &H28 - directory entry for the dictionary file

LIT_NFA:	DB	&H83,"LI","T"+&H80	;NFA
		DW	0			;LFA
LIT:		DW	LIT_MC			;CFA
LIT_MC:		LDIW	$2,(IX+$31)
		JP	PUSH1

EXECUTE_NFA:	DB	&H87,"EXECUT","E"+&H80
		DW	LIT_NFA
EXECUTE:	DW	EXECUTE_MC
EXECUTE_MC:	PPUW	$2
		JP	NEXT2

BRANCH_NFA:	DB	&H86,"BRANC","H"+&H80
		DW	EXECUTE_NFA
BRANCH:		DW	BRANCH_MC

ZEROBRANCH_NFA:	DB	&H87,"0BRANC","H"+&H80
		DW	BRANCH_NFA
ZERO_BRANCH:	DW	ZERO_BRANCH_MC
ZERO_BRANCH_MC:	PPUW	$0
		ORC	$1,$0
		JR	NZ,NOBRANCH
; add the displacement to the IP
BRANCH_MC:	GRE	IX,$2
		LDW	$0,(IX+$31)
		ADW	$2,$0
		PRE	IX,$2,JR NEXT1A

; (LOOP) oczekuje w slowniku (po swoim wskazaniu) dlugosci skoku do DO 
PAR_LOOP_NFA:	DB	&H86,"(LOOP",")"+&H80
		DW	ZEROBRANCH_NFA
PAR_LOOP:	DW	PAR_LOOP_MC
PAR_LOOP_MC:	PPSM	$2,4
		ADW	$2,$30		;zwiekszenie zmiennej sterujacej o 1
; test warunku zakonczenia petli
PAR_LOOP_MC1:	LDW	$0,$2
		SBW	$0,$4		;porownanie zm. ster. i ograniczenia
		BIU	$1		;znak -> Carry
		JR	C,PAR_LOOP_MC2	;kontynuacja petli gdy wynik ujemny
; zakonczenie petli lub nie wykonanie skoku warunkowego
NOBRANCH:	LDI	$0,(IX+$30)	;IX<-IX+2, skip the displacement
NEXT1A:		JP	NEXT1
; kontynuacja petli
PAR_LOOP_MC2:	PHSM	$5,4
		JR	BRANCH_MC

PARPLLOOP_NFA:	DB	&H87,"(+LOOP",")"+&H80
		DW	PAR_LOOP_NFA
PAR_PLUS_LOOP:	DW	PARPLUSLOOP_MC
PARPLUSLOOP_MC:	PPSM	$2,4
		PPUW	$0
		ADW	$2,$0,JR PAR_LOOP_MC1	;zwiekszenie zmiennej ster.

; ( Limit Init	- - )
PAR_DO_NFA:	DB	&H84,"(DO",")"+&H80
		DW	PARPLLOOP_NFA
PAR_DO:		DW	PAR_DO_MC
; umieszczenie na stosie powr. liczb Limit i Init zdjetych ze stosu parametrow
PAR_DO_MC:	PPUM	$0,4
		PHSM	$3,4
		JP	NEXT1

; skopiowanie ze stosu powrotow zmiennej sterujacej
I_NFA:		DB	&H81,"I"+&H80
		DW	PAR_DO_NFA
I:		DW	I_MC
I_MC:		PPSW	$2
		PHSW	$3
PUSH1A:		JP	PUSH1

; skopiowanie ze stosu powrotow zmiennej sterujacej zewnetrznej petli
J_NFA:		DB	&H81,"J"+&H80
		DW	I_NFA
J:		DW	J_MC
J_MC:		PPSM	$0,6
		PHSM	$5,6
		PHUW	$5
		JP	NEXT1

DIGIT_NFA:	DB	&H85,"DIGI","T"+&H80
		DW	J_NFA
DIGIT:		DW	DIGIT_MC
DIGIT_MC:	PPUM	$0,4		;$0,$1 = podstawa liczbowa
					;$2 = znak ASCII
		LD	$3,$31
		SBC	$2,"a"
		JR	C,DIGIT_MC1
		SB	$2,&H20
DIGIT_MC1:	SBC	$2,"9"+1
		JR	C,DIGIT_MC2
		SBC	$2,"A"
		JR	C,DIGIT_FALSE
DIGIT_MC2:	SB	$2,"0"
		JR	C,DIGIT_FALSE
		SBC	$2,10
		JR	C,DIGIT_MC3
		SB	$2,7
DIGIT_MC3:	SBCW	$2,$0		;porownanie cyfry z podstawa liczbowa
		JR	NC,DIGIT_FALSE
		PHUW	$3		;wartosc cyfry
		LD	$2,$30,JR PUSH1A	;oraz True
DIGIT_FALSE:	LD	$2,$31,JR PUSH1A	;zwraca False

PAR_FIND_NFA:	DB	&H86,"(FIND",")"+&H80
		DW	DIGIT_NFA
PAR_FIND:	DW	PAR_FIND_MC
PAR_FIND_MC:	GRE	IX,$6			;przechowanie IP
		PPUM	$2,4	;$2,$3 = adres pola nazwy w slowniku
				;$4,$5 = adres pola nazwy szukanego slowa
		ADW	$4,$25
		PRE	IX,$4
		PRE	IZ,$4
; w szukanym slowie zamieniamy male litery na duze
		LDW	$23,&H015B		;system function TCAPS
		LDI	$1,(IZ+$31)		;ilosc znakow
PAR_FIND_LOOP0:	LDD	$0,(IZ+$31)
		CAL	SYSCALL
		STI	$0,(IZ+$31)
		SB	$1,$30
		JR	NZ,PAR_FIND_LOOP0
		LDW	$0,$2
;
PAR_FIND_LOOP1:	LDW	$4,$0			;save the virtual NFA
		ADW	$0,$25
		PRE	IZ,$0
		LDD	$0,(IZ+$31)		;bajt sterujacy NFA
		LDD	$1,(IX+$31)		;ilosc znakow szukanej nazwy
		XR	$0,$1			;porownanie ilosci znakow nazw
		AN	$0,&H3F			;dlugosc + bit SMUDGE
		JR	NZ,PAR_FIND_NEXT	;skok dla nazw roznej dlugosci
						;lub ustawionego bitu SMUDGE
; $0=0, $1=dlugosc nazwy
PAR_FIND_LOOP2:	AD	$0,$30
		LD	$3,(IX+$0)
		LD	$2,(IZ+$0)
		XR	$3,$2
		BIU	$3			;przy porownaniu ignorujemy bit 7
		JR	NZ,PAR_FIND_NEXT	;skok dla roznych znakow w nazwach
		SB	$1,$30
		JR	NZ,PAR_FIND_LOOP2
; nazwa zostala znaleziona, $0=dlugosc nazwy, $1=0
		LDW	$2,5		;dlugosc pol LFA, CFA + bajt ster. NFA
		ADW	$2,$4
		ADW	$2,$0		;$2,$3 wskazuje PFA odszukanej definicji
		LDD	$0,(IZ+$31)	;bajt sterujacy NFA
		PHUM	$3,6
		JR	PAR_FIND_END
; szukanie nastepnej nazwy
PAR_FIND_NEXT:	LDI	$0,(IZ+$31)	;bajt sterujacy NFA
		AN	$0,&H1F		;dlugosc nazwy w slowniku
		LDW	$0,(IZ+$0)	;pole LFA
		ORC	$1,$0
		JR	NZ,PAR_FIND_LOOP1
; pole LFA zawiera 0 - osiagnieta pierwsza definicja slownika
		PHUW	$1
PAR_FIND_END:	PRE	IX,$6,JR NEXT1B	;odtworzenie IP

; modyfikacje zwiazane pobieraniem znakow z pliku tekstowego
; 1. znak &H1A traktowany jest tak samo jak zero
; 2. przed porownaniem z ogranicznikiem znaki o innych kodach < &H20
;    zamieniane sa na spacje
ENCLOSE_NFA:	DB	&H87,"ENCLOS","E"+&H80
		DW	PAR_FIND_NFA
ENCLOSE:	DW	ENCLOSE_MC
ENCLOSE_MC:	PPUM	$0,4		;$2,$3=address, $0=terminating char.
		PHUW	$3		;address
		CAL	REAL_ADDR
; szukamy pierwszego znaku nie bedacego ogranicznikiem
ENCLOSE_MC_1:	LDI	$1,(IZ+$31)
		SBC	$1,$31
		JR	Z,ENCLOSE_MC_6
		SBC	$1,&H1A
		JR	Z,ENCLOSE_MC_6
		SBC	$1,&H20
		JR	NC,ENCLOSE_MC_2
		LD	$1,&H20
ENCLOSE_MC_2:	SBC	$1,$0
		JR	Z,ENCLOSE_MC_1
; znaleziono pierwszy znak nie bedacy ogranicznikiem
		GRE	IZ,$4
		SBW	$4,$2
		SBW	$4,$30
		PHUW	$5		;n1
; szukamy ogranicznika
ENCLOSE_MC_3:	LDI	$1,(IZ+$31)
		SBC	$1,$31
		JR	Z,ENCLOSE_MC_5
		SBC	$1,&H1A
		JR	Z,ENCLOSE_MC_5
		SBC	$1,&H20
		JR	NC,ENCLOSE_MC_4
		LD	$1,&H20
ENCLOSE_MC_4:	SBC	$1,$0
		JR	NZ,ENCLOSE_MC_3
; znaleziono ogranicznik
		GRE	IZ,$4
		SBW	$4,$2
		LDW	$2,$4			;n3
		SBW	$4,$30,JR ENCLOSE_MC_7	;n2
; szukajac ogranicznika znaleziono zero lub &H1A
ENCLOSE_MC_5:	GRE	IZ,$4
		SBW	$4,$2
		SBW	$4,$30			;n2
		LDW	$2,$4,JR ENCLOSE_MC_7	;n3
; szukajac znaku nie bedacego ogranicznikiem znaleziono zero lub &H1A
ENCLOSE_MC_6:	GRE	IZ,$4
		SBW	$4,$2
		SBW	$4,$30
		PHUW	$5		;n1
		LDW	$2,$4		;n3
		ADW	$4,$30		;n2
ENCLOSE_MC_7:	PHUM	$5,4
NEXT1B:		JP	NEXT1

EMIT_NFA:	DB	&H84,"EMI","T"+&H80
		DW	ENCLOSE_NFA
EMIT:		DW	EMIT_MC
EMIT_MC:	PPUW	$16
		LDW	$23,&HFF9E		;system function OUTAC
		CAL	SYSCALL
		CAL	USER1
		LD	$0,&H14			;user variable OUT
		ADW	(IZ+$0),$30
		JP	NEXT1

KEY_NFA:	DB	&H83,"KE","Y"+&H80
		DW	EMIT_NFA
KEY:		DW	KEY_MC
KEY_MC:		LDW	$23,&HE8B9		;system function KYIN
		CAL	SYSCALL
		PHU	$31
		PHU	$0
		JP	NEXT1

QUEST_TERM_NFA:	DB	&H89,"?TERMINA","L"+&H80
		DW	KEY_NFA
QUEST_TERMINAL:	DW	QUEST_TERM_MC
QUEST_TERM_MC:	LDW	$23,&H9E3B	;system function INKEY
		CAL	SYSCALL
		PHU	$31
		PHU	$17
		JP	NEXT1

CR_NFA:		DB	&H82,"C","R"+&H80
		DW	QUEST_TERM_NFA
CR:		DW	CR_MC
CR_MC:		LDW	$23,&H95CE	;system function OUTCR
		CAL	SYSCALL
		JP	NEXT1

CMOVE_NFA:	DB	&H85,"CMOV","E"+&H80
		DW	CR_NFA
CMOVE:		DW	CMOVE_MC
CMOVE_MC:	PPUW	$4		;$4,$5 = number of bytes
		PPUW	$2
		CAL	REAL_ADDR
		GRE	IZ,$0		;$0,$1 = destination address
		PPUW	$2
		CAL	REAL_ADDR	;$2,$3 = source address
		LDW	$23,&H9048	;block transfer
		CAL	SYSCALL
		JP	NEXT1

U_MUL_NFA:	DB	&H82,"U","*"+&H80
		DW	CMOVE_NFA
U_MUL:		DW	U_MUL_MC
U_MUL_MC:	PPUM	$0,4
		XRM	$4,$4,6
		SBCW	$0,$2
		JR	NC,U_MUL_MC1
; swap $0,$1 <-> $2,$3 to choose the smaller value for the multiplier
		PHUW	$3
		LDW	$2,$0
		PPUW	$0
; $2,$3 = multiplier, $0,$1,$4,$5 = multiplicand, $8,$9,$6,$7 = product
U_MUL_MC1:	ANC	$2,$30		;if lowest bit of multiplier is 1...
		JR      Z,U_MUL_MC3
		ADW	$8,$0		;then add multiplicand to the product
		JR	NC,U_MUL_MC2
		ADW	$6,$30
U_MUL_MC2:	ADW	$6,$4
U_MUL_MC3:	BIDW	$3		;shift multiplier down 1 bit
		JR	Z,U_MUL_MC4	;end of multiplication
		BIUW	$0		;shift multiplicand up 1 bit
		ROUW	$4,JR U_MUL_MC1
U_MUL_MC4:	PHUM	$9,4
		JP	NEXT1

U_DIV_NFA:	DB	&H82,"U","/"+&H80
		DW	U_MUL_NFA
U_DIV:		DW	U_DIV_MC
U_DIV_MC:	PPUM	$0,6
; $0,$1 = divisor, $2,$3 = dividend upper word, $4,$5 = dividend lower word
		LD	$6,16		;counter
U_DIV_MC1:	BIUW	$4
		ROUW	$2
		JR	C,U_DIV_MC2
		SBCW	$2,$0
		JR	C,U_DIV_MC3
U_DIV_MC2:	SBW	$2,$0
		OR	$4,$30
U_DIV_MC3:	SB	$6,$30
		JR	NZ,U_DIV_MC1
; $2,$3 = remainder, $4,$5 = quotient
		PHUW	$3
		PHUW	$5
		JP	NEXT1

AND_NFA:	DB	&H83,"AN","D"+&H80
		DW	U_DIV_NFA
AND_:		DW	AND_MC
AND_MC:		PPUM	$0,4
		ANW	$2,$0
PUSH1B:		JP	PUSH1

OR_NFA:		DB	&H82,"O","R"+&H80
		DW	AND_NFA
OR_:		DW	OR_MC
OR_MC:		PPUM	$0,4
		ORW	$2,$0,JR PUSH1B

XOR_NFA:	DB	&H83,"XO","R"+&H80
		DW	OR_NFA
XOR_:		DW	XOR_MC
XOR_MC:		PPUM	$0,4
		XRW	$2,$0,JR PUSH1B

SP_FETCH_NFA:	DB	&H83,"SP","@"+&H80
		DW	XOR_NFA
SP_FETCH:	DW	SP_FETCH_MC
SP_FETCH_MC:	GRE	US,$2,JR PUSH1B

SP_STORE_NFA:	DB	&H83,"SP","!"+&H80
		DW	SP_FETCH_NFA
SP_STORE:	DW	SP_STORE_MC
SP_STORE_MC:	CAL	USER1
		LDW	$0,(IZ+$31)		;user variable S0
		PRE	US,$0,JR NEXT1C

RP_FETCH_NFA:	DB	&H83,"RP","@"+&H80
		DW	SP_STORE_NFA
RP_FETCH:	DW	RP_FETCH_MC
RP_FETCH_MC:	GRE	SS,$2,JR PUSH1C

RP_STORE_NFA:	DB	&H83,"RP","!"+&H80
		DW	RP_FETCH_NFA
RP_STORE:	DW	RP_STORE_MC
RP_STORE_MC:	CAL	USER1
		LD	$0,&H02
		LDW	$0,(IZ+$0)	;user variable R0
		PRE	SS,$0,JR NEXT1C

SEMICOLONS_NFA:	DB	&H82,";","S"+&H80
		DW	RP_STORE_NFA
SEMICOLON_S:	DW	SEMICOLONS_MC
SEMICOLONS_MC:	PPSW	$0		;odtworzenie virtual IP ze stosu powrotow
		ADW	$0,$25		;virtual->real
		PRE	IX,$0,JR NEXT1C

; przepisanie na stosie powrotow zmiennej sterujacej do Limit
LEAVE_NFA:	DB	&H85,"LEAV","E"+&H80
		DW	SEMICOLONS_NFA
LEAVE:		DW	LEAVE_MC
LEAVE_MC:	PPSM	$0,4
		LDW	$2,$0
		PHSM	$3,4
NEXT1C:		JP	NEXT1

TO_R_NFA:	DB	&H82,">","R"+&H80
		DW	LEAVE_NFA
TO_R:		DW	TO_R_MC
TO_R_MC:	PPUW	$0
		PHSW	$1
		JP	NEXT1

R_FROM_NFA:	DB	&H82,"R",">"+&H80
		DW	TO_R_NFA
R_FROM:		DW	R_FROM_MC
R_FROM_MC:	PPSW	$2
PUSH1C:		JP	PUSH1

R_NFA:		DB	&H81,"R"+&H80
		DW	R_FROM_NFA
R:		DW	I_MC

ZERO_EQ_NFA:	DB	&H82,"0","="+&H80
		DW	R_NFA
ZERO_EQUAL:	DW	ZERO_EQ_MC
ZERO_EQ_MC:	LDW	$2,$30
		PPUW	$0
		ORC	$1,$0
		JR	Z,PUSH1C
		LD	$2,$31,JR PUSH1C

ZERO_LT_NFA:	DB	&H82,"0","<"+&H80
		DW	ZERO_EQ_NFA
ZERO_LESS_THAN:	DW	ZERO_LT_MC
ZERO_LT_MC:	PPUW	$0
		XRW	$2,$2
		BIUW	$1,JR PUSH1C

ADD_NFA:	DB	&H81,"+"+&H80
		DW	ZERO_LT_NFA
ADD_:		DW	ADD_MC
ADD_MC:		PPUM	$0,4
		ADW	$2,$0,JR PUSH1C

D_ADD_NFA:	DB	&H82,"D","+"+&H80
		DW	ADD_NFA
D_ADD:		DW	D_ADD_MC
D_ADD_MC:	PPUM	$0,8
		ADW	$6,$2
		JR	NC,D_ADD_MC1
		ADW	$4,$30
D_ADD_MC1:	ADW	$4,$0
		PHUM	$7,4
		JP	NEXT1

MINUS_NFA:	DB	&H85,"MINU","S"+&H80
		DW	D_ADD_NFA
MINUS:		DW	MINUS_MC
MINUS_MC:	PPUW	$2
		CMPW	$2,JR PUSH1C

D_MINUS_NFA:	DB	&H86,"DMINU","S"+&H80
		DW	MINUS_NFA
D_MINUS:	DW	D_MINUS_MC
D_MINUS_MC:	PPUW	$2
		PPUW	$0
		CMPM	$0,4
		PHUW	$1
		JP	PUSH1

OVER_NFA:	DB	&H84,"OVE","R"+&H80
		DW	D_MINUS_NFA
OVER:		DW	OVER_MC
OVER_MC:	PPUM	$0,4
		PHUM	$3,4
		JP	PUSH1

DROP_NFA:	DB	&H84,"DRO","P"+&H80
		DW	OVER_NFA
DROP:		DW	DROP_MC
DROP_MC:	PPUW	$0
		JP	NEXT1

SWAP_NFA:	DB	&H84,"SWA","P"+&H80
		DW	DROP_NFA
SWAP:		DW	SWAP_MC
SWAP_MC:	PPUM	$0,4
		PHUW	$1
		JP	PUSH1

DUP_NFA:	DB	&H83,"DU","P"+&H80
		DW	SWAP_NFA
DUP_:		DW	DUP_MC
DUP_MC:		PPUW	$2
		PHUW	$3
		JP	PUSH1

DOUBLE_DUP_NFA:	DB	&H84,"2DU","P"+&H80
		DW	DUP_NFA
DOUBLE_DUP:	DW	DOUBLE_DUP_MC
DOUBLE_DUP_MC:	PPUM	$0,4
		PHUM	$3,4
		PHUM	$3,4
		JP	NEXT1

PLUS_STORE_NFA:	DB	&H82,"+","!"+&H80
		DW	DOUBLE_DUP_NFA
PLUS_STORE:	DW	PLUS_STORE_MC
PLUS_STORE_MC:	PPUM	$2,4
		CAL	REAL_ADDR
		ADW	(IZ+$31),$4
		JP	NEXT1

TOGGLE_NFA:	DB	&H86,"TOGGL","E"+&H80
		DW	PLUS_STORE_NFA
TOGGLE:		DW	TOGGLE_MC
TOGGLE_MC:	PPUM	$0,4
		CAL	REAL_ADDR
		LDW	$4,(IZ+$31)
		XRW	$4,$0,JR STORE_MC_1

AT_NFA:		DB	&H81,"@"+&H80
		DW	TOGGLE_NFA
AT:		DW	AT_MC
AT_MC:		PPUW	$2
		CAL	REAL_ADDR
		LDW	$2,(IZ+$31)
PUSH1D:		JP	PUSH1

C_AT_NFA:	DB	&H82,"C","@"+&H80
		DW	AT_NFA
C_AT:		DW	C_AT_MC
C_AT_MC:	PPUW	$2
		CAL	REAL_ADDR
		LDD	$2,(IZ+$31)
		LD	$3,$31,JR PUSH1D

DOUBLE_AT_NFA:	DB	&H82,"2","@"+&H80
		DW	C_AT_NFA
DOUBLE_AT:	DW	DOUBLE_AT_MC
DOUBLE_AT_MC:	PPUW	$2
		CAL	REAL_ADDR
		LDM	$0,(IZ+$31),4
		PHUM	$3,4
		JP	NEXT1

STORE_NFA:	DB	&H81,"!"+&H80
		DW	DOUBLE_AT_NFA
STORE:		DW	STORE_MC
STORE_MC:	PPUM	$2,4
		CAL	REAL_ADDR
STORE_MC_1:	STW	$4,(IZ+$31)
		JP	NEXT1

C_STORE_NFA:	DB	&H82,"C","!"+&H80
		DW	STORE_NFA
C_STORE:	DW	C_STORE_MC
C_STORE_MC:	PPUM	$2,4
		CAL	REAL_ADDR
		STD	$4,(IZ+$31)
		JP	NEXT1

DOUBL_STOR_NFA:	DB	&H82,"2","!"+&H80
		DW	C_STORE_NFA
DOUBL_STORE:	DW	DOUBL_STOR_MC
DOUBL_STOR_MC:	PPUM	$2,6
		CAL	REAL_ADDR
		STM	$4,(IZ+$31),4
		JP	NEXT1

NOOP_NFA:	DB	&H84,"NOO","P"+&H80
		DW	DOUBL_STOR_NFA
NOOP:		DW	NOOP_MC
; NOOP deliberately doesn't share machine code with other commands in order
; to be used as a breakpoint while debugging
NOOP_MC:	JP	NEXT1

ZERO_NFA:	DB	&H81,"0"+&H80
		DW	NOOP_NFA
ZERO:		DW	CONSTANT_MC,0

ONE_NFA:	DB	&H81,"1"+&H80
		DW	ZERO_NFA
ONE:		DW	CONSTANT_MC,1

TWO_NFA:	DB	&H81,"2"+&H80
		DW	ONE_NFA
TWO:		DW	CONSTANT_MC,2

THREE_NFA:	DB	&H81,"3"+&H80
		DW	TWO_NFA
THREE:		DW	CONSTANT_MC,3

BL_NFA:		DB	&H82,"B","L"+&H80
		DW	THREE_NFA
BL_:		DW	CONSTANT_MC," "

C_SLASH_L_NFA:	DB	&H83,"C/","L"+&H80
		DW	BL_NFA
C_SLASH_L:	DW	CONSTANT_MC,32

; slowo wlasciwie nic nie robi, poniewaz ORIGIN_ADDR=0
PLUSORIGIN_NFA:	DB	&H87,"+ORIGI","N"+&H80
		DW	C_SLASH_L_NFA
PLUS_ORIGIN:	DW	PLUSORIGIN_MC
PLUSORIGIN_MC:	PPUW	$2
		LDW	$0,ORIGIN_ADDR
		ADW	$2,$0
		JP	PUSH1

S0_NFA:		DB	&H82,"S","0"+&H80
		DW	PLUSORIGIN_NFA
S_ZERO:		DW	USER_MC,&H00

R0_NFA:		DB	&H82,"R","0"+&H80
		DW	S0_NFA
R_ZERO:		DW	USER_MC,&H02

TIB_NFA:	DB	&H83,"TI","B"+&H80
		DW	R0_NFA
TIB:		DW	USER_MC,&H04

WIDTH_NFA:	DB	&H85,"WIDT","H"+&H80
		DW	TIB_NFA
WIDTH_:		DW	USER_MC,&H06

WARNING_NFA:	DB	&H87,"WARNIN","G"+&H80
		DW	WIDTH_NFA
WARNING:	DW	USER_MC,&H08

FENCE_NFA:	DB	&H85,"FENC","E"+&H80
		DW	WARNING_NFA
FENCE:		DW	USER_MC,&H0A

DP_NFA:		DB	&H82,"D","P"+&H80
		DW	FENCE_NFA
DP:		DW	USER_MC,&H0C

VOC_LINK_NFA:	DB	&H88,"VOC-LIN","K"+&H80
		DW	DP_NFA
VOC_LINK:	DW	USER_MC,&H0E

BLK_NFA:	DB	&H83,"BL","K"+&H80
		DW	VOC_LINK_NFA
BLK:		DW	USER_MC,&H10

IN_NFA:		DB	&H82,"I","N"+&H80
		DW	BLK_NFA
IN_:		DW	USER_MC,&H12

OUT_NFA:	DB	&H83,"OU","T"+&H80
		DW	IN_NFA
OUT_:		DW	USER_MC,&H14

STATE_NFA:	DB	&H85,"STAT","E"+&H80
		DW	OUT_NFA
STATE:		DW	USER_MC,&H1A

BASE_NFA:	DB	&H84,"BAS","E"+&H80
		DW	STATE_NFA
BASE:		DW	USER_MC,&H1C

DPL_NFA:	DB	&H83,"DP","L"+&H80
		DW	BASE_NFA
DPL:		DW	USER_MC,&H1E

FLD_NFA:	DB	&H83,"FL","D"+&H80
		DW	DPL_NFA
FLD_:		DW	USER_MC,&H20

CSP_NFA:	DB	&H83,"CS","P"+&H80
		DW	FLD_NFA
CSP:		DW	USER_MC,&H22

R_HASH_NFA:	DB	&H82,"R","#"+&H80
		DW	CSP_NFA
R_HASH:		DW	USER_MC,&H24

HLD_NFA:	DB	&H83,"HL","D"+&H80
		DW	R_HASH_NFA
HLD:		DW	USER_MC,&H26

USE_NFA:	DB	&H83,"US","E"+&H80
		DW	HLD_NFA
USE:		DW	USER_MC,&H28

CONTEXT_NFA:	DB	&H87,"CONTEX","T"+&H80
		DW	USE_NFA
CONTEXT:	DW	VARIABLE_MC,FORTH_CONTEXT

CURRENT_NFA:	DB	&H87,"CURREN","T"+&H80
		DW	CONTEXT_NFA
CURRENT:	DW	VARIABLE_MC,FORTH_CURRENT

ADD_1_NFA:	DB	&H82,"1","+"+&H80
		DW	CURRENT_NFA
ADD_1:		DW	ADD_1_MC
ADD_1_MC:	PPUW	$2
ADD_1_2:	ADW	$2,$30
PUSH1E:		JP	PUSH1

ADD_2_NFA:	DB	&H82,"2","+"+&H80
		DW	ADD_1_NFA
ADD_2:		DW	ADD_2_MC
ADD_2_MC:	PPUW	$2
		ADW	$2,$30,JR ADD_1_2

SUB_NFA:	DB	&H81,"-"+&H80
		DW	ADD_2_NFA
SUB_:		DW	SUB_MC
SUB_MC:		PPUM	$0,4
		SBW	$2,$0,JR PUSH1E

HERE_NFA:	DB	&H84,"HER","E"+&H80
		DW	SUB_NFA
HERE:		DW	COLON_MC
		DW	DP,AT,SEMICOLON_S

PAD_NFA:	DB	&H83,"PA","D"+&H80
		DW	HERE_NFA
PAD:		DW	COLON_MC
		DW	HERE,LIT,128,ADD_,SEMICOLON_S

RESIZE_NFA:	DB	&H86,"RESIZ","E"+&H80
		DW	PAD_NFA
RESIZE:		DW	RESIZE_MC
RESIZE_MC:	CAL	USER1
		LD	$0,&H0C		;offset of the DP
		LDW	$4,(IZ+$0)	;virtual HERE
		ADW	$4,$25		;real HERE
		LDW	$0,384		;optimal free space
		LDW	$2,$27
		SBW	$2,$4		;$2,$3 = free space in the dictionary file
		JR	C,RESIZE_EXPAND
		LDW	$6,256		;min. free space
		SBCW	$2,$6
		JR	C,RESIZE_EXPAND
		LDW	$6,512		;max. free space
		SBCW	$2,$6
		JR	C,NEXT1D	;no need to resize
; shrink the dictionary file
		LDW	$23,&HB26C	;shrink memory block for a seq. file
		SBW	$2,$0
		LDW	$0,$4,JR RESIZE_MC_1
; expand the dictionary file
RESIZE_EXPAND:	LDW	$23,&HB292	;expand memory block for a seq. file
		SBW	$0,$2
		LDW	$2,$27
		SBW	$2,$30		;skip the EOF byte
RESIZE_MC_1:	CAL	SYSCALL
NEXT1D:		JP	NEXT1

FREE_NFA:	DB	&H84,"FRE","E"+&H80
		DW	RESIZE_NFA
FREE:		DW	COLON_MC
		DW	LIT,&H6945,DOUBLE_AT		;MEMEN, DATDI
		DW	SUB_,SEMICOLON_S

; dodane RESIZE
ALLOT_NFA:	DB	&H85,"ALLO","T"+&H80
		DW	FREE_NFA
ALLOT:		DW	COLON_MC
; czy jest dostateczna ilosc wolnej pamieci?
; optimal_free_space - minimal_free_space = 128
		DW	DUP_,LIT,128,ADD_,FREE,U_LESS_THAN
		DW	ZERO_BRANCH,ALLOT_3-ALLOT_1+2
ALLOT_1:	DW	HERE,ADD_
; czy nie zostanie przekroczony maksymalny wirtualny adres?
		DW	DUP_,LIT,&H5E00,U_LESS_THAN
		DW	ZERO_BRANCH,ALLOT_3-ALLOT_2+2
ALLOT_2:	DW	DP,STORE,RESIZE,SEMICOLON_S
ALLOT_3:	DW	TWO,ERROR		;dictionary full

COMMA_NFA:	DB	&H81,","+&H80
		DW	ALLOT_NFA
COMMA:		DW	COLON_MC
		DW	HERE,STORE,TWO,ALLOT,SEMICOLON_S

C_COMMA_NFA:	DB	&H82,"C",","+&H80
		DW	COMMA_NFA
C_COMMA:	DW	COLON_MC
		DW	HERE,C_STORE,ONE,ALLOT,SEMICOLON_S

EQUALS_NFA:	DB	&H81,"="+&H80
		DW	C_COMMA_NFA
EQUALS:		DW	COLON_MC
		DW	SUB_,ZERO_EQUAL,SEMICOLON_S

GT_THAN_NFA:	DB	&H81,">"+&H80
		DW	EQUALS_NFA
GREATER_THAN:	DW	GT_THAN_MC
GT_THAN_MC:	PPUW	$2
		PPUW	$0
		JR	LESS_THAN_MC_1

LESS_THAN_NFA:	DB	&H81,"<"+&H80
		DW	GT_THAN_NFA
LESS_THAN:	DW	LESS_THAN_MC
LESS_THAN_MC:	PPUM	$0,4
LESS_THAN_MC_1:	XRW	$4,$4
		SBW	$2,$0
		BIUW	$3,JR ROT_MC_1

U_LESSTHAN_NFA:	DB	&H82,"U","<"+&H80
		DW	LESS_THAN_NFA
U_LESS_THAN:	DW	U_LESS_THAN_MC
U_LESS_THAN_MC:	PPUM	$0,4
		XRW	$4,$4
		SBCW	$2,$0
		ROU	$4,JR ROT_MC_1

ROT_NFA:	DB	&H83,"RO","T"+&H80
		DW	U_LESSTHAN_NFA
ROT:		DW	ROT_MC
ROT_MC:		PPUM	$0,6
		PHUM	$3,4
ROT_MC_1:	PHUW	$5
		JP	NEXT1

SPACE_NFA:	DB	&H85,"SPAC","E"+&H80
		DW	ROT_NFA
SPACE:		DW	COLON_MC
		DW	BL_,EMIT,SEMICOLON_S

DASH_DUP_NFA:	DB	&H84,"-DU","P"+&H80
		DW	SPACE_NFA
DASH_DUP:	DW	COLON_MC
		DW	DUP_,ZERO_BRANCH,4,DUP_,SEMICOLON_S

TRAVERSE_NFA:	DB	&H88,"TRAVERS","E"+&H80
		DW	DASH_DUP_NFA
TRAVERSE:	DW	COLON_MC
		DW	SWAP
TRAVERSE_1:	DW	OVER,ADD_,DUP_,C_AT,LIT,127,GREATER_THAN
		DW	ZERO_BRANCH,TRAVERSE_1-TRAVERSE_2+2
TRAVERSE_2:	DW	SWAP,DROP,SEMICOLON_S

LATEST_NFA:	DB	&H86,"LATES","T"+&H80
		DW	TRAVERSE_NFA
LATEST:		DW	COLON_MC
		DW	CURRENT,AT,AT,SEMICOLON_S

LFA_NFA:	DB	&H83,"LF","A"+&H80
		DW	LATEST_NFA
LFA:		DW	COLON_MC
		DW	LIT,4,SUB_,SEMICOLON_S

CFA_NFA:	DB	&H83,"CF","A"+&H80
		DW	LFA_NFA
CFA:		DW	COLON_MC
		DW	TWO,SUB_,SEMICOLON_S

NFA_NFA:	DB	&H83,"NF","A"+&H80
		DW	CFA_NFA
NFA:		DW	COLON_MC
		DW	LIT,5,SUB_,LIT,-1,TRAVERSE,SEMICOLON_S

PFA_NFA:	DB	&H83,"PF","A"+&H80
		DW	NFA_NFA
PFA:		DW	COLON_MC
		DW	ONE,TRAVERSE,LIT,5,ADD_,SEMICOLON_S

STORE_CSP_NFA:	DB	&H84,"!CS","P"+&H80
		DW	PFA_NFA
STORE_CSP:	DW	COLON_MC
		DW	SP_FETCH,CSP,STORE,SEMICOLON_S

SMUDGE_NFA:	DB	&H86,"SMUDG","E"+&H80
		DW	STORE_CSP_NFA
SMUDGE:		DW	COLON_MC
		DW	LATEST,LIT,32,TOGGLE,SEMICOLON_S

HEX_NFA:	DB	&H83,"HE","X"+&H80
		DW	SMUDGE_NFA
HEX:		DW	COLON_MC
		DW	LIT,16,BASE,STORE,SEMICOLON_S

DECIMAL_NFA:	DB	&H87,"DECIMA","L"+&H80
		DW	HEX_NFA
DECIMAL:	DW	COLON_MC
		DW	LIT,10,BASE,STORE,SEMICOLON_S

COUNT_NFA:	DB	&H85,"COUN","T"+&H80
		DW	DECIMAL_NFA
COUNT:		DW	COLON_MC
		DW	DUP_,ADD_1,SWAP,C_AT,SEMICOLON_S

TYPE_NFA:	DB	&H84,"TYP","E"+&H80
		DW	COUNT_NFA
TYPE_:		DW	COLON_MC
		DW	DASH_DUP
		DW	ZERO_BRANCH,TYPE_3-TYPE_0+2	;don't print if n=0
TYPE_0:		DW	OVER,ADD_,SWAP,PAR_DO
TYPE_1:		DW	I,C_AT,EMIT
		DW	PAR_LOOP,TYPE_1-TYPE_2+2
TYPE_2:		DW	SEMICOLON_S
TYPE_3:		DW	DROP,SEMICOLON_S		;drop the address

DASH_TRAIL_NFA:	DB	&H89,"-TRAILIN","G"+&H80
		DW	TYPE_NFA
DASH_TRAILING:	DW	COLON_MC
		DW	DUP_,ZERO,PAR_DO
DASH_TRAIL_1:	DW	OVER,OVER,ADD_,ONE,SUB_,C_AT,BL_,SUB_
		DW	ZERO_BRANCH,DASH_TRAIL_4-DASH_TRAIL_2+2
DASH_TRAIL_2:	DW	LEAVE
		DW	BRANCH,DASH_TRAIL_5-DASH_TRAIL_4+2
DASH_TRAIL_4:	DW	ONE,SUB_			;drop the space
DASH_TRAIL_5:	DW	PAR_LOOP,DASH_TRAIL_1-DASH_TRAIL_6+2
DASH_TRAIL_6:	DW	SEMICOLON_S

PAR_DOT_Q_NFA:	DB	&H84,"(.",&H22,")"+&H80
		DW	DASH_TRAIL_NFA
PAR_DOT_QUOTE:	DW	COLON_MC
		DW	R,COUNT,DUP_,ADD_1,R_FROM,ADD_,TO_R
		DW	TYPE_,SEMICOLON_S

S_TO_D_NFA:	DB	&H84,"S->","D"+&H80
		DW	PAR_DOT_Q_NFA
S_TO_D:		DW	S_TO_D_MC
S_TO_D_MC:	PPUW	$2
		XRW	$0,$0
		ANC	$3,&H80
		JR	Z,S_TO_D_MC_1
		NAW	$0,$0
S_TO_D_MC_1:	PHUM	$3,4
		JP	NEXT1

PLUS_MINUS_NFA:	DB	&H82,"+","-"+&H80
		DW	S_TO_D_NFA
PLUS_MINUS:	DW	COLON_MC
		DW	ZERO_LESS_THAN,ZERO_BRANCH,4,MINUS,SEMICOLON_S

D_PL_MIN_NFA:	DB	&H83,"D+","-"+&H80
		DW	PLUS_MINUS_NFA
D_PLUS_MINUS:	DW	COLON_MC
		DW	ZERO_LESS_THAN,ZERO_BRANCH,4,D_MINUS,SEMICOLON_S

ABS_NFA:	DB	&H83,"AB","S"+&H80
		DW	D_PL_MIN_NFA
ABS:		DW	COLON_MC
		DW	DUP_,PLUS_MINUS,SEMICOLON_S

D_ABS_NFA:	DB	&H84,"DAB","S"+&H80
		DW	ABS_NFA
D_ABS:		DW	COLON_MC
		DW	DUP_,D_PLUS_MINUS,SEMICOLON_S

MIN_NFA:	DB	&H83,"MI","N"+&H80
		DW	D_ABS_NFA
MIN:		DW	COLON_MC
		DW	OVER,OVER,GREATER_THAN,ZERO_BRANCH,4
		DW	SWAP,DROP,SEMICOLON_S

MAX_NFA:	DB	&H83,"MA","X"+&H80
		DW	MIN_NFA
MAX:		DW	COLON_MC
		DW	OVER,OVER,LESS_THAN,ZERO_BRANCH,4
		DW	SWAP,DROP,SEMICOLON_S

M_MUL_NFA:	DB	&H82,"M","*"+&H80
		DW	MAX_NFA
M_MUL:		DW	COLON_MC
		DW	OVER,OVER,XOR_,TO_R,ABS
		DW	SWAP,ABS,U_MUL,R_FROM,D_PLUS_MINUS
		DW	SEMICOLON_S

M_DIV_NFA:	DB	&H82,"M","/"+&H80
		DW	M_MUL_NFA
M_DIV:		DW	COLON_MC
		DW	OVER,TO_R,TO_R,D_ABS,R,ABS,U_DIV
		DW	R_FROM,R,XOR_,PLUS_MINUS,SWAP,R_FROM,PLUS_MINUS,SWAP
		DW	SEMICOLON_S

MUL_NFA:	DB	&H81,"*"+&H80
		DW	M_DIV_NFA
MUL_:		DW	COLON_MC
		DW	M_MUL,DROP,SEMICOLON_S

DIVIDE_MOD_NFA:	DB	&H84,"/MO","D"+&H80
		DW	MUL_NFA
DIVIDE_MOD:	DW	COLON_MC
		DW	TO_R,S_TO_D,R_FROM,M_DIV,SEMICOLON_S

DIVIDE_NFA:	DB	&H81,"/"+&H80
		DW	DIVIDE_MOD_NFA
DIVIDE:		DW	COLON_MC
		DW	DIVIDE_MOD,SWAP,DROP,SEMICOLON_S

MOD_NFA:	DB	&H83,"MO","D"+&H80
		DW	DIVIDE_NFA
MOD_:		DW	COLON_MC
		DW	DIVIDE_MOD,DROP,SEMICOLON_S

MULDIV_MOD_NFA:	DB	&H85,"*/MO","D"+&H80
		DW	MOD_NFA
MUL_DIV_MOD:	DW	MUL_DIV_MOD_MC
MUL_DIV_MOD_MC:	DW	COLON_MC
		DW	TO_R,M_MUL,R_FROM,M_DIV,SEMICOLON_S

MUL_DIVIDE_NFA:	DB	&H82,"*","/"+&H80
		DW	MULDIV_MOD_NFA
MUL_DIVIDE:	DW	COLON_MC
		DW	MUL_DIV_MOD,SWAP,DROP,SEMICOLON_S

M_DIV_MOD_NFA:	DB	&H85,"M/MO","D"+&H80
		DW	MUL_DIVIDE_NFA
M_DIVIDE_MOD:	DW	COLON_MC
		DW	TO_R,ZERO,R,U_DIV,R_FROM,SWAP,TO_R,U_DIV,R_FROM
		DW	SEMICOLON_S

SPACES_NFA:	DB	&H86,"SPACE","S"+&H80
		DW	M_DIV_MOD_NFA
SPACES:		DW	COLON_MC
		DW	ZERO,MAX,DASH_DUP
		DW	ZERO_BRANCH,SPACES_2-SPACES_1+2
SPACES_1:	DW	ZERO,PAR_DO,SPACE,PAR_LOOP,-4
SPACES_2:	DW	SEMICOLON_S

LT_HASH_NFA:	DB	&H82,"<","#"+&H80
		DW	SPACES_NFA
LT_HASH:	DW	COLON_MC
		DW	PAD,HLD,STORE,SEMICOLON_S

HASH_GT_NFA:	DB	&H82,"#",">"+&H80
		DW	LT_HASH_NFA
HASH_GT:	DW	COLON_MC
		DW	DROP,DROP,HLD,AT,PAD,OVER,SUB_,SEMICOLON_S

HOLD_NFA:	DB	&H84,"HOL","D"+&H80
		DW	HASH_GT_NFA
HOLD:		DW	COLON_MC
		DW	LIT,-1,HLD,PLUS_STORE,HLD,AT,C_STORE,SEMICOLON_S

SIGN_NFA:	DB	&H84,"SIG","N"+&H80
		DW	HOLD_NFA
SIGN:		DW	COLON_MC
		DW	ROT,ZERO_LESS_THAN
		DW	ZERO_BRANCH,SIGN_2-SIGN_1+2
SIGN_1:		DW	LIT,45,HOLD
SIGN_2:		DW	SEMICOLON_S

HASH_NFA:	DB	&H81,"#"+&H80
		DW	SIGN_NFA
HASH:		DW	COLON_MC
		DW	BASE,AT,M_DIVIDE_MOD,ROT,LIT,9,OVER,LESS_THAN
		DW	ZERO_BRANCH,HASH_2-HASH_1+2
HASH_1:		DW	LIT,7,ADD_
HASH_2:		DW	LIT,"0",ADD_,HOLD,SEMICOLON_S

HASH_S_NFA:	DB	&H82,"#","S"+&H80
		DW	HASH_NFA
HASH_S:		DW	COLON_MC
HASH_S_1:	DW	HASH,OVER,OVER,OR_,ZERO_EQUAL
		DW	ZERO_BRANCH,HASH_S_1-HASH_S_2+2
HASH_S_2:	DW	SEMICOLON_S

D_DOT_R_NFA:	DB	&H83,"D.","R"+&H80
		DW	HASH_S_NFA
D_DOT_R:	DW	COLON_MC
		DW	TO_R,SWAP,OVER,D_ABS
		DW	LT_HASH,HASH_S,SIGN,HASH_GT
		DW	R_FROM,OVER,SUB_,SPACES,TYPE_,SEMICOLON_S

DOT_R_NFA:	DB	&H82,".","R"+&H80
		DW	D_DOT_R_NFA
DOT_R:		DW	COLON_MC
		DW	TO_R,S_TO_D,R_FROM,D_DOT_R,SEMICOLON_S

D_DOT_NFA:	DB	&H82,"D","."+&H80
		DW	DOT_R_NFA
D_DOT:		DW	COLON_MC
		DW	ZERO,D_DOT_R,SPACE,SEMICOLON_S

DOT_NFA:	DB	&H81,"."+&H80
		DW	D_DOT_NFA
DOT:		DW	COLON_MC
		DW	S_TO_D,D_DOT,SEMICOLON_S

U_DOT_NFA:	DB	&H82,"U","."+&H80
		DW	DOT_NFA
U_DOT:		DW	COLON_MC
		DW	ZERO,D_DOT,SEMICOLON_S

QUEST_MARK_NFA:	DB	&H81,"?"+&H80
		DW	U_DOT_NFA
QUESTION_MARK:	DW	COLON_MC
		DW	AT,DOT,SEMICOLON_S

EXPECT_NFA:	DB	&H86,"EXPEC","T"+&H80
		DW	QUEST_MARK_NFA
EXPECT:		DW	EXPECT_MC
EXPECT_MC:	PRE	IZ,&H68C8
		LD	$0,&H06
		ST	$0,(IZ-$30)		;LCDST
		LDD	$0,(IZ+$31)		;EDCSR
		STD	$0,(IZ+&H04)		;MOEDB
		LDW	$23,&H93E6		;input
		CAL	SYSCALL
		GRE	IX,$6
		CAL	&H05EA			;process the logical line
		CAL	&H05FD			;append zero
		PPUM	$0,4
		CAL	REAL_ADDR
; copy the typed line from the EDTOP buffer to the specified address and
; position the cursor at the end of the line
; at this point IX=&H68C8, IZ=destination address, $0,$1=number of characters
		LDD	$2,(IX+&H04)		;MOEDB
		PRE	IX,&H6100		;EDTOP
EXPECT_MC_1:	ORC	$1,$0
EXPECT_MC_2:	JR	Z,EXPECT_MC_4
		LD	$3,(IX+$2)
		SBC	$3,$31
		JR	Z,EXPECT_MC_4
		STI	$3,(IZ+$31)
		AD	$2,$30
		JR	C,EXPECT_MC_3
		SBW	$0,$30,JR EXPECT_MC_2
EXPECT_MC_3:	SB	$2,$30
EXPECT_MC_4:	PRE	IX,&H68C8
		STD	$2,(IX+$31)		;EDSCR
		STD	$31,(IZ+$31)
		PRE	IX,$6,JR NEXT1E

QUERY_NFA:	DB	&H85,"QUER","Y"+&H80
		DW	EXPECT_NFA
QUERY:		DW	COLON_MC
		DW	TIB,AT,LIT,80,EXPECT,ZERO,IN_,STORE,SEMICOLON_S

; slowo &H00 przerywajace petle INTERPRET przy kompilacji z TIB
EOL_NFA:	DB	&HC1,&H00+&H80
		DW	QUERY_NFA
		DW	SEMICOLONS_MC

; nowe slowo &H1A przerywajace petle INTERPRET przy kompilacji z pliku
EOF_NFA:	DB	&HC1,&H1A+&H80
		DW	EOL_NFA
		DW	SEMICOLONS_MC

; oczekuje typu pliku i adresu nazwy pliku
; zwraca adres wpisu katalogu lub 0 gdy plik nie znaleziony
FILE_NFA:	DB	&H84,"FIL","E"+&H80
		DW	EOF_NFA
FILE:		DW	FILE_MC
FILE_MC:	PPUM	$2,4
		CAL	REAL_ADDR
		LDI	$17,(IZ+$31)	;dlugosc nazwy
		GRE	IZ,$15		;adres nazwy
		LDW	$23,&HB21C	;expand file name
		CAL	SYSCALL
		LDW	$23,&HE818	;system function FNSCH
		CAL	SYSCALL
		JR	NC,FILE_MC_1
		XRW	$6,$6		;file not found
FILE_MC_1:	PHUW	$7
NEXT1E:		JP	NEXT1

FILL_NFA:	DB	&H84,"FIL","L"+&H80
		DW	FILE_NFA
FILL:		DW	FILL_MC
FILL_MC:	PPUW	$6		;data
FILL_MC_1:	LD	$7,$6
		LDW	$8,$6
		LDM	$10,$6,4
		LDW	$23,&H0170	;fill memory block
FILL_MC_2:	PPUW	$4		;number of bytes
		PPUW	$2		;address
		CAL	REAL_ADDR
		CAL	SYSCALL
		JP	NEXT1

ERASE_NFA:	DB	&H85,"ERAS","E"+&H80
		DW	FILL_NFA
ERASE:		DW	ERASE_MC
ERASE_MC:	LDW	$23,&H016E	;CLRME, clear memory block
		JR	FILL_MC_2

BLANKS_NFA:	DB	&H86,"BLANK","S"+&H80
		DW	ERASE_NFA
BLANKS:		DW	BLANKS_MC
BLANKS_MC:	LD	$6," ",JR FILL_MC_1

PAR_NUMBER_NFA:	DB	&H88,"(NUMBER",")"+&H80
		DW	BLANKS_NFA
PAR_NUMBER:	DW	COLON_MC
PAR_NUMBER_1:	DW	ADD_1,DUP_,TO_R,C_AT,BASE,AT,DIGIT
PAR_NUMBER_2:	DW	ZERO_BRANCH,PAR_NUMBER_6-PAR_NUMBER_3+2
PAR_NUMBER_3:	DW	SWAP,BASE,AT,U_MUL,DROP,ROT,BASE,AT
		DW	U_MUL,D_ADD,DPL,AT,ADD_1
		DW	ZERO_BRANCH,PAR_NUMBER_5-PAR_NUMBER_4+2
PAR_NUMBER_4:	DW	ONE,DPL,PLUS_STORE
PAR_NUMBER_5:	DW	R_FROM,BRANCH,PAR_NUMBER_1-PAR_NUMBER_6+2
PAR_NUMBER_6:	DW	R_FROM,SEMICOLON_S

; zmiana w stosunku do fig-Forth: komunikaty o bledach odczytywane sa ze
; zbioru o nazwie ERRORS.FOR
MESSAGE_NFA:	DB	&H87,"MESSAG","E"+&H80
		DW	PAR_NUMBER_NFA
MESSAGE:	DW	COLON_MC
		DW	WARNING,AT,ZERO_BRANCH,MESSAGE_4-MESSAGE_1+2
; otwarcie zbioru ERRORS.FOR
MESSAGE_1:	DW	LIT,&H24,LIT,ERRORS_NAME,FILE
		DW	DUP_,ZERO_BRANCH,MESSAGE_3-MESSAGE_2+2
; na stosie jest adres wpisu katalogu, obliczanie adresu komunikatu
MESSAGE_2:	DW	ADD_1,AT,SWAP,LIT,32,MUL_,ADD_
; wyswietlenie komunikatu ze zbioru
		DW	LIT,30,DASH_TRAILING,TYPE_
		DW	SEMICOLON_S
; druk numeru bledu
MESSAGE_3:	DW	DROP	;zgubienie danej zwroconej przez FILE
MESSAGE_4:	DW	PAR_DOT_QUOTE
		DB	5,"MSG #"
		DW	DOT,SEMICOLON_S

ERRORS_NAME:	DB	10,"ERRORS.FOR"

ERROR_NFA:	DB	&H85,"ERRO","R"+&H80
		DW	MESSAGE_NFA
ERROR:		DW	COLON_MC
		DW	WARNING,AT,ZERO_LESS_THAN
		DW	ZERO_BRANCH,4,ABORT
		DW	SPACE,HERE,COUNT,TYPE_,PAR_DOT_QUOTE
		DB	3," ? "
		DW	MESSAGE,SP_STORE,IN_,AT,BLK,AT,QUIT,SEMICOLON_S

QUEST_ERR_NFA:	DB	&H86,"?ERRO","R"+&H80
		DW	ERROR_NFA
QUESTION_ERROR:	DW	COLON_MC
		DW	SWAP,ZERO_BRANCH,6
		DW	ERROR,SEMICOLON_S
		DW	DROP,SEMICOLON_S

QUEST_COMP_NFA:	DB	&H85,"?COM","P"+&H80
		DW	QUEST_ERR_NFA
QUESTION_COMP:	DW	COLON_MC
		DW	STATE,AT,ZERO_EQUAL,LIT,17,QUESTION_ERROR,SEMICOLON_S

QUEST_EXEC_NFA:	DB	&H85,"?EXE","C"+&H80
		DW	QUEST_COMP_NFA
QUESTION_EXEC:	DW	COLON_MC
		DW	STATE,AT,LIT,18,QUESTION_ERROR,SEMICOLON_S

QUEST_CSP_NFA:	DB	&H84,"?CS","P"+&H80
		DW	QUEST_EXEC_NFA
QUESTION_CSP:	DW	COLON_MC
		DW	SP_FETCH,CSP,AT,SUB_,LIT,20,QUESTION_ERROR,SEMICOLON_S

COMPILE_NFA:	DB	&H87,"COMPIL","E"+&H80
		DW	QUEST_CSP_NFA
COMPILE:	DW	COLON_MC
		DW	QUESTION_COMP,R_FROM,DUP_,ADD_2,TO_R,AT,COMMA
		DW	SEMICOLON_S

L_BRACKET_NFA:	DB	&HC1,"["+&H80
		DW	COMPILE_NFA
LEFT_BRACKET:	DW	COLON_MC
		DW	ZERO,STATE,STORE,SEMICOLON_S

R_BRACKET_NFA:	DB	&HC1,"]"+&H80
		DW	L_BRACKET_NFA
RIGHT_BRACKET:	DW	COLON_MC
		DW	LIT,&HC0,STATE,STORE,SEMICOLON_S

; slowo zgodne z fig-Forth, inaczej zdefiniowane pobieranie z pliku
WORD_NFA:	DB	&H84,"WOR","D"+&H80
		DW	R_BRACKET_NFA
WORD_:		DW	COLON_MC
		DW	BLK,AT,ZERO_BRANCH,WORD_3-WORD_1+2
WORD_1:		DW	BLK,AT,ADD_1
		DW	BRANCH,WORD_4-WORD_3+2
WORD_3:		DW	TIB
WORD_4:		DW	AT,IN_,AT,ADD_,SWAP,ENCLOSE,HERE,LIT,34,BLANKS
		DW	IN_,PLUS_STORE,OVER,SUB_,TO_R,R,HERE,C_STORE,ADD_,HERE
		DW	ADD_1,R_FROM,CMOVE,SEMICOLON_S

NUMBER_NFA:	DB	&H86,"NUMBE","R"+&H80
		DW	WORD_NFA
NUMBER:		DW	COLON_MC
		DW	ZERO,ZERO,ROT,DUP_,ADD_1,C_AT,LIT,"-",EQUALS
		DW	DUP_,TO_R,ADD_,LIT,-1
NUMBER_1:	DW	DPL,STORE,PAR_NUMBER,DUP_,C_AT,BL_,SUB_
		DW	ZERO_BRANCH,NUMBER_3-NUMBER_2+2
NUMBER_2:	DW	DUP_,C_AT,LIT,".",SUB_,ZERO,QUESTION_ERROR,ZERO
		DW	BRANCH,NUMBER_1-NUMBER_3+2
NUMBER_3:	DW	DROP,R_FROM,ZERO_BRANCH,4,D_MINUS,SEMICOLON_S

DASH_FIND_NFA:	DB	&H85,"-FIN","D"+&H80
		DW	NUMBER_NFA
DASH_FIND:	DW	COLON_MC
		DW	BL_,WORD_,HERE,CONTEXT,AT,AT,PAR_FIND,DUP_,ZERO_EQUAL
		DW	ZERO_BRANCH,DASH_FIND_2-DASH_FIND_1+2
DASH_FIND_1:	DW	DROP,HERE,LATEST,PAR_FIND
DASH_FIND_2:	DW	SEMICOLON_S

ID_DOT_NFA:	DB	&H83,"ID","."+&H80
		DW	DASH_FIND_NFA
ID_DOT:		DW	COLON_MC
		DW	COUNT,LIT,&H1F,AND_
		DW	DASH_DUP,ZERO_BRANCH,ID_DOT_4-ID_DOT_1+2
ID_DOT_1:	DW	OVER,ADD_,SWAP,PAR_DO
ID_DOT_2:	DW	I,C_AT,LIT,&H7F,AND_,EMIT,PAR_LOOP,ID_DOT_2-ID_DOT_3+2
ID_DOT_3:	DW	SPACE,SEMICOLON_S
ID_DOT_4:	DW	DROP,SEMICOLON_S

CREATE_NFA:	DB	&H86,"CREAT","E"+&H80
		DW	ID_DOT_NFA
CREATE:		DW	COLON_MC
		DW	DASH_FIND,ZERO_BRANCH,CREATE_2-CREATE_1+2
CREATE_1:	DW	DROP,NFA,ID_DOT,LIT,4,MESSAGE,SPACE
CREATE_2:	DW	HERE,DUP_,C_AT,WIDTH_,AT,MIN
		DW	ADD_1,ALLOT,DUP_,LIT,&HA0,TOGGLE
		DW	HERE,ONE,SUB_,LIT,&H80,TOGGLE
		DW	LATEST,COMMA,CURRENT,AT,STORE
		DW	HERE,ADD_2,COMMA,SEMICOLON_S

PARSEMCODE_NFA:	DB	&H87,"(;CODE",")"+&H80
		DW	CREATE_NFA
PAR_SEM_CODE:	DW	COLON_MC
		DW	R_FROM,LATEST,PFA,CFA,STORE,SEMICOLON_S

COLON_NFA:	DB	&HC1,":"+&H80
		DW	PARSEMCODE_NFA
		DW	COLON_MC
		DW	QUESTION_EXEC,STORE_CSP,CURRENT,AT,CONTEXT,STORE
		DW	CREATE,RIGHT_BRACKET,PAR_SEM_CODE
; program pola kodu slow definiowanych w Forth
COLON_MC:	GRE	IX,$0		;real IP
		SBW	$0,$25		;real->virtual address conversion
		PHSW	$1		;virtual IP -> return stack
		PRE	IX,$2		;real W
		LDI	$0,(IX+$30)	;IP <- W+2
		JP	NEXT1

SEMICOLON_NFA:	DB	&HC1,";"+&H80
		DW	COLON_NFA
		DW	COLON_MC
		DW	QUESTION_CSP,COMPILE,SEMICOLON_S,SMUDGE,LEFT_BRACKET
		DW	SEMICOLON_S

DOES_NFA:	DB	&H85,"DOES",">"+&H80
		DW	SEMICOLON_NFA
DOES:		DW	COLON_MC
		DW	R_FROM,LATEST,PFA,STORE,PAR_SEM_CODE
DOES_MC:	GRE	IX,$0		;real IP
		SBW	$0,$25		;real->virtual address conversion
		PHSW	$1		;IP -> return stack
;
		PRE	IZ,$2		;real W
		LD	$0,2
		LDW	$0,(IZ+$0)
		ADW	$0,$25
		PRE	IX,$0		;real address in the PFA field -> IP
;
		SBW	$2,$25		;virtual W
		LDW	$0,4
		ADW	$2,$0,JR PUSH1F	;virtual address after PFA -> data stack

CONSTANT_NFA:	DB	&H88,"CONSTAN","T"+&H80
		DW	DOES_NFA
CONSTANT:	DW	COLON_MC
		DW	CREATE,COMMA,SMUDGE,PAR_SEM_CODE
CONSTANT_MC:	PRE	IZ,$2			;real W
		LD	$0,2
		LDW	$2,(IZ+$0)
PUSH1F:		JP	PUSH1

VARIABLE_NFA:	DB	&H88,"VARIABL","E"+&H80
		DW	CONSTANT_NFA
		DW	COLON_MC
		DW	CONSTANT,PAR_SEM_CODE
VARIABLE_MC:	SBW	$2,$25			;virtual W
		LDW	$0,2
		ADW	$2,$0,JR PUSH1F		;virtual W+2 -> data stack

USER_NFA:	DB	&H84,"USE","R"+&H80
		DW	VARIABLE_NFA
		DW	COLON_MC
		DW	CONSTANT,PAR_SEM_CODE
USER_MC:	PRE	IZ,$2			;real W
		LD	$0,2
		LDW	$0,(IZ+$0)
		LDW	$2,ORIGIN_ADDR+&H26	;USER_AREA_PTR
		CAL	REAL_ADDR
		LDW	$2,(IZ+$31)	;virtual address of the USER_AREA
		ADW	$2,$0,JR PUSH1F

BUILDS_NFA:	DB	&H87,"<BUILD","S"+&H80
		DW	USER_NFA
BUILDS:		DW	COLON_MC
		DW	ZERO,CONSTANT,SEMICOLON_S

LITERAL_NFA:	DB	&HC7,"LITERA","L"+&H80
		DW	BUILDS_NFA
LITERAL:	DW	COLON_MC
		DW	STATE,AT,ZERO_BRANCH,8
		DW	COMPILE,LIT,COMMA,SEMICOLON_S

DLITERAL_NFA:	DB	&HC8,"DLITERA","L"+&H80
		DW	LITERAL_NFA
DLITERAL:	DW	COLON_MC
		DW	STATE,AT,ZERO_BRANCH,8
		DW	SWAP,LITERAL,LITERAL,SEMICOLON_S

QUESTSTACK_NFA:	DB	&H86,"?STAC","K"+&H80
		DW	DLITERAL_NFA
QUESTION_STACK:	DW	COLON_MC
		DW	SP_FETCH,S_ZERO,AT,SWAP,U_LESS_THAN
		DW	ONE,QUESTION_ERROR
		DW	SP_FETCH,S_ZERO,AT,LIT,128,SUB_,U_LESS_THAN
		DW	LIT,7,QUESTION_ERROR,SEMICOLON_S

INTERPRET_NFA:	DB	&H89,"INTERPRE","T"+&H80
		DW	QUESTSTACK_NFA
INTERPRET:	DW	COLON_MC
INTERPRET_1:	DW	QUESTION_STACK,DASH_FIND
		DW	ZERO_BRANCH,INTERPRET_5-INTERPRET_2+2
INTERPRET_2:	DW	STATE,AT,LESS_THAN
		DW	ZERO_BRANCH,INTERPRET_4-INTERPRET_3+2
INTERPRET_3:	DW	CFA,COMMA,BRANCH,INTERPRET_1-INTERPRET_4+2
INTERPRET_4:	DW	CFA,EXECUTE,BRANCH,INTERPRET_1-INTERPRET_5+2
INTERPRET_5:	DW	HERE,NUMBER,DPL,AT,ADD_1
		DW	ZERO_BRANCH,INTERPRET_8-INTERPRET_7+2
INTERPRET_7:	DW	DLITERAL,BRANCH,INTERPRET_1-INTERPRET_8+2
INTERPRET_8:	DW	DROP,LITERAL,BRANCH,INTERPRET_1-INTERPRET_9+2
INTERPRET_9:

VOCABULARY_NFA:	DB	&HCA,"VOCABULAR","Y"+&H80
		DW	INTERPRET_NFA
    		DW	COLON_MC
		DW	BUILDS
		DW	LIT,&HA081,COMMA
		DW	CURRENT,AT,TWO,SUB_,COMMA
		DW	HERE,VOC_LINK,AT,COMMA,VOC_LINK,STORE
		DW	DOES
VOCABULARY_1:	DW	ADD_2,CONTEXT,STORE,SEMICOLON_S

FORTH_NFA:	DB	&HC5,"FORT","H"+&H80
		DW	VOCABULARY_NFA
FORTH:		DW	DOES_MC
		DW	VOCABULARY_1
		DW	&HA081
FORTH_CONTEXT:
FORTH_CURRENT:	DW	LAST_DEFIN
FORTH_LINK:	DW	0

DEFIN_NFA:	DB	&H8B,"DEFINITION","S"+&H80
		DW	FORTH_NFA
DEFINITIONS:	DW	COLON_MC
		DW	CONTEXT,AT,CURRENT,STORE,SEMICOLON_S

LEFT_PAREN_NFA:	DB	&HC1,"("+&H80
		DW	DEFIN_NFA
		DW	COLON_MC
		DW	LIT,")",WORD_,SEMICOLON_S

QUIT_NFA:	DB	&H84,"QUI","T"+&H80
		DW	LEFT_PAREN_NFA
QUIT:		DW	COLON_MC
QUIT_PFA:	DW	ZERO,BLK,STORE,LEFT_BRACKET
QUIT_1:		DW	RP_STORE,CR,QUERY,SPACE,INTERPRET,STATE,AT,ZERO_EQUAL
		DW	ZERO_BRANCH,QUIT_1-QUIT_2+2
QUIT_2:		DW	PAR_DOT_QUOTE
		DB	3," ok"
		DW	BRANCH,QUIT_1-QUIT_3+2
QUIT_3:

ABORT_NFA:	DB	&H85,"ABOR","T"+&H80
		DW	QUIT_NFA
ABORT:		DW	COLON_MC
		DW	SP_STORE,DECIMAL,QUESTION_STACK,PAR_DOT_QUOTE
		DB	ABORT_2-ABORT_1
ABORT_1:	DB	&H0C,"Forth for PB-1000"
ABORT_2:	DW	CR
		DW	FORTH,DEFINITIONS
		DW	QUIT

COLD_NFA:	DB	&H84,"COL","D"+&H80
		DW	ABORT_NFA
COLD:		DW	COLON_MC
COLD_PFA:	DW	LIT,S0_INIT,LIT,USER_AREA_PTR,AT
		DW	LIT,16,CMOVE
		DW	LIT,LAST_FORTH_DEF,AT
		DW	LIT,FORTH_CONTEXT,STORE
		DW	RESIZE,ABORT

; new word compiled by LOAD
PAR_LOAD_NFA:	DB	&H86,"(LOAD",")"+&H80
		DW	COLD_NFA
PAR_LOAD:	DW	COLON_MC
; move the return pointer past the file name
		DW	R,R,C_AT,ADD_1,R_FROM,ADD_,TO_R
; expects a pointer to the file entry on the stack
PAR_LOAD_1:	DW	LIT,&H24,SWAP,FILE
		DW	DUP_,ZERO_EQUAL,LIT,9,QUESTION_ERROR
		DW	BLK,AT,TO_R,BLK,STORE
		DW	IN_,AT,TO_R,ZERO,IN_,STORE
		DW	INTERPRET
		DW	R_FROM,IN_,STORE,R_FROM,BLK,STORE
		DW	SEMICOLON_S

; similar function as in the fig-Forth but differently defined,
; interprets a script from a text file,
; expects a filename following the word LOAD
LOAD_NFA:	DB	&HC4,"LOA","D"+&H80
		DW	PAR_LOAD_NFA
LOAD:		DW	COLON_MC
		DW	BL_,STATE,AT
		DW	ZERO_BRANCH,LOAD_2-LOAD_1+2
LOAD_1:		DW	COMPILE,PAR_LOAD,WORD_,HERE,C_AT,ADD_1,ALLOT
		DW	SEMICOLON_S
LOAD_2:		DW	WORD_,HERE,BRANCH,PAR_LOAD_1-LOAD_3+2
LOAD_3:

NEXT1_NFA:	DB	&H85,"NEXT","1"+&H80
		DW	LOAD_NFA
		DW	CONSTANT_MC,NEXT1

SYSCALL_NFA:	DB	&H87,"SYSCAL","L"+&H80
		DW	NEXT1_NFA
		DW	CONSTANT_MC,SYSCALL

REAL_ADDR_NFA:	DB	&H89,"REAL_ADD","R"+&H80
		DW	SYSCALL_NFA
		DW	CONSTANT_MC,REAL_ADDR

LAST_DEFIN:

MON_NFA:	DB	&H83,"MO","N"+&H80
		DW	REAL_ADDR_NFA
MON:		DW	COLON_MC
		DW	PAR_DOT_QUOTE
		DB	5," Bye!"
		DW	CR,MON_MC_1
MON_MC_1:	DW	MON_MC_2
MON_MC_2:	LD	$0,DP_INIT
		PRE	IZ,$25
		LDW	$0,(IZ+$0)
		ADW	$0,$25		;virtual->real address conversion
		LDW	$2,$27		;end address of the file
		SBW	$2,$30		;skip the EOF byte
		SBW	$2,$0		;$2,$3 = free space in the dictionary file
		JR	C,MON_MC_3
		LDW	$23,&HB26C	;shrink memory block for a seq. file
		CAL	SYSCALL
MON_MC_3:	LDW	$23,&H94E2	;CAL mode (&HECA6 for MENU)
		JP	SYSJUMP

HERE_ADDR:
