/* This program converts a compressed .PGM file to a text .B74 file. */

#include <stdio.h>
#include <string.h>
#include "typedefs.h"
#include "pgm2b74.h"


uchar pgmbuf[MAXPGM+1];	/* buffer for the compressed source file */
uint pgmlen;		/* length of the compressed source file */

uint pseglen;		/* length of the program segment */

char linebuf[MAXLINE+512];	/* buffer for the output text line */
char *lineptr;

struct codetab codes1[] = {
  { "",		extend	},	/* code 0x80 */
  { "DISPLAY",	token	},	/* code 0x81 */
  { "REM",	token	},	/* code 0x82 */
  { "DIM",	token	},	/* code 0x83 */
  { "IMAGE",	token	},	/* code 0x84 */
  { "STOP",	token	},	/* code 0x85 */
  { "END",	token	},	/* code 0x86 */
  { "LET",	token	},	/* code 0x87 */
  { "INPUT",	token	},	/* code 0x88 */
  { "LINPUT",	token	},	/* code 0x89 */
  { "PRINT",	token	},	/* code 0x8A */
  { "PAUSE",	token	},	/* code 0x8B */
  { "OPEN",	token	},	/* code 0x8C */
  { "CLOSE",	token	},	/* code 0x8D */
  { "RESTORE",	token	},	/* code 0x8E */
  { "",		unknown	},	/* code 0x8F */
  { "RANDOMIZE",token	},	/* code 0x90 */
  { "ON",	token	},	/* code 0x91 */
  { "GOTO",	token	},	/* code 0x92 */
  { "GOSUB",	token	},	/* code 0x93 */
  { "RETURN",	token	},	/* code 0x94 */
  { "CALL",	token	},	/* code 0x95 */
  { "",		unknown	},	/* code 0x96 */
  { "",		unknown	},	/* code 0x97 */
  { "SUB",	token	},	/* code 0x98 */
  { "SUBEXIT",	token	},	/* code 0x99 */
  { "SUBEND",	token	},	/* code 0x9A */
  { "FOR",	token	},	/* code 0x9B */
  { "NEXT",	token	},	/* code 0x9C */
  { "IF",	token	},	/* code 0x9D */
  { "ELSE",	token2	},	/* code 0x9E */
  { "",		unknown	},	/* code 0x9F */
  { "!",	token2	},	/* code 0xA0 */
  { "READ",	token	},	/* code 0xA1 */
  { "DATA",	token	},	/* code 0xA2 */
  { "ACCEPT",	token	},	/* code 0xA3 */
  { "",		unknown	},	/* code 0xA4 */
  { "",		unknown	},	/* code 0xA5 */
  { "",		unknown	},	/* code 0xA6 */
  { "",		unknown	},	/* code 0xA7 */
  { "",		unknown	},	/* code 0xA8 */
  { "",		unknown	},	/* code 0xA9 */
  { "THEN",	token	},	/* code 0xAA */
  { "TO",	token	},	/* code 0xAB */
  { "STEP",	token	},	/* code 0xAC */
  { ",",	token	},	/* code 0xAD */
  { ";",	token	},	/* code 0xAE */
  { ")",	token	},	/* code 0xAF */
  { "OR",	token	},	/* code 0xB0 */
  { "AND",	token	},	/* code 0xB1 */
  { "XOR",	token	},	/* code 0xB2 */
  { "<>",	token	},	/* code 0xB3 */
  { "<=",	token	},	/* code 0xB4 */
  { ">=",	token	},	/* code 0xB5 */
  { "=",	token	},	/* code 0xB6 */
  { "<",	token	},	/* code 0xB7 */
  { ">",	token	},	/* code 0xB8 */
  { "&",	token	},	/* code 0xB9 */
  { "+",	token	},	/* code 0xBA */
  { "-",	token	},	/* code 0xBB */
  { "*",	token	},	/* code 0xBC */
  { "/",	token	},	/* code 0xBD */
  { "^",	token	},	/* code 0xBE */
  { "",		unknown	},	/* code 0xBF */
  { "(",	token	},	/* code 0xC0 */
  { "NOT",	token	},	/* code 0xC1 */
  { "",		number	},	/* code 0xC2 */
  { "",		number	},	/* code 0xC3 */
  { "",		number	},	/* code 0xC4 */
  { "",		number	},	/* code 0xC5 */
  { "",		number	},	/* code 0xC6 */
  { "",		number	},	/* code 0xC7 */
  { "",		number	},	/* code 0xC8 */
  { "",		string1	},	/* code 0xC9 */
  { "",		string2	},	/* code 0xCA */
  { "",		lineno	},	/* code 0xCB */
  { "",		unknown	},	/* code 0xCC */
  { "SEG$",	token	},	/* code 0xCD */
  { "RPT$",	token	},	/* code 0xCE */
  { "POS",	token	},	/* code 0xCF */
  { "LEN",	token	},	/* code 0xD0 */
  { "VAL",	token	},	/* code 0xD1 */
  { "NUMERIC",	token	},	/* code 0xD2 */
  { "ASC",	token	},	/* code 0xD3 */
  { "RND",	token	},	/* code 0xD4 */
  { "PI",	token	},	/* code 0xD5 */
  { "KEY$",	token	},	/* code 0xD6 */
  { "CHR$",	token	},	/* code 0xD7 */
  { "STR$",	token	},	/* code 0xD8 */
  { "ABS",	token	},	/* code 0xD9 */
  { "ACOS",	token	},	/* code 0xDA */
  { "ASIN",	token	},	/* code 0xDB */
  { "ATN",	token	},	/* code 0xDC */
  { "COS",	token	},	/* code 0xDD */
  { "EXP",	token	},	/* code 0xDE */
  { "INT",	token	},	/* code 0xDF */
  { "ATANH",	token	},	/* code 0xE0 */
  { "LN",	token	},	/* code 0xE1 */
  { "LOG",	token	},	/* code 0xE2 */
  { "SGN",	token	},	/* code 0xE3 */
  { "SIN",	token	},	/* code 0xE4 */
  { "SQR",	token	},	/* code 0xE5 */
  { "TAN",	token	},	/* code 0xE6 */
  { "EOF",	token	},	/* code 0xE7 */
  { "FRE",	token	},	/* code 0xE8 */
  { "SINH",	token	},	/* code 0xE9 */
  { "COSH",	token	},	/* code 0xEA */
  { "TANH",	token	},	/* code 0xEB */
  { "ASINH",	token	},	/* code 0xEC */
  { "ACOSH",	token	},	/* code 0xED */
  { "NULL",	token	},	/* code 0xEE */
  { "VALIDATE",	token	},	/* code 0xEF */
  { "#",	token	},	/* code 0xF0 */
  { "ALL",	token	},	/* code 0xF1 */
  { "TAB",	token	},	/* code 0xF2 */
  { "USING",	token	},	/* code 0xF3 */
  { "INTERNAL",	token	},	/* code 0xF4 */
  { "OUTPUT",	token	},	/* code 0xF5 */
  { "UPDATE",	token	},	/* code 0xF6 */
  { "APPEND",	token	},	/* code 0xF7 */
  { "VARIABLE",	token	},	/* code 0xF8 */
  { "SIZE",	token	},	/* code 0xF9 */
  { "AT",	token	},	/* code 0xFA */
  { "REC",	token	},	/* code 0xFB */
  { "ERASE",	token	},	/* code 0xFC */
  { "RELATIVE",	token	},	/* code 0xFD */
  { "",		unknown	}	/* code 0xFE */
};


/* extended codes */
struct codetab codes2[] = {
  { "RUN",	token	},	/* code 0x40 */
  { "DELETE",	token	},	/* code 0x41 */
  { "FORMAT",	token	},	/* code 0x42 */
  { "BREAK",	token	},	/* code 0x43 */
  { "UNBREAK",	token	},	/* code 0x44 */
  { "DEG",	token	},	/* code 0x45 */
  { "RAD",	token	},	/* code 0x46 */
  { "GRAD",	token	},	/* code 0x47 */
  { "WARNING",	token	},	/* code 0x48 */
  { "ERROR",	token	},	/* code 0x49 */
  { "PROTECTED",token	},	/* code 0x4A */
  { "DIGIT",	token	},	/* code 0x4B */
  { "UALPHANUM",token	},	/* code 0x4C */
  { "UALPHA",	token	},	/* code 0x4D */
  { "ALPHANUM",	token	},	/* code 0x4E */
  { "ALPHA",	token	}	/* code 0x4F */
};


int main (int argc, char **argv)
{
  FILE *fp;
  int x;
  uint i, c;
  uchar *ptr;

  if (argc<=1)
  {
    fprintf (stderr, "\nNo source file specified.\n");
    return 1;
  }

  if ((fp = fopen(*++argv, "rb")) == NULL)
  {
    fprintf (stderr, "\nCannot open source file: %s\n", *argv);
    return 1;
  }

  for (pgmlen=0; pgmlen<MAXPGM+1; pgmlen++)
  {
    if ((x = getc(fp)) == EOF)
      break;
    pgmbuf[pgmlen] = (uchar) x;
  }
  (void) fclose (fp);

  if (argc<=2)
  {
    filext (*argv, ".B74");
  }
  else
  {
    argv++;
  }
  if ((fp = fopen(*argv, "wt")) == NULL)
  {
    fprintf (stderr, "\nCannot open output file: %s\n", *argv);
    return 1;
  }

  if (pgmlen<9 || pgmlen>MAXPGM)
  {
    fprintf (stderr, "\nInvalid length of source file: %s\n", *argv);
    return 1;
  }

  pseglen = getword(&pgmbuf[2]);
  ptr = &pgmbuf[pseglen];
  if (pseglen < 5 || pseglen > pgmlen-4 ||
	pgmbuf[0] != (uchar) 0x80 || pgmbuf[1] != (uchar) 0x03 ||
	ptr[0] != (uchar) 0x7F || ptr[1] != (uchar) 0x03 ||
	ptr[2] != (uchar) 0x86 || ptr[3] != (uchar) 0x00	)
  {
    fprintf (stderr, "\nInvalid signature of source file: %s\n", *argv);
    return 1;
  }

/* lines decoding loop */
  for (ptr = &pgmbuf[4]; ptr < &pgmbuf[pseglen-1];
	ptr += 2 + (uint) ptr[2])
  {

/* decode a single BASIC line at "ptr" to the "linebuf" */
    lineptr = linebuf;
    lineptr += sprintf (lineptr, "%u ", getword(ptr));	/* line number */
    i = 3;	/* 2 bytes of the line number + 1 byte of the line length */
    while (i < 2 + (uint) ptr[2])	/* decoding loop */
    {
      c = (uint) ptr[i++];
      if (c == 0)			/* end of statement */
      {
        if (i < 2 + (uint) ptr[2])
        {
          *lineptr++ = ':';	/* separator between multiple statements */
        }
      }
      else if (c >= 0x20 && c < 0x80)	/* variable name */
      {
        if (islastnamechar((int) *(lineptr-1)) != 0)
        {
          *lineptr++ = ' ';
        }
        varname(c);
      }
      else if (c >= 0x80 && c < 0xFF)	/* item from the codes table */
      {
/* insert space between items when:
   previus ends with letter, digit or '$'
   next begins with letter, or '#', or is a number, or is a string	*/
        if ( islastnamechar((int) *(lineptr-1)) != 0 &&
		( isalpha((int) *codes1[c - 0x80].string) != 0 ||
		  *codes1[c - 0x80].string == '#' ||
		  (c >= 0xC2 && c <= 0xCB) ) )
        {
          *lineptr++ = ' ';
        }
        lineptr += sprintf (lineptr, "%s", codes1[c - 0x80].string);
        i += (*(codes1[c - 0x80].rout))(&ptr[i]);
      }
      else				/* unknown */
      {
        lineptr += sprintf (lineptr, "{%02X} ", c);
      }
      if (lineptr >= linebuf+MAXLINE)
      {
        lineptr += sprintf (lineptr, " {line_too_long}");
        break;
      }
    }
    lineptr += sprintf (lineptr, "\n");

/* write decoded BASIC line to the output file */
    if (fprintf (fp, "%s", linebuf) == EOF)
    {
      printf ("\nFile write error, disk full?\n");
      (void) fclose (fp);
      return 1;
    }
  }

  (void) fclose (fp);
  return 0;
}


/* replace the extension of the "filename" with "extension" */
void filext (char *filename, char *extension)
{
  while (*filename != '\0' && *filename != '.')
    filename++;
  while ((*filename++ = *extension++) != '\0')
    ;
}


/* test if "c" is a valid character terminating the name */
int islastnamechar (int c)
{
  return isalpha(c) | isdigit(c) | (int) (c == (int) '$');
}


/* get a 16-bit word from the pgmbuf */
uint getword (uchar *ptr)
{
  return (uint) ptr[0] + 256 * (uint) ptr[1];
}


/* place the variable name at the "lineptr" */
void varname (uint index)
{
  uint i, j, length;

  i = pseglen+4;
  j = (uint) pgmbuf[i++];	/* index of first free variable */

/* test if index of the variable < index of first free variable */
  if (index >= j)
  {
    lineptr += sprintf (lineptr, "{undefined_variable_%02X}", index);
    return;
  }

/* find the variable in the variable segment */
  while (index < --j)
  {
    i += 1 + (uint) pgmbuf[i];
    if (i >= pgmlen)
    {
      lineptr += sprintf (lineptr, "{undefined_variable_%02X}", index);
      return;
    }
  }

/* test if the variable length is valid */
  if ((length = (uint) pgmbuf[i++]) == 0)
  {
    lineptr += sprintf (lineptr, "{undefined_variable_%02X}", index);
    return;
  }

/* copy characters */
  do {
    *lineptr++ = (char) (pgmbuf[i] & (uchar) 0x7F);
  } while (pgmbuf[i++] < (uchar) 0x80 && --length != 0);
}


/* all following functions expect a pointer to the pgmbuf and return count
   of processed bytes */

uint token (/*@unused@*/ uchar *pgmptr)
{
  return 0;
}


/* replace the colon before the statement with space */
uint token2 (uchar *pgmptr)
{
  char *ptr;
  ptr = lineptr - strlen(codes1[(uint)*--pgmptr - 0x80].string) - 1;
  if (*ptr == ':')
  {
    *ptr = ' ';
  }
  return 0;
}


/* prefix, next byte contains code of command from the "codes2" list */
uint extend (uchar *pgmptr)
{
  uint c;
  c = (uint) *pgmptr;
  if (c >= 0x40 && c <= 0x4F)
  {
    if (islastnamechar((int) *(lineptr-1)) != 0 &&
		isalpha((int) *codes2[c - 0x40].string) != 0)
    {
      *lineptr++ = ' ';
    }
    lineptr += sprintf (lineptr, "%s", codes2[c - 0x40].string);
    return 1 + (*(codes2[c - 0x40].rout))(pgmptr);
  }
  lineptr += sprintf (lineptr, " {80}{%02X} ", c);
  return 1;
}


/* next 2 bytes contain the line number */
uint lineno (uchar *pgmptr)
{
  lineptr += sprintf (lineptr, "%u", getword(pgmptr));
  return 2;
}


/* next byte contains the string length followed by the string itself
   within quotes */
uint string1 (uchar *pgmptr)
{
  uint i, length;

  length = (uint) *pgmptr++;
  *lineptr++ = '\"';
  for (i=0; i<length; i++)
  {
    *lineptr++ = (char) *pgmptr++;
  }
  *lineptr++ = '\"';
  return length+1;
}

/* next byte contains the string length followed by the string itself
   without quotes */
uint string2 (uchar *pgmptr)
{
  uint i, length;

  length = (uint) *pgmptr++;
  for (i=0; i<length; i++)
  {
    *lineptr++ = (char) *pgmptr++;
  }
  return length+1;
}


/* code contains the number length, followed by the number in the RADIX-100
   format */
uint number (uchar *pgmptr)
{
  uint i, length, le, lo;
  int exponent;
  char digits[15];
  char *digptr1, *digptr2;

  length = (uint) *--pgmptr & 0x0F;
/* test for minus sign */
  exponent = (int) *++pgmptr;
  if ((exponent & 0x80) != 0)
  {
    *lineptr++ = '-';
    exponent ^= 0xFF;
  }
  exponent = 2 * exponent - 0x7F;
/* copy the mantissa to the "digits" buffer */
  digptr2 = digits;
  for (i=1; i<length; i++)
  {
    digptr2 += sprintf (digptr2, "%02X", (uint) *++pgmptr);
  }
/* remove the leading zero */
  if (digits[0] == '0')
  {
    digptr1 = &digits[1];
    exponent--;
  }
  else
  {
    digptr1 = &digits[0];
  }
/* remove trailing zeroes */
  while (digptr2 > digptr1 && *(digptr2-1) == '0')
  {
    *--digptr2 = '\0';
  }
  i = (uint) (digptr2 - digptr1);		/* count of digits */

/* test for zero value */
  if (i == 0)
  {
    *lineptr++ = '0';
  }

  else
  {
/* count of characters in an expotential display format */
    le = i + 3;	/* dec. point, char. 'E', first digit of the exponent */
    if (exponent < 0) le++;			/* minus sign character */
    if (exponent < -9 || exponent > 9) le++;	/* second digit */
    if (exponent < -99 || exponent > 99) le++;	/* third digit */
/* count of characters in an ordinary display format */
    lo = i + 1;				/* decimal point */
    if (exponent < 0)
    {
      lo -= (uint) exponent;		/* additional leading zeroes */
    }
    else
    {
      if ((uint) exponent >= i)
      {
        lo = (uint) exponent + 1;	/* additional trailing zeroes */
      }
    }
/* chose the shorter display format */
    if (lo < 7 || lo < le)
    {
      if (exponent < 0)
      {
        *lineptr++ = '0';
        *lineptr++ = '.';
        while (++exponent < 0)
        {
          *lineptr++ = '0';	/* leading zeroes */
        }
        lineptr += sprintf (lineptr, "%s", digptr1);
      }
      else if ((uint) exponent < i-1)
      {
        while (exponent-- >= 0)
        {
          *lineptr++ = *digptr1++;	/* digits before the decimal point */
        }
        lineptr += sprintf (lineptr, ".%s", digptr1);
      }
      else
      {
        lineptr += sprintf (lineptr, "%s", digptr1);
        exponent++;
        exponent -= (int) i;
        while (exponent-- > 0)
        {
          *lineptr++ = '0';		/* trailing zeroes */
        }
      }
    }
    else
    {
      *lineptr++ = *digptr1++;	/* first digit before the decimal point */
      if (i > 1)
      {
        *lineptr++ = '.';
      }
      lineptr += sprintf (lineptr, "%sE%d", digptr1, exponent);
    }

  }
  return length;
}


uint unknown (uchar *pgmptr)
{
  lineptr += sprintf (lineptr, " {%02X} ", (uint) *--pgmptr);
  return 0;
}
