/* input/output functions, hardware independent module */

#include <stdint.h>
#include "common.h"
#include "io.h"
#include "main.h"


extern uint8_t ram[RAMSIZE];


/* 0770: returns the character code of the pressed key, or 0 if the key is not
 pressed or not recognized */
int keyb0770 (void)
{
/* 0066: basic mode */
  const uint8_t tab0066[] = { 0x20, 0x2E, 0x2D, 0x2B, 0x2A, 0x2F, 0x7B, 0x3D };
/* 00A2: [MODE] on, [EXT] off */
  const uint8_t tab00a2[] = { 0x1E, 0x2E, 0xBB, 0xBD, 0xBE, 0xBF, 0xA0, 0xBC };
/* 00AA: [S] on */
  const uint8_t tab00aa[] = { 0x20, 0x5E, 0x5F, 0x7E, 0x3E, 0x3C, 0x7C, 0x5C };
/* 00B2: [MODE] on, [EXT] on */
  const uint8_t tab00b2[] = { 0x1F, 0x2E, 0x9B, 0x9D, 0x9E, 0x9F, 0x80, 0x9C };
/* 00C5: graphic characters (letter keys in mode [EXT] on, [S] on) */
  const uint8_t tab00c5[] = {
     0x0B, 0x0C, 0x0D, 0x0E, 0x40, 0x10, 0x11, 0x12,
     0x19, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x1A,
     0x25, 0x1B, 0x1C, 0x5B, 0x26, 0x0F, 0x27, 0x1D,
     0x5D, 0x7F };
/* 00BA, 006E: conversion table in the [S] mode */
  const char tab00ba[] = "EIOPQRTUWY";
  const char tab006e[] = "#:;,!$(?\")";

  int i, j;
  uint16_t mode;

  i = keyb0820 ();
  if (i == 0)
  {
    return 0;
  }
  mode = read16 (0x8256);

/* 077C: */
  ram[0] &= ~0x06;		/* [S] and [F] segments */
  lcd (0x80, ram[0]);

/* 0788: */
  if (i > 0x0A)
  {
/* 078E: */
    if (i <= 0x14)
    {
/* 0794: numeric keys (key scan codes 0x0B to 0x14) */
      i += 0x25;		/* gives codes 0x30-0x39 for keys 0-9 */
    }

/* 079A: */
    else if (i <= 0x1C)
    {
/* 07A0: space bar, dot, operators and exponent (key scan codes 0x15 to 0x1C) */
      i -= 0x15;
      if ((mode & 0x0220) == 0)
      {
/* 07A0: [MODE] off */
        i = (int) (unsigned int) tab0066[i];
      }
      else if ((mode & 0x0020) != 0)
      {
/* 07AA: [S] on */
	i = (int) (unsigned int) tab00aa[i];
      }
      else if ((mode & 0x0040) == 0)
      {
/* 07B4: [MODE] on, [EXT] off */
        i = (int) (unsigned int) tab00a2[i];
      }
      else
      {
/* 07BE: [MODE] on, [EXT] on */
        i = (int) (unsigned int) tab00b2[i];
      }
    }

    else
    {
/* 07C4: letters (key scan codes 0x1D to 0x36) */
      i += 0x24;		/* gives codes 0x41-0x5A for keys A-Z */
      if ((mode & 0x0010) != 0)
      {
/* 07CC: [F] on */
      }
/* 07CE: */
      else if ((mode & 0x0040) != 0)
      {
/* 07F8: [EXT] on */
        if ((mode & 0x0220) == 0)
        {
/* 07F8: [MODE] off, [S] off */
          i += 0x40;		/* Russian lower case letters */
        }
        else if ((mode & 0x0200) != 0)
	{
/* 0802: [MODE] on */
          i += 0x60;		/* Russian upper case letters */
        }
        else
	{
/* 080C: [EXT] on, [S] on */
          i = (int) (unsigned int) tab00c5[i-0x41];	/* graphic characters */
        }
      }
/* 07D4: */
      else if ((mode & 0x0020) == 0)
      {
/* 07EC: [S] off */
        if ((mode & 0x0200) != 0)
	{
/* 07F2: [MODE on] */
          i += 0x20;		/* lower case letters */
        }
      }
      else
      {
/* 07DA: [S] on */
        for (j = 0; tab00ba[j] != 0; ++j)
        {
          if (i == (int) tab00ba[j])
          {
            i = (int) tab006e[j];
            break;
          }
        }
      }
    }
  }

/* 0810: [MODE] off, [S] off, [F] off */
  ram[0x8256u-RAMSTART] &= ~0x30;
  ram[0x8257u-RAMSTART] &= ~0x02;
  return i;
}


/* 0820: returns the scan code of the pressed key, or 0 if the key is not
 pressed or not recognized,
 handles the keyboard mode keys */
int keyb0820 (void)
{
  const uint8_t mode_bits[4][6] =
  {
/* 0920: key [.], mode EXT */
    { 0x40, 0x00,	/* mode bits to be changed */
      0x30, 0x02,	/* mode bits to be cleared */
      0x07, 0x01	/* bit masks for display memory */ },
/* 08FC: key [MODE], code 0x01 */
    { 0x00, 0x02, 0x30, 0x00, 0x06, 0x06 },
/* 0908: key [S], code 0x02 */
    { 0x20, 0x00, 0x10, 0x02, 0x06, 0x02 },
/* 0914: key [F], code 0x03 */
    { 0x10, 0x00, 0x20, 0x02, 0x06, 0x04 }
  };
  int i;

  i = key_scan ();
  if (i != 0)
  {
    if (i == 0x16 /*key [.]*/ && (ram[0x8257u-RAMSTART] & 0x02) != 0)
    {
      i = 0;		/* mode EXT */
    }
    if (i < 4)
    {
      ram[0x8256u-RAMSTART] ^= mode_bits[i][0];
      ram[0x8257u-RAMSTART] ^= mode_bits[i][1];
      ram[0x8256u-RAMSTART] &= ~mode_bits[i][2];
      ram[0x8257u-RAMSTART] &= ~mode_bits[i][3];
      ram[0] &= ~mode_bits[i][4];
      if ( (ram[0x8256u-RAMSTART] & mode_bits[i][0]) != 0 ||
           (ram[0x8257u-RAMSTART] & mode_bits[i][1]) != 0 )
      {
        ram[0] |= mode_bits[i][5];
      }
      lcd (0x80, ram[0]);
      i = 0;
    }
  }
  return i;
}


/* 097C: display a character with scrolling (when necessary) */
void display_char (unsigned int code)
{
  uint8_t x, pos;
  unsigned int dst;

  if ((ram[0x8264u-RAMSTART] & 0x80) == 0)
  {
/* 0984: the screen needs to be cleared */
    clear_screen ();
    ram[0x8264u-RAMSTART] |= 0x80;
  }

/* 098E: determine the cursor position from which scrolling is needed */
  pos = ((ram[0x8265u-RAMSTART] & 0x80) != 0) ? SCREEN_WIDTH-1 : SCREEN_WIDTH;
  if (ram[0x8269u /*cursor position*/ - RAMSTART] >= pos)
  {
/* 09A0: scrolling needed */
    if ((ram[0x8264u-RAMSTART] & 0x40) != 0)
    {
/* 09A8: delay between scrollings needed */
      delay_ms (((ram[0x8264u-RAMSTART] & 0x01) == 0) ? 200 : 1400);
    }
/* 09C8: scrolling */
    dst = 0;
    ram[0x8269u /*cursor position*/ - RAMSTART] = --pos;
    do {
/* 09D2: */
      ++dst;
      do {
/* 09D4: */
        x = ram[dst+8];
        lcd (dst+0x80, x);
        ram[dst] = x;
      } while ((++dst & 7) != 0);
    } while (--pos != 0);
    ram[0x8264u-RAMSTART] &= ~0x01;
  }

/* 09EC: */
  lcd_char (code);
  ++ram[0x8269u /*cursor position*/ - RAMSTART];
}


/* 09F8: display a character at position 8269 */
void lcd_char (unsigned int code)
{
/* 3AB0: font table, character codes 0x00-0xBF */
  const uint8_t charset[192][7] = {
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
    { 0x00, 0x01, 0x03, 0x07, 0x0F, 0x1F, 0x00 },
    { 0x00, 0x10, 0x18, 0x1C, 0x1E, 0x1F, 0x00 },
    { 0x00, 0x1F, 0x1E, 0x1C, 0x18, 0x10, 0x00 },
    { 0x00, 0x1F, 0x0F, 0x07, 0x03, 0x01, 0x00 },
    { 0x00, 0x1F, 0x0E, 0x04, 0x0E, 0x1F, 0x00 },
    { 0x00, 0x11, 0x1B, 0x1F, 0x1B, 0x11, 0x00 },
    { 0x08, 0x04, 0x04, 0x02, 0x04, 0x04, 0x08 },
    { 0x02, 0x04, 0x04, 0x08, 0x04, 0x04, 0x02 },
    { 0x00, 0x16, 0x09, 0x15, 0x12, 0x0D, 0x00 },
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x15 },
    { 0x00, 0x0E, 0x11, 0x11, 0x11, 0x0E, 0x00 },
    { 0x1F, 0x11, 0x02, 0x04, 0x02, 0x11, 0x1F },
    { 0x06, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00 },
    { 0x00, 0x00, 0x04, 0x0A, 0x1F, 0x00, 0x00 },
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1F },
    { 0x00, 0x11, 0x0A, 0x04, 0x0A, 0x11, 0x00 },
    { 0x00, 0x04, 0x00, 0x1F, 0x00, 0x04, 0x00 },
    { 0x04, 0x0E, 0x1F, 0x1F, 0x15, 0x04, 0x0E },
    { 0x00, 0x0A, 0x1F, 0x1F, 0x0E, 0x04, 0x00 },
    { 0x00, 0x04, 0x0E, 0x1F, 0x0E, 0x04, 0x00 },
    { 0x04, 0x0E, 0x04, 0x1F, 0x1F, 0x04, 0x0E },
    { 0x00, 0x00, 0x09, 0x09, 0x17, 0x01, 0x01 },
    { 0x0E, 0x11, 0x11, 0x11, 0x0A, 0x0A, 0x1B },
    { 0x04, 0x04, 0x04, 0x04, 0x15, 0x0E, 0x04 },
    { 0x00, 0x04, 0x02, 0x1F, 0x02, 0x04, 0x00 },
    { 0x00, 0x04, 0x08, 0x1F, 0x08, 0x04, 0x00 },
    { 0x11, 0x0A, 0x1F, 0x04, 0x1F, 0x04, 0x04 },
    { 0x00, 0x1F, 0x11, 0x11, 0x11, 0x1F, 0x00 },
    { 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x00 },
    { 0x03, 0x02, 0x02, 0x0E, 0x12, 0x12, 0x0E },
    { 0x00, 0x00, 0x03, 0x02, 0x0E, 0x12, 0x0E },
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
    { 0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x04 },
    { 0x0A, 0x0A, 0x0A, 0x00, 0x00, 0x00, 0x00 },
    { 0x0A, 0x0A, 0x1F, 0x0A, 0x1F, 0x0A, 0x0A },
    { 0x04, 0x1E, 0x05, 0x0E, 0x14, 0x0F, 0x04 },
    { 0x03, 0x13, 0x08, 0x04, 0x02, 0x19, 0x18 },
    { 0x06, 0x09, 0x05, 0x02, 0x15, 0x09, 0x16 },
    { 0x06, 0x04, 0x02, 0x00, 0x00, 0x00, 0x00 },
    { 0x08, 0x04, 0x02, 0x02, 0x02, 0x04, 0x08 },
    { 0x02, 0x04, 0x08, 0x08, 0x08, 0x04, 0x02 },
    { 0x00, 0x04, 0x15, 0x0E, 0x15, 0x04, 0x00 },
    { 0x00, 0x04, 0x04, 0x1F, 0x04, 0x04, 0x00 },
    { 0x00, 0x00, 0x00, 0x00, 0x06, 0x04, 0x02 },
    { 0x00, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x00 },
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06 },
    { 0x00, 0x10, 0x08, 0x04, 0x02, 0x01, 0x00 },
    { 0x0E, 0x11, 0x19, 0x15, 0x13, 0x11, 0x0E },
    { 0x04, 0x06, 0x04, 0x04, 0x04, 0x04, 0x0E },
    { 0x0E, 0x11, 0x10, 0x08, 0x04, 0x02, 0x1F },
    { 0x1F, 0x08, 0x04, 0x08, 0x10, 0x11, 0x0E },
    { 0x08, 0x0C, 0x0A, 0x09, 0x1F, 0x08, 0x08 },
    { 0x1F, 0x01, 0x0F, 0x10, 0x10, 0x11, 0x0E },
    { 0x0C, 0x02, 0x01, 0x0F, 0x11, 0x11, 0x0E },
    { 0x1F, 0x10, 0x08, 0x04, 0x04, 0x04, 0x04 },
    { 0x0E, 0x11, 0x11, 0x0E, 0x11, 0x11, 0x0E },
    { 0x0E, 0x11, 0x11, 0x1E, 0x10, 0x08, 0x06 },
    { 0x00, 0x06, 0x06, 0x00, 0x06, 0x06, 0x00 },
    { 0x00, 0x06, 0x06, 0x00, 0x06, 0x04, 0x02 },
    { 0x08, 0x04, 0x02, 0x01, 0x02, 0x04, 0x08 },
    { 0x00, 0x00, 0x1F, 0x00, 0x1F, 0x00, 0x00 },
    { 0x02, 0x04, 0x08, 0x10, 0x08, 0x04, 0x02 },
    { 0x0E, 0x11, 0x10, 0x08, 0x04, 0x00, 0x04 },
    { 0x00, 0x0F, 0x10, 0x16, 0x15, 0x15, 0x0E },
    { 0x0E, 0x11, 0x11, 0x1F, 0x11, 0x11, 0x11 },
    { 0x0F, 0x11, 0x11, 0x0F, 0x11, 0x11, 0x0F },
    { 0x0E, 0x11, 0x01, 0x01, 0x01, 0x11, 0x0E },
    { 0x07, 0x09, 0x11, 0x11, 0x11, 0x09, 0x07 },
    { 0x1F, 0x01, 0x01, 0x0F, 0x01, 0x01, 0x1F },
    { 0x1F, 0x01, 0x01, 0x0F, 0x01, 0x01, 0x01 },
    { 0x0E, 0x11, 0x01, 0x1D, 0x11, 0x11, 0x1E },
    { 0x11, 0x11, 0x11, 0x1F, 0x11, 0x11, 0x11 },
    { 0x0E, 0x04, 0x04, 0x04, 0x04, 0x04, 0x0E },
    { 0x1C, 0x08, 0x08, 0x08, 0x08, 0x09, 0x06 },
    { 0x11, 0x09, 0x05, 0x03, 0x05, 0x09, 0x11 },
    { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x1F },
    { 0x11, 0x1B, 0x15, 0x15, 0x11, 0x11, 0x11 },
    { 0x11, 0x11, 0x13, 0x15, 0x19, 0x11, 0x11 },
    { 0x0E, 0x11, 0x11, 0x11, 0x11, 0x11, 0x0E },
    { 0x0F, 0x11, 0x11, 0x0F, 0x01, 0x01, 0x01 },
    { 0x0E, 0x11, 0x11, 0x11, 0x15, 0x09, 0x16 },
    { 0x0F, 0x11, 0x11, 0x0F, 0x05, 0x09, 0x11 },
    { 0x1E, 0x01, 0x01, 0x0E, 0x10, 0x10, 0x0F },
    { 0x1F, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04 },
    { 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x0E },
    { 0x11, 0x11, 0x11, 0x11, 0x11, 0x0A, 0x04 },
    { 0x11, 0x11, 0x11, 0x15, 0x15, 0x1B, 0x11 },
    { 0x11, 0x11, 0x0A, 0x04, 0x0A, 0x11, 0x11 },
    { 0x11, 0x11, 0x0A, 0x04, 0x04, 0x04, 0x04 },
    { 0x1F, 0x10, 0x08, 0x04, 0x02, 0x01, 0x1F },
    { 0x0E, 0x02, 0x02, 0x02, 0x02, 0x02, 0x0E },
    { 0x00, 0x02, 0x1F, 0x04, 0x1F, 0x08, 0x00 },
    { 0x0E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x0E },
    { 0x04, 0x0E, 0x15, 0x04, 0x04, 0x04, 0x04 },
    { 0x00, 0x08, 0x04, 0x02, 0x1F, 0x00, 0x1F },
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
    { 0x00, 0x00, 0x0E, 0x10, 0x1E, 0x11, 0x1E },
    { 0x01, 0x01, 0x0D, 0x13, 0x11, 0x11, 0x0E },
    { 0x00, 0x00, 0x0E, 0x01, 0x01, 0x11, 0x0E },
    { 0x10, 0x10, 0x16, 0x19, 0x11, 0x11, 0x0E },
    { 0x00, 0x00, 0x0E, 0x11, 0x1F, 0x01, 0x0E },
    { 0x08, 0x14, 0x04, 0x0E, 0x04, 0x04, 0x04 },
    { 0x00, 0x00, 0x1E, 0x11, 0x1E, 0x10, 0x0E },
    { 0x01, 0x01, 0x0D, 0x13, 0x11, 0x11, 0x11 },
    { 0x04, 0x00, 0x06, 0x04, 0x04, 0x04, 0x0E },
    { 0x08, 0x00, 0x08, 0x08, 0x08, 0x09, 0x06 },
    { 0x02, 0x02, 0x12, 0x0A, 0x06, 0x0A, 0x12 },
    { 0x06, 0x04, 0x04, 0x04, 0x04, 0x04, 0x0E },
    { 0x00, 0x00, 0x0B, 0x15, 0x15, 0x15, 0x15 },
    { 0x00, 0x00, 0x0D, 0x13, 0x11, 0x11, 0x11 },
    { 0x00, 0x00, 0x0E, 0x11, 0x11, 0x11, 0x0E },
    { 0x00, 0x00, 0x0F, 0x11, 0x0F, 0x01, 0x01 },
    { 0x00, 0x00, 0x1E, 0x11, 0x1E, 0x10, 0x10 },
    { 0x00, 0x00, 0x1A, 0x06, 0x02, 0x02, 0x02 },
    { 0x00, 0x00, 0x1E, 0x01, 0x0E, 0x10, 0x0F },
    { 0x00, 0x04, 0x0E, 0x04, 0x04, 0x14, 0x08 },
    { 0x00, 0x00, 0x11, 0x11, 0x11, 0x19, 0x16 },
    { 0x00, 0x00, 0x11, 0x11, 0x11, 0x0A, 0x04 },
    { 0x00, 0x00, 0x11, 0x11, 0x15, 0x15, 0x0A },
    { 0x00, 0x00, 0x13, 0x0C, 0x04, 0x06, 0x19 },
    { 0x00, 0x00, 0x11, 0x12, 0x0C, 0x04, 0x03 },
    { 0x00, 0x00, 0x1F, 0x08, 0x04, 0x02, 0x1F },
    { 0x00, 0x00, 0x0F, 0x01, 0x07, 0x01, 0x0F },
    { 0x00, 0x00, 0x1F, 0x0A, 0x0A, 0x0A, 0x19 },
    { 0x1C, 0x00, 0x07, 0x01, 0x07, 0x01, 0x07 },
    { 0x00, 0x02, 0x04, 0x08, 0x1F, 0x00, 0x1F },
    { 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F },
    { 0x00, 0x00, 0x09, 0x15, 0x17, 0x15, 0x09 },
    { 0x00, 0x00, 0x0E, 0x10, 0x1E, 0x11, 0x1E },
    { 0x1E, 0x01, 0x0E, 0x11, 0x11, 0x11, 0x0E },
    { 0x00, 0x00, 0x09, 0x09, 0x09, 0x1F, 0x10 },
    { 0x0E, 0x10, 0x1E, 0x11, 0x11, 0x11, 0x0E },
    { 0x00, 0x00, 0x0E, 0x11, 0x1F, 0x01, 0x0E },
    { 0x00, 0x04, 0x0E, 0x15, 0x15, 0x0E, 0x04 },
    { 0x00, 0x00, 0x0E, 0x10, 0x0E, 0x01, 0x0E },
    { 0x00, 0x00, 0x13, 0x0C, 0x04, 0x06, 0x19 },
    { 0x00, 0x00, 0x11, 0x11, 0x11, 0x19, 0x16 },
    { 0x00, 0x00, 0x15, 0x11, 0x11, 0x19, 0x16 },
    { 0x00, 0x00, 0x12, 0x0A, 0x06, 0x0A, 0x12 },
    { 0x00, 0x00, 0x1C, 0x12, 0x12, 0x12, 0x11 },
    { 0x00, 0x00, 0x11, 0x1B, 0x15, 0x11, 0x11 },
    { 0x00, 0x00, 0x11, 0x11, 0x1F, 0x11, 0x11 },
    { 0x00, 0x00, 0x0E, 0x11, 0x11, 0x11, 0x0E },
    { 0x00, 0x00, 0x0D, 0x13, 0x11, 0x11, 0x11 },
    { 0x00, 0x00, 0x1E, 0x11, 0x1E, 0x14, 0x13 },
    { 0x00, 0x00, 0x0F, 0x11, 0x0F, 0x01, 0x01 },
    { 0x00, 0x00, 0x0E, 0x11, 0x01, 0x01, 0x0E },
    { 0x00, 0x00, 0x0B, 0x15, 0x15, 0x15, 0x15 },
    { 0x00, 0x00, 0x11, 0x12, 0x0C, 0x04, 0x03 },
    { 0x00, 0x00, 0x15, 0x15, 0x0E, 0x15, 0x15 },
    { 0x03, 0x05, 0x05, 0x0F, 0x11, 0x11, 0x0E },
    { 0x00, 0x00, 0x01, 0x01, 0x0F, 0x11, 0x0F },
    { 0x00, 0x00, 0x11, 0x11, 0x13, 0x15, 0x17 },
    { 0x00, 0x00, 0x0E, 0x10, 0x0C, 0x11, 0x0E },
    { 0x00, 0x00, 0x15, 0x15, 0x15, 0x15, 0x1F },
    { 0x00, 0x00, 0x0F, 0x10, 0x1E, 0x10, 0x0F },
    { 0x00, 0x00, 0x15, 0x15, 0x15, 0x1F, 0x10 },
    { 0x00, 0x00, 0x11, 0x11, 0x1E, 0x10, 0x10 },
    { 0x00, 0x0A, 0x0E, 0x11, 0x1F, 0x01, 0x0E },
    { 0x09, 0x15, 0x15, 0x17, 0x15, 0x15, 0x09 },
    { 0x1C, 0x12, 0x11, 0x1F, 0x11, 0x11, 0x11 },
    { 0x1F, 0x01, 0x01, 0x0F, 0x11, 0x11, 0x0F },
    { 0x09, 0x09, 0x09, 0x09, 0x09, 0x1F, 0x10 },
    { 0x0C, 0x0A, 0x0A, 0x0A, 0x0A, 0x1F, 0x11 },
    { 0x1F, 0x01, 0x01, 0x0F, 0x01, 0x01, 0x1F },
    { 0x04, 0x0E, 0x15, 0x15, 0x15, 0x0E, 0x04 },
    { 0x1F, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
    { 0x11, 0x11, 0x0A, 0x04, 0x0A, 0x11, 0x11 },
    { 0x11, 0x11, 0x19, 0x15, 0x13, 0x11, 0x11 },
    { 0x15, 0x11, 0x19, 0x15, 0x13, 0x11, 0x11 },
    { 0x11, 0x09, 0x05, 0x03, 0x05, 0x09, 0x11 },
    { 0x1C, 0x12, 0x11, 0x11, 0x11, 0x11, 0x11 },
    { 0x11, 0x1B, 0x15, 0x15, 0x11, 0x11, 0x11 },
    { 0x11, 0x11, 0x11, 0x1F, 0x11, 0x11, 0x11 },
    { 0x0E, 0x11, 0x11, 0x11, 0x11, 0x11, 0x0E },
    { 0x1F, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11 },
    { 0x1E, 0x11, 0x11, 0x1E, 0x14, 0x12, 0x11 },
    { 0x0F, 0x11, 0x11, 0x0F, 0x01, 0x01, 0x01 },
    { 0x0E, 0x11, 0x01, 0x01, 0x01, 0x11, 0x0E },
    { 0x1F, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04 },
    { 0x11, 0x11, 0x11, 0x1E, 0x10, 0x10, 0x0F },
    { 0x15, 0x15, 0x0E, 0x04, 0x0E, 0x15, 0x15 },
    { 0x0F, 0x11, 0x11, 0x0F, 0x11, 0x11, 0x0F },
    { 0x01, 0x01, 0x01, 0x0F, 0x11, 0x11, 0x0F },
    { 0x11, 0x11, 0x11, 0x13, 0x15, 0x15, 0x17 },
    { 0x0E, 0x11, 0x10, 0x0C, 0x10, 0x11, 0x0E },
    { 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x1F },
    { 0x0E, 0x11, 0x10, 0x1E, 0x10, 0x11, 0x0E },
    { 0x15, 0x15, 0x15, 0x15, 0x15, 0x1F, 0x10 },
    { 0x11, 0x11, 0x11, 0x1E, 0x10, 0x10, 0x10 },
    { 0x0A, 0x1F, 0x01, 0x07, 0x01, 0x01, 0x1F }
  };
  unsigned int src, dst;
  uint8_t x;

/* 0A16: address in the display memory */
  dst = (unsigned int) ram[0x8269u /*cursor position*/ - RAMSTART] * 8 + 1;

  if (code == 0x60)
  {
/* 0A12: user defined character */
    for (src=0x81ADu-RAMSTART; src<0x81B4u-RAMSTART; ++src)
    {
      x = ram[src];
      ram[dst] = x;
      lcd (dst+0x80, x);
      ++dst;
    }
  }
  else if (code < 0xC0)
/* 09FC: fixed patterns */
  {
    for (src=0; src<7; ++src)
    {
      x = charset[code][src];
      ram[dst] = x;
      lcd (dst+0x80, x);
      ++dst;
    }
  }
}


/* 0A32: clear screen */
void clear_screen (void)
{
  unsigned int dst;

  dst = 0;
  do {
    ++dst;
    do {
      ram[dst] = 0;
      lcd (dst+0x80, 0);
    } while ((++dst & 7) != 0);
  } while (dst < SCREEN_RAM_SIZE);
  lcd (0xE0, SCREEN_WIDTH);	/* hide the cursor */
  ram[0x8264u-RAMSTART] |= 1;	/* long delay between scrollings */
  ram[0x8269u-RAMSTART] = 0;	/* cursor position */
}


/* 0A72: display a number decimal using the four 7-segm. digits */
void display_7seg (unsigned int n)
{
  const uint8_t segments[4][10][3] =
  {
/* 0B22: segments for the 4th digit
 The third byte will be written to address E0, which sets the hardware
 cursor position. Value 0C means that no cursor will be shown. */
    {
      { 0x1A, 0x07, 0x0C },
      { 0x00, 0x03, 0x0C },
      { 0x1C, 0x06, 0x0C },
      { 0x14, 0x07, 0x0C },
      { 0x06, 0x03, 0x0C },
      { 0x16, 0x05, 0x0C },
      { 0x1E, 0x05, 0x0C },
      { 0x02, 0x07, 0x0C },
      { 0x1E, 0x07, 0x0C },
      { 0x16, 0x07, 0x0C }
    },
/* 0B04: segments for the 3rd digit */
    {
      { 0x10, 0x1E, 0x01 },
      { 0x00, 0x18, 0x00 },
      { 0x00, 0x17, 0x01 },
      { 0x00, 0x1D, 0x01 },
      { 0x10, 0x19, 0x00 },
      { 0x10, 0x0D, 0x01 },
      { 0x10, 0x0F, 0x01 },
      { 0x10, 0x18, 0x01 },
      { 0x10, 0x1F, 0x01 },
      { 0x10, 0x1D, 0x01 }
    },
/* 0AE6: segments for the 2nd digit */
    {
      { 0x14, 0x0F, 0x00 },
      { 0x00, 0x06, 0x00 },
      { 0x18, 0x0D, 0x00 },
      { 0x08, 0x0F, 0x00 },
      { 0x0C, 0x06, 0x00 },
      { 0x0C, 0x0B, 0x00 },
      { 0x1C, 0x0B, 0x00 },
      { 0x04, 0x0E, 0x00 },
      { 0x1C, 0x0F, 0x00 },
      { 0x0C, 0x0F, 0x00 }
    },
/* 0AC8: segments for the 1st digit */
    {
      { 0x1D, 0x03, 0x00 },	/* 0 */
      { 0x10, 0x01, 0x00 },	/* 1 */
      { 0x0E, 0x03, 0x00 },	/* 2 */
      { 0x1A, 0x03, 0x00 },	/* 3 */
      { 0x13, 0x01, 0x00 },	/* 4 */
      { 0x1B, 0x02, 0x00 },	/* 5 */
      { 0x1F, 0x02, 0x00 },	/* 6 */
      { 0x11, 0x03, 0x00 },	/* 7 */
      { 0x1F, 0x03, 0x00 },	/* 8 */
      { 0x1B, 0x03, 0x00 }	/* 9 */
    }
  };
/* 0AC0: addresses of the 7-segment fonts in the display memory */
  const unsigned int dtab[4] = { 0x50, 0x40, 0x38, 0x30 };
  unsigned int dst, i, j, digit;

/* 0A72: clear the segments */
  for (dst=0x30; dst<0x60; dst+=8)
  {
    ram[dst] = 0;
    lcd (dst+0x80, 0);
  }

/* 0A86: */
  for (digit=0; digit<4; ++digit)
  {
    j = n % 10;
    n /= 10;
    dst = dtab[digit];
    for (i=0; i<3; ++i)
    {
      ram[dst] |= segments[digit][j][i];
      lcd (dst+0x80, ram[dst]);
      dst += 8;
    }
    if (n == 0)
    {
      break;
    }
  }
}


/* 1248: display a zero terminated string stored in the RAM,
 printing can be interrupted with STOP, resumed with EXE, aborted with AC */
int display_string (unsigned int addr)
{
  unsigned int x;
  int i;

  while ((x = (unsigned int) ram[addr++-RAMSTART]) != 0)
  {
    i = keys_stop_exe_ac ();
    if (i != 0)
    {
      return i;
    }
    display_char (x);
  }
  return 0;
}


/* 1258: test for keys STOP, EXE, AC */
int keys_stop_exe_ac (void)
{
  int i;

  if ((ram[0x8265u-RAMSTART] & 0x01) != 0)
  {
/* STOP key pressed */
    do {
      i = keyb0770 ();
      ram[0x8265u-RAMSTART] &= ~0x03;
    } while (i != 7 /*[AC]*/ && i != 4 /*[EXE]*/);
    if (i == 7)
    {
/* 128C: key AC */
      ram[0x50] = 0;
      ram[0x58] = 0;
      lcd (0xD0, 0);
      lcd (0xD8, 0);
      ram[0x8256u-RAMSTART] &= ~0x31;
      ram[0x8257u-RAMSTART] &= ~0xC6;
      return 1;
//12A2:	mov	#812C,sp
//12A6:	mov	#0BAA,-(sp)
//12AA:	jmp	05C0
    }
/* 127A: key code 4, EXE */
    if ((ram[0x8257u-RAMSTART] & 0x02) != 0)
    {
/* 1284: mode on */
      ram[0x8265u-RAMSTART] |= 0x02;
    }
  }
  return 0;
}


/* display a zero terminated string stored in the RAM */
void display_short_string (unsigned int addr)
{
  unsigned int x;

  while ((x = (unsigned int) ram[addr++-RAMSTART]) != 0)
  {
    display_char (x);
  }
}
