library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

entity xc9536xl is
    Port ( data_bus : inout  STD_LOGIC_VECTOR (7 downto 0);
           addr_bus : in  STD_LOGIC_VECTOR (2 downto 0);
           host_wr : in  STD_LOGIC;
           host_rd : in  STD_LOGIC;
           cs7 : in  STD_LOGIC;
           oe7 : in  STD_LOGIC;
           p4 : in  STD_LOGIC; -- '1' when reset, '0' when normal operation
           pon : in  STD_LOGIC; -- '1' when host turned on and supply voltage present
           int1 : out  STD_LOGIC;
           led : out  STD_LOGIC; -- falling edge flashes a LED
           lcx245_g : out  STD_LOGIC;
           lcx245_dir : out  STD_LOGIC; -- '1' when host reads data
           ft245_wr : out  STD_LOGIC;
           ft245_rd : out  STD_LOGIC;
           ft245_rxf : in  STD_LOGIC;
           hc221 : in  STD_LOGIC; -- inverted output of a monostable triggered by rising edge of "ft245_rxf"
           ft245_txe : in  STD_LOGIC );
    attribute BUFG : string;
    attribute BUFG of cs7 : signal is "CLK";
end xc9536xl;

architecture Behavioral of xc9536xl is

    signal host_rd1 : STD_LOGIC;
    signal host_wr2 : STD_LOGIC;
    signal ft245_rd1 : STD_LOGIC;
    signal ft245_wr1 : STD_LOGIC;

    signal inreg000 : STD_LOGIC_VECTOR (7 downto 0);
    signal inreg100 : STD_LOGIC_VECTOR (7 downto 0);
    signal inregs : STD_LOGIC_VECTOR (7 downto 0);

    signal outreg101 : STD_LOGIC_VECTOR (7 downto 0);
    signal txen : STD_LOGIC; -- bit 0 of the output register "001"
    signal rxen : STD_LOGIC; -- bit 1 of the output register "001"
    signal irqen : STD_LOGIC; -- bit 0 of the output register "010"
    signal ack_latch : STD_LOGIC; -- bit 2 of the input register "100"
    signal strobe_latch : STD_LOGIC; -- bit 0 of the input register "100"
    signal prn_data : STD_LOGIC; -- '1' when the contents of the outreg101 is placed on the "data_bus"

begin

    host_rd1 <= '1' when host_rd = '0' and host_wr = '1' and pon = '1' and oe7 = '0' else '0';
    host_wr2 <= '1' when host_wr = '0' and host_rd = '1' and pon = '1' else '0';
    ft245_rd1 <= '0' when addr_bus = "010" and host_rd1 = '1' and rxen = '1' else '1';
    ft245_wr1 <= '1' when (addr_bus = "011" and txen = '1' and host_wr2 = '1' and cs7 = '0') or strobe_latch = '1' else '0';

    lcx245_g <= prn_data;
    lcx245_dir <= host_rd1;
    ft245_rd <= ft245_rd1;
    ft245_wr <= ft245_wr1;
    led <= '0' when ft245_wr1 = '1' or ft245_rd1 = '0' else '1';
    int1 <= '1' when rxen = '1' and irqen = '1' and ft245_rxf = '0' and hc221 = '1' else '0';

-- writing to the control register 001
process (p4, cs7)
begin
    if p4 = '1' then -- asynchronous reset
        txen <= '0';
        rxen <= '0';
    elsif rising_edge(cs7) and host_wr2 = '1' and addr_bus = "001" then
        txen <= data_bus(0);
        rxen <= data_bus(1);
    end if;
end process;

-- writing to the control register 010
process (p4, cs7)
begin
    if p4 = '1' then -- asynchronous reset
        irqen <= '0';
    elsif rising_edge(cs7) and host_wr2 = '1' and addr_bus = "010" then
        irqen <= data_bus(0);
    end if;
end process;

-- writing to the output register 101 (printer data)
process (p4, cs7)
begin
    if p4 = '1' then -- asynchronous reset
        outreg101 <= "00000000";
    elsif rising_edge(cs7) and host_wr2 = '1' and addr_bus = "101" then
        outreg101 <= data_bus;
    end if;
end process;

-- writing to the "ack_latch"
process (p4, cs7)
begin
    if p4 = '1' then -- asynchronous reset
        ack_latch <= '0';
    elsif rising_edge(cs7) and host_wr2 = '1' and addr_bus = "110" and (data_bus(0) = '1' or data_bus(2) = '1') then
        ack_latch <= data_bus(0);
    end if;
end process;

-- writing to the "strobe_latch"
process (p4, host_wr2, addr_bus, cs7)
begin
    if p4 = '1' or host_wr2 = '0' or addr_bus /= "110" then -- asynchronous reset
        strobe_latch <= '0';
    elsif rising_edge(cs7) then
        strobe_latch <= data_bus(0);
    end if;
end process;

-- writing to the "prn_data"
process (strobe_latch, cs7)
begin
    if strobe_latch = '1' then -- asynchronous set
        prn_data <= '1';
    elsif falling_edge(cs7) then -- synchronous clear
        prn_data <= '0';
    end if;
end process;

-- reading the serial port status register 000
    inreg000(0) <= ft245_txe;
    inreg000(1) <= not ft245_rxf;
    inreg000(2) <= '1'; -- CTS
    inreg000(3) <= '1'; -- DSR
    inreg000(4) <= '1'; -- DCD
    inreg000(5) <= '0';
    inreg000(6) <= '0';
    inreg000(7) <= '0';

-- reading the printer status register 100
    inreg100(0) <= ft245_txe; -- BUSY status bit
    inreg100(1) <= '1'; -- no ERROR status bit
    inreg100(2) <= ack_latch;
    inreg100(3) <= '0';
    inreg100(4) <= '0';
    inreg100(5) <= '0';
    inreg100(6) <= '0';
    inreg100(7) <= '0';

-- reading the status registers
    with addr_bus select
        inregs <= inreg000 when "000",
                  "00011000" when "001",
                  "01010101" when "011", -- Port A (fixed option code)
                  inreg100 when "100",
                  "00000000" when others;

    data_bus <= outreg101 when prn_data = '1' else inregs when host_rd1 = '1' and addr_bus /= "010" else "ZZZZZZZZ";

end Behavioral;

